function M = fixedrankfactory_2factors(m, n, k)
% Manifold of m-by-n matrices of rank k with balanced quotient geometry.
%
% function M = fixedrankfactory_2factors(m, n, k)
%
% The first-order geometry follows the balanced quotient geometry described 
% in the paper, 
% "Linear regression under fixed-rank constraints: a Riemannian approach",
% G. Meyer, S. Bonnabel and R. Sepulchre, ICML 2011.
%
% Paper link: http://www.icml-2011.org/papers/350_icmlpaper.pdf.
%
% The second-order geometry follows from the paper
% "Fixed-rank matrix factorizations and Riemannian low-rank optimization",
% B. Mishra, R. Meyer, S. Bonnabel and R. Sepulchre,
% Computational Statistics, 29(3 - 4), pp. 591 - 621, 2014.
%
% A point X on the manifold is represented as a structure with two
% fields: L and R. The matrices L (mxk) and R (nxk) are full column-rank
% matrices such that X = L*R'.
%
% Tangent vectors are represented as a structure with two fields: L, R.
% 
% For first-order geometry, please cite the Manopt paper as well as the research paper:
%     @InProceedings{meyer2011linear,
%       Title        = {Linear regression under fixed-rank constraints: a {R}iemannian approach},
%       Author       = {Meyer, G. and Bonnabel, S. and Sepulchre, R.},
%       Booktitle    = {{28th International Conference on Machine Learning}},
%       Year         = {2011},
%       Organization = {{ICML}}
%     }
%
% For second-order geometry, please cite the Manopt paper as well as the research paper:
%     @Article{mishra2014fixedrank,
%       Title   = {Fixed-rank matrix factorizations and {Riemannian} low-rank optimization},
%       Author  = {Mishra, B. and Meyer, G. and Bonnabel, S. and Sepulchre, R.},
%       Journal = {Computational Statistics},
%       Year    = {2014},
%       Number  = {3-4},
%       Pages   = {591--621},
%       Volume  = {29},
%       Doi     = {10.1007/s00180-013-0464-z}
%     }
%
%
% See also fixedrankembeddedfactory fixedrankfactory_3factors fixedrankfactory_2factors_preconditioned

% This file is part of Manopt: www.manopt.org.
% Original author: Bamdev Mishra, Dec. 30, 2012.
% Contributors:
% Change log:
%
%   July 10, 2013 (NB):
%       Added vec, mat, tangent, tangent2ambient.
%
%   July 03, 2015 (BM):
%      Cosmetic changes including avoiding storing the inverse of a
%       k-by-k matrix.
%
%   Apr. 17, 2018 (NB):
%      Using built-in sylvester instead of lyap, which requires a toolbox.
%
%   Sep.  6, 2018 (NB):
%       Removed M.exp() as it was not implemented.
    
    
    M.name = @() sprintf('LR'' quotient manifold of %dx%d matrices of rank %d', m, n, k);
    
    M.dim = @() (m+n-k)*k;
    
    % Some precomputations at the point X to be used in the inner product 
    % (and pretty much everywhere else).
    function X = prepare(X)
        if ~all(isfield(X,{'LtL','RtR'}))
            L = X.L;
            R = X.R;
            X.LtL = L'*L;
            X.RtR = R'*R;
        end
    end
    
    % Choice of the metric is motivated by the symmetry present in the
    % space. The metric is the natural Grassmannian metric on L and R.
    M.inner = @iproduct;
    function ip = iproduct(X, eta, zeta)
        X = prepare(X);
        ip = trace(X.LtL\(eta.L'*zeta.L)) + trace( X.RtR\(eta.R'*zeta.R));
    end
    
    M.norm = @(X, eta) sqrt(M.inner(X, eta, eta));
    
    M.typicaldist = @() 10*k;
    
    symm = @(M) .5*(M+M');
    
    M.egrad2rgrad = @egrad2rgrad;
    function rgrad = egrad2rgrad(X, egrad)
        X = prepare(X);
        rgrad.L = egrad.L*X.LtL;
        rgrad.R = egrad.R*X.RtR;
    end
    
    M.ehess2rhess = @ehess2rhess;
    function Hess = ehess2rhess(X, egrad, ehess, eta)
        X = prepare(X);
        
        % Riemannian gradient computation.
        rgrad = egrad2rgrad(X, egrad);
        
        % Directional derivative of the Riemannian gradient.
        Hess.L = ehess.L*X.LtL + 2*egrad.L*symm(eta.L'*X.L);
        Hess.R = ehess.R*X.RtR + 2*egrad.R*symm(eta.R'*X.R);
        
        % We need a correction term for the non-constant metric.
        Hess.L = Hess.L - rgrad.L*(X.LtL\(symm(X.L'*eta.L))) - eta.L*(X.LtL\(symm(X.L'*rgrad.L))) + X.L*(X.LtL\(symm(eta.L'*rgrad.L)));
        Hess.R = Hess.R - rgrad.R*(X.RtR\(symm(X.R'*eta.R))) - eta.R*(X.RtR\(symm(X.R'*rgrad.R))) + X.R*(X.RtR\(symm(eta.R'*rgrad.R)));
        
        % Projection onto the horizontal space.
        Hess = M.proj(X, Hess);
    end
    
    M.proj = @projection;
    % Projection of the vector eta in the ambient space onto the horizontal space.
    function etaproj = projection(X, eta)
        X = prepare(X);
        
        SS = (X.LtL)*(X.RtR);
        AS = (X.LtL)*(X.R'*eta.R) - (eta.L'*X.L)*(X.RtR);
        Omega = sylvester_nochecks(SS, SS, AS); % could be sped up since SS appears twice
        etaproj.L = eta.L + X.L*Omega';
        etaproj.R = eta.R - X.R*Omega;
    end
    
    M.tangent = M.proj;
    M.tangent2ambient = @(X, eta) eta;
    
    M.retr = @retraction;
    function Y = retraction(X, eta, t)
        if nargin < 3
            t = 1.0;
        end
        
        Y.L = X.L + t*eta.L;
        Y.R = X.R + t*eta.R;
        
        % Numerical conditioning step: A simpler version.
        % We need to ensure that L and R do not have very relative
        % skewed norms.
        
        scaling = norm(X.L, 'fro')/norm(X.R, 'fro');
        scaling = sqrt(scaling);
        Y.L = Y.L / scaling;
        Y.R = Y.R * scaling;
        
        % These are reused in the computation of the gradient and Hessian.
        Y = prepare(Y);
    end
    
    
    M.hash = @(X) ['z' hashmd5([X.L(:) ; X.R(:)])];
    
    M.rand = @random;
    function X = random()
        % A random point on the total space.
        X.L = randn(m, k);
        X.R = randn(n, k);
        X = prepare(X);
    end
    
    M.randvec = @randomvec;
    function eta = randomvec(X)
        % A random vector in the horizontal space.
        eta.L = randn(m, k);
        eta.R = randn(n, k);
        eta = projection(X, eta);
        nrm = M.norm(X, eta);
        eta.L = eta.L / nrm;
        eta.R = eta.R / nrm;
    end
    
    M.lincomb = @lincomb;
    
    M.zerovec = @(X) struct('L', zeros(m, k),'R', zeros(n, k));
    
    M.transp = @(x1, x2, d) projection(x2, d);
    
    % vec and mat are not isometries, because of the unusual inner metric.
    M.vec = @(X, U) [U.L(:) ; U.R(:)];
    M.mat = @(X, u) struct('L', reshape(u(1:(m*k)), m, k), ...
        'R', reshape(u((m*k+1):end), n, k));
    M.vecmatareisometries = @() false;
    
end

% Linear combination of tangent vectors.
function d = lincomb(x, a1, d1, a2, d2) %#ok<INUSL>
    
    if nargin == 3
        d.L = a1*d1.L;
        d.R = a1*d1.R;
    elseif nargin == 5
        d.L = a1*d1.L + a2*d2.L;
        d.R = a1*d1.R + a2*d2.R;
    else
        error('Bad use of fixedrankfactory_2factors.lincomb.');
    end
    
end





