import os
import sys
import json
import yaml
import shutil
import tempfile
from time import time
from build_info_retrieval import read_file,get_html_content_in_markdown,list_files, read_file, validate_internal_path, search_online_using_tavily
from bash_executor import BashCodeExecutor

from autogen import GroupChat, GroupChatManager,UserProxyAgent, AssistantAgent
from autogen.coding import LocalCommandLineCodeExecutor, DockerCommandLineCodeExecutor
from dotenv import load_dotenv
load_dotenv()


class multi_agent():
    def __init__(self, model_name, api_key, readme_path , clone_dir, docker_image="ubuntu:20.04", max_turns=15):
        self.max_turns = max_turns
        self.model_name = model_name
        self.api_key = api_key
        self.timeout_bash = 60
        self.temp_dir = tempfile.TemporaryDirectory()
        self.llm_config = {
                'cache_seed': 41,
                'model': self.model_name,
                'api_key': self.api_key,
                # 'temperature': self.temperature,
                # 'timeout': self.timeout_llm,
            }
        self.docker_image = docker_image
        self.readme_path = readme_path
        self.clone_dir = clone_dir
        self.repo_name = os.path.basename(clone_dir)

    def _create_code_executor(self, if_docker=False, docker_image="ubuntu:20.04"):
        # ==========================
        # Reference: Autogen documentation
        # https://microsoft.github.io/autogen/docs/tutorial/code-executors/#local-execution
        
        # This creates a docker executor for better isolation and security.
        # ==========================
            
        # Create a temporary directory to store the code files.
        if not if_docker:
            print("Using local executor")
            # Create a local command line code executor.
            executor = BashCodeExecutor(
            timeout = int(self.timeout_bash),
            # virtual_env_context = create_virtual_env(self.venv_dir),
            # work_dir = self.temp_dir.name
            )
        # Create a Docker command line code executor.
        else:
            print("Using docker executor with image: ", docker_image)
            executor = DockerCommandLineCodeExecutor(
                image=docker_image,  # Execute code using the given docker image name.
                container_name='autogen_docker_executor',  # Name of the container.
                timeout=self.timeout_bash,  # Timeout for each code execution in seconds.
                work_dir=self.temp_dir.name,  # Use the temporary directory to store the code files.
                # work_dir=self.project_dir,  # Use the temporary directory to store the code files.
                # bind_dir=self.project_dir,
            )
        return executor
    
    def generate_agents(self):
        execution_agent = UserProxyAgent(
                name = 'Execution',
                system_message = f'You are an AI assistant that can run bash commands and conduct the process of Github repository compilation. \n', 
                # llm_config = self.llm_config,
                description = 'Can execute bash commands and Python Codes',
                human_input_mode = 'NEVER',    # ALWAYS, TERMINATE, NEVER
                code_execution_config = {'executor': self._create_code_executor(if_docker=True, docker_image=self.docker_image)},
            )
        
        information_retrieval_agent = AssistantAgent(
            name = 'information_retrieval_agent',
            description = 'This agent determines that with a given information source, is there any information (instructions or dependencies) for building a C/C++ project from source. If so, it will retrieve the information with a variety of given tools. If not, it will state that there is no information for building a C/C++ project from source, and suggest additional links to sources that may contain the demanded information, if presented.',
            system_message= f'''You are an AI assistant whose sole task is to locate and extract precise “build from source” instructions for any open-source repository targeting Ubuntu. To do this reliably across very different projects, follow this ordered strategy:

            1. **Local inspection**  
            a. Read through the content of `README.md` (or README.rst, etc.) in full.  
            b. If you see local files named `INSTALL.md`, `BUILD.md`, `docs/`, or similar, read those next.  
            c. Search within those files for keywords like “build”, “install”, “cmake”, “configure”, “make”, etc., and extract any relevant instructions.

            2. **Learn from inbound links**  
            a. In the README or docs, collect *any* URLs that look like official docs (URLs containing “docs”, “installation”, “catboost.ai/docs”, etc.).  
            b. Fetch those URLs and scan them for Ubuntu-specific build or installation instructions.

            3. **Fallback web search**  
            If neither local files nor linked URLs contain clear build steps, perform a web search with a query of the form: ' build from source ubuntu' 
            
            —then fetch and parse the top result that appears to be an official guide or installation page.

            4. **Tool usage**  
            You may call and only call the Execution agent to use the following tools.
            - Use **read_file(path)** to open and read local files(which may needs to be validated based on the absolute path of the README file {self.readme_path}).  
            - Use **list_files(path)** to list all files in a directory.  
            - Use **search_online_using_tavily(query)** to run a search from Internet when needed.  
            - Use **get_html_content_in_markdown(url)** to read contents from a external url in Markdown format.

            Retrieval is considered as finished as soon as you’ve assembled a complete, step-by-step Ubuntu build procedure (including dependencies, configuration flags, commands, and any prerequisites). If you encounter multiple valid methods (e.g. binary vs. source), default to *source* and note any alternatives at the end. You should then send the instructions to the Bash_Command_Generator agent for compilation.
            
            Note that 
            ''',
            llm_config=self.llm_config,
            human_input_mode="NEVER",

        )
        
        coordinator_agent = AssistantAgent(
            name = 'coordinator_agent',
            description = 'This agent plans the steps of building a C/C++ project from source code. It achieves the goal by tasking the other agent to retrieval information from internal paths (files) or external urls (webpages) embedded in the initial file and inspecting the files within the project codebase. If the other agent replies with additional links that are of interest, the coordinator agent will analyze and task the information retrieval agent to retrieve information from these links.',
            system_message="""
            You are the C/C++ Build Coordinator. Your primary responsibility is to plan the steps required to build a C/C++ project from source code. To achieve this, follow these guidelines:
                1.	Analyze the Initial File:
                •	Carefully review the provided file to identify any embedded internal file paths or external URLs that may contain build instructions, dependency lists, or relevant configuration details for the C/C++ project.
                2.	Coordinate Information Retrieval:
                •	When you identify potential sources (either internal paths or external URLs), task the designated information retrieval agent to fetch and extract the relevant details from these sources.
                •	Monitor the responses from the retrieval agent for additional links or references that might be of interest.
                3.	Evaluate Additional Links:
                •	If the retrieval agent supplies further links, analyze them to determine their relevance to the build process.
                •	If they are pertinent, instruct the retrieval agent to retrieve the content from these additional links.
                4.	Planning the Build Process:
                •	Use the gathered information to plan a comprehensive, step-by-step build process for the C/C++ project.
                •	Ensure that all dependencies, configurations, and build instructions are clearly identified and documented.
                
                NOTE: Terminate the process if you think the detailed compilation steps, including installing necessary dependencies, configurations, build instructions, are collected and sufficient to proceed with the build or if no further relevant details are available. You would need to say TERMINATE in the chat. 
                Do not repeat the same conversation. 
            """,
            is_termination_msg=lambda msg: msg.get("content") is not None and "TERMINATE" in msg["content"],
            llm_config=self.llm_config,
            human_input_mode="NEVER",
            )
        
        information_retrieval_agent.register_for_llm(name="read_file", description="Read contents from a local file")(read_file)
        information_retrieval_agent.register_for_llm(name="get_html_content_in_markdown", description="Read contents from a external url")(get_html_content_in_markdown)
        information_retrieval_agent.register_for_llm(name="search_online_using_tavily", description="Search online by writing a query")(search_online_using_tavily) 
        information_retrieval_agent.register_for_llm(name="list_files", description="List all files in a directory")(list_files)   
        
        # coordinator_agent.register_for_llm(name="read_file", description="Read contents from a local file")(read_file)
        # coordinator_agent.register_for_llm(name="list_files", description="Get the list of files from a codebase on the first level")(list_files)
        # coordinator_agent.register_for_llm(name="read_readme", description="Read readme content from readme file")(read_file)        
        # coordinator_agent.register_for_llm(name="validate_internal_path", description="Validate an internal file path from a project root directory.")(validate_internal_path)   
            
        execution_agent.register_for_execution(name="read_file")(read_file)
        execution_agent.register_for_execution(name="get_html_content_in_markdown")(get_html_content_in_markdown)
        # execution_agent.register_for_execution(name="validate_internal_path")(validate_internal_path)
        execution_agent.register_for_execution(name="list_files")(list_files)
        execution_agent.register_for_execution(name="search_online_using_tavily")(search_online_using_tavily)

        # execution_agent.register_for_execution(name="read_readme")(read_file)

        # initializer = UserProxyAgent(
        #     name="Init",
        #     human_input_mode="NEVER",
        # )        
        initializer = None
        return information_retrieval_agent, coordinator_agent, execution_agent,initializer
    
    # def register_functions(information_retrieval_agent, coordinator_agent):
    def initial_message_retriever(self):
        self.readme_content = read_file(self.readme_path)
        return (

            f'I have cloned the GitHub repository, {self.repo_name}, and the absolute path to the cloned repository is {self.clone_dir}.\n'
            f'The repository contains the following files:\n{list_files(self.clone_dir)}\n'
            f'Please help me find the build instructions for this repository by starting off from the README. For the internal file path that may contain the build instructions, validate the path based on the absolute path and access the file content. For urls, you can accessing content of them. Finally, feel free to search the internet for the build instructions. Note that sometimes the build instructions may require you explore for a few times to find them.\n'
            f'The README content is as follows:\n{self.readme_content}\n'
            f'The README file is located at {self.readme_path}.\n'
        )


    def initialize_group_chat(self):
        information_retrieval_agent, coordinator_agent, execution_agent,initializer = self.generate_agents()

        # group_chat = GroupChat(
        #     agents=[information_retrieval_agent, coordinator_agent, execution_agent],
        #     messages=[],
        # )
        
        # manager = GroupChatManager(groupchat=group_chat, llm_config=self.llm_config)
        

        
        # manager.initiate_chat(coordinator_agent, message=f"Topic: find the build instructions for a C/C++ project from source code. The initial path of the readme is located at {self.readme_path}")
        chat_history_1 = execution_agent.initiate_chat(
            information_retrieval_agent, max_turns=self.max_turns,
            message=self.initial_message_retriever(), silent=False, clear_history=False, summary_args={'summary_method': 'reflection_with_llm', 'summary_prompt': 'Summarize the takeaway from the conversation. Make sure to add ALL the bash commands.'},
        )
        print(information_retrieval_agent.last_message(execution_agent))

agents = multi_agent(model_name="o3-mini", api_key=os.environ.get('OPENAI_KEY'), docker_image="sz904/compilation_base_image:8",
readme_path="/mnt/midnight/steven_zhang/LLM_assisted_compilation/cloned_repos/obs-studio/README.rst",
clone_dir="/mnt/midnight/steven_zhang/LLM_assisted_compilation/cloned_repos/obs-studio",
max_turns=10)
agents.initialize_group_chat()