import os
import tempfile
import pytest
from validation import find_c_cpp_files, find_linux_compiled_artifacts

@pytest.fixture
def temp_dir_with_files():
    with tempfile.TemporaryDirectory() as temp_dir:
        # Create some C/C++ files
        open(os.path.join(temp_dir, 'file1.c'), 'w').close()
        open(os.path.join(temp_dir, 'file2.cpp'), 'w').close()
        open(os.path.join(temp_dir, 'file3.h'), 'w').close()
        open(os.path.join(temp_dir, 'file4.hpp'), 'w').close()
        os.makedirs(os.path.join(temp_dir, 'src'))    
        open(os.path.join(temp_dir, 'src/', 'file5.c'), 'w').close()
        # Create some directories to exclude
        os.makedirs(os.path.join(temp_dir, 'Test'))
        open(os.path.join(temp_dir, 'Test', 'file5.c'), 'w').close()
        
        yield temp_dir

@pytest.fixture
def temp_dir_with_artifacts():
    with tempfile.TemporaryDirectory() as temp_dir:
        # Create some compiled artifacts
        open(os.path.join(temp_dir, 'file1.o'), 'w').close()
        open(os.path.join(temp_dir, 'file2.so'), 'w').close()
        open(os.path.join(temp_dir, 'file3.out'), 'w').close()
        
        # Create some directories to exclude
        os.makedirs(os.path.join(temp_dir, 'Test'))
        open(os.path.join(temp_dir, 'Test', 'file4.o'), 'w').close()
        
        yield temp_dir

def test_find_c_cpp_files(temp_dir_with_files):
    result = find_c_cpp_files(temp_dir_with_files)
    assert len(result) == 3  # Only .c and .cpp files should be included by default
    assert os.path.join(temp_dir_with_files, 'file1.c') in result
    assert os.path.join(temp_dir_with_files, 'file2.cpp') in result

    result_with_headers = find_c_cpp_files(temp_dir_with_files, include_headers=True)
    assert len(result_with_headers) == 5  # .c, .cpp, .h, and .hpp files should be included
    assert os.path.join(temp_dir_with_files, 'file3.h') in result_with_headers
    assert os.path.join(temp_dir_with_files, 'file4.hpp') in result_with_headers
    assert os.path.join(temp_dir_with_files, 'src/', 'file5.c') in result_with_headers
    print(result_with_headers)

def test_find_linux_compiled_artifacts(temp_dir_with_artifacts):
    result = find_linux_compiled_artifacts(temp_dir_with_artifacts)
    assert len(result) == 3  # Only .o, .so, and .out files should be included
    assert os.path.join(temp_dir_with_artifacts, 'file1.o') in result
    assert os.path.join(temp_dir_with_artifacts, 'file2.so') in result
    assert os.path.join(temp_dir_with_artifacts, 'file3.out') in result
    assert os.path.join(temp_dir_with_artifacts, 'Test', 'file4.o') not in result

