from __future__ import annotations

"""Base and registry for model runners.
"""

from abc import ABC, abstractmethod
from typing import Any, Callable, Dict, Type


class ModelRunner(ABC):
    name: str = "base"
    system_prompt: str = ""

    @abstractmethod
    def load_model(self) -> None:
        """Load/initialize model resources."""

    @abstractmethod
    def build_conversation(self, user_content: list[dict]) -> Any:
        """Build a model-specific conversation using the model's system_prompt."""

    @abstractmethod
    def run_model(self, conversation: Any) -> Any:
        """Run inference on the provided conversation and return output (string or dict)."""


# Registry
MODEL_REGISTRY: Dict[str, Type[ModelRunner]] = {}

def register_model(name: str) -> Callable[[Type[ModelRunner]], Type[ModelRunner]]:
    def decorator(cls: Type[ModelRunner]) -> Type[ModelRunner]:
        MODEL_REGISTRY[name] = cls
        cls.name = name
        return cls

    return decorator


def get_model_runner(name: str, *args: Any, **kwargs: Any) -> ModelRunner:
    try:
        cls = MODEL_REGISTRY[name]
    except KeyError as exc:
        available = ", ".join(MODEL_REGISTRY)
        raise ValueError(f"Model '{name}' is not registered. Available: {available}") from exc
    return cls(*args, **kwargs)
