from __future__ import annotations

"""Base class for tasks"""

from abc import ABC, abstractmethod
from typing import Any, Callable, Dict, Iterable, List, Optional, Tuple, Type


class Task(ABC):
    name: str = "base"

    @abstractmethod
    def run(
        self,
        model_runner: Any,
        *,
        max_samples: Optional[int] = None,
    ) -> Dict[str, Any]:
        """Execute evaluation for this task using the provided model runner.

        Should return a dict with summary information if applicable.
        """


# Registry
TASK_REGISTRY: Dict[str, Type[Task]] = {}


def register_task(name: str) -> Callable[[Type[Task]], Type[Task]]:
    def decorator(cls: Type[Task]) -> Type[Task]:
        TASK_REGISTRY[name] = cls
        cls.name = name
        return cls

    return decorator


def get_task(name: str, *args: Any, **kwargs: Any) -> Task:
    try:
        cls = TASK_REGISTRY[name]
    except KeyError as exc:
        available = ", ".join(TASK_REGISTRY)
        raise ValueError(f"Task '{name}' is not registered. Available: {available}") from exc
    return cls(*args, **kwargs)

