#!/bin/bash

set -euo pipefail

# Configuration
MODEL="${MODEL:-mistralai/Mistral-7B-v0.1}"
# MODEL="${MODEL:-google/gemma-2-9b}"
GPU_ID=${GPU_ID:-0}

# Fill your BHRA run directories here
RUN_DIRS=(
  # "/path/to/experiments/block_hadamard_hira_arithmetic/Mistral-7B-v0.1/2025..._rank_32_blocks4_lr1e-4_alpha_32train50000" 
)

for RUN_DIR in "${RUN_DIRS[@]}"; do
  echo "=== Processing: $RUN_DIR ==="

  METHOD=$(basename "$(dirname "$RUN_DIR")")
  echo "Extracted method: $METHOD"

  FINAL_MODEL_PATH="$RUN_DIR/final_model"
  MERGED_MODEL_PATH="$RUN_DIR/merged_model"

  echo "=== Starting Merging ==="
  if [ ! -d "$FINAL_MODEL_PATH" ]; then
    echo "Error: Final model not found at $FINAL_MODEL_PATH"; continue
  fi

  CUDA_VISIBLE_DEVICES=$GPU_ID python merge_save_bhra.py \
    --base_model "$MODEL" \
    --adapter_path "$FINAL_MODEL_PATH" \
    --output_path "$MERGED_MODEL_PATH"

  echo "=== Starting Evaluation ==="
  if [ ! -d "$MERGED_MODEL_PATH" ]; then
    echo "Error: Merged model not found at $MERGED_MODEL_PATH"; continue
  fi

  # Resolve absolute path for local directory so vLLM treats it as local model
  if command -v realpath >/dev/null 2>&1; then
    ABS_MERGED_MODEL_PATH=$(realpath "$MERGED_MODEL_PATH")
  else
    ABS_MERGED_MODEL_PATH=$(python - << 'PY'
import os, sys
print(os.path.abspath(sys.argv[1]))
PY
"$MERGED_MODEL_PATH")
  fi

  # Prepare eval data with explicit answer-tag requirement (does not modify eval code)
  GSM8K_TAGGED="$RUN_DIR/gsm8k_test_tagged.jsonl"
  MATH_TAGGED="$RUN_DIR/MATH_test_tagged.jsonl"
  python scripts/prepare_eval_answer_tag.py \
    --input data/math_eval/gsm8k_test.jsonl \
    --output "$GSM8K_TAGGED" \
    --type gsm8k
  python scripts/prepare_eval_answer_tag.py \
    --input data/math_eval/MATH_test.jsonl \
    --output "$MATH_TAGGED" \
    --type math

  CUDA_VISIBLE_DEVICES=$GPU_ID python instruction_tuning_eval/gsm8k_eval.py \
    --model "$ABS_MERGED_MODEL_PATH" \
    --data_file "$GSM8K_TAGGED" \
    --batch_size 128 \
    --tensor_parallel_size 1 \
    --run_dir "$RUN_DIR"

  CUDA_VISIBLE_DEVICES=$GPU_ID python instruction_tuning_eval/MATH_eval.py \
    --model "$ABS_MERGED_MODEL_PATH" \
    --data_file "$MATH_TAGGED" \
    --batch_size 64 \
    --tensor_parallel_size 1 \
    --run_dir "$RUN_DIR"

  echo "=== Cleaning up merged model ==="
  if [ -d "$MERGED_MODEL_PATH" ]; then
    rm -rf "$MERGED_MODEL_PATH" && echo "Cleanup done" || echo "Cleanup failed"
  else
    echo "Merged model directory not found - nothing to clean up"
  fi

  echo "=== Processing Complete for $RUN_DIR ==="
  cat << EOF > "$RUN_DIR/run_info.txt"
Run processed at: $(date)
Method: $METHOD
Base model: $MODEL
EOF

done

echo "=== All Run Directories Processed ==="
