from dataclasses import dataclass, field
import json
import pathlib
from typing import Dict, Optional

import numpy as np
import torch
from torch.utils.data import Dataset
import transformers
from transformers import Trainer
from transformers.trainer_pt_utils import LabelSmoother
from datasets import load_dataset

from conversation import SeparatorStyle
from conversation import get_conv_template

IGNORE_TOKEN_ID = LabelSmoother.ignore_index

@dataclass
class ModelArguments:
    model_name_or_path: Optional[str] = field(default="facebook/opt-125m")
    flash_attn: bool = False


@dataclass
class DataArguments:
    data_path: str = field(
        default=None, metadata={"help": "Path to the training data."}
    )
    lazy_preprocess: bool = False
    conv_template: str = field(default = "vicuna-1.1")


@dataclass
class TrainingArguments(transformers.TrainingArguments):
    cache_dir: Optional[str] = field(default=None)
    optim: str = field(default="adamw_torch")
    model_max_length: int = field(
        default=512,
        metadata={
            "help": "Maximum sequence length. Sequences will be right padded (and possibly truncated)."
        },
    )
    min_lr: float = field(
        default = None
    )


local_rank = None


def rank0_print(*args):
    if local_rank == 0:
        print(*args)


def safe_save_model_for_hf_trainer(trainer: transformers.Trainer, output_dir: str):
    """Collects the state dict and dump to disk."""
    state_dict = trainer.model.state_dict()
    if trainer.args.should_save:
        cpu_state_dict = {key: value.cpu() for key, value in state_dict.items()}
        del state_dict
        trainer._save(output_dir, state_dict=cpu_state_dict)  # noqa


def preprocess(
    sources,
    tokenizer: transformers.PreTrainedTokenizer,
    conv_template = "vicuna-1.1",
) -> Dict:

    conv = get_conv_template(conv_template)
    conv.sep = '\n'
    conv.sep_style = SeparatorStyle.ADD_COLON_NEW_LINE
    if tokenizer.eos_token != conv.sep2:
        if len(tokenizer("<|eot_id|>", add_special_tokens=False).input_ids) == 1:
            conv.sep2 = "<|eot_id|>"
        elif len(tokenizer("<|im_end|>", add_special_tokens=False).input_ids) == 1:
            conv.sep2 = "<|im_end|>"
        else:
            rank0_print(
                f"WARNING: tokenizer.eos_token is not the same as conv.sep2. "
                f"Using {conv.sep2} instead of {tokenizer.eos_token}"
            )
            conv.sep2 = tokenizer.eos_token

    roles = {"human": conv.roles[0], "gpt": conv.roles[1]}

    # Apply prompt templates
    conversations = []
    for i, source in enumerate(sources):
        if roles[source[0]["from"]] != conv.roles[0]:
            # Skip the first one if it is not from human
            source = source[1:]

        conv.messages = []
        for j, sentence in enumerate(source):
            role = roles[sentence["from"]]
            assert role == conv.roles[j % 2], breakpoint()
            conv.append_message(role, sentence["value"])
        conversations.append(conv.get_prompt())

    # Tokenize conversations
    input_ids = tokenizer(
        conversations,
        return_tensors="pt",
        padding="max_length",
        max_length=tokenizer.model_max_length,
        truncation=True,
    ).input_ids

    targets = input_ids.clone()

    assert (conv.sep_style == SeparatorStyle.ADD_COLON_NEW_LINE)

    if conv.sep_style == SeparatorStyle.ADD_COLON_NEW_LINE:
        sep = conv.sep + conv.roles[1] + ":\n"
        for idx, (conversation, target) in enumerate(zip(conversations, targets)):
            total_len = int(target.ne(tokenizer.pad_token_id).sum())
    
            turns = conversation.split(conv.sep2)
            
            if tokenizer.bos_token is not None:
                if tokenizer.padding_side == "right":
                    cur_len = 1
                    target[:cur_len] = IGNORE_TOKEN_ID
                else:
                    cur_len = int(target.eq(tokenizer.pad_token_id).sum()) + 1
                    target[:cur_len] = IGNORE_TOKEN_ID
            else:
                cur_len = 0

            for i, turn in enumerate(turns):
                if turn == "":
                    break
                turn_len = len(tokenizer(turn).input_ids)
    
                parts = turn.split(sep)
                if len(parts) != 2:
                    break
                parts[0] += sep

                if tokenizer.bos_token is not None:
                    # "-2" is hardcoded for the LLaMA tokenizer to make the offset correct.
                    instruction_len = len(tokenizer(parts[0]).input_ids) - 1
                else:
                    instruction_len = len(tokenizer(parts[0]).input_ids)
    
                if i != 0 and not tokenizer.legacy:
                    # The legacy and non-legacy modes handle special tokens differently
                    instruction_len -= 1

                # Ignore the user instructions
                target[cur_len : cur_len + instruction_len] = IGNORE_TOKEN_ID
                
                cur_len += turn_len
                if tokenizer.bos_token is None:
                    cur_len += 1

                if i != 0 and not tokenizer.legacy:
                    # The legacy and non-legacy modes handle special tokens differently
                    cur_len -= 1
    
            target[cur_len:] = IGNORE_TOKEN_ID
    
            if False:  # Inspect and check the correctness of masking
                z = target.clone()
                z = torch.where(z == IGNORE_TOKEN_ID, tokenizer.unk_token_id, z)
                rank0_print(tokenizer.decode(z))
    else:
        raise NotImplementedError()

    if cur_len < tokenizer.model_max_length:
        if cur_len != total_len:
            target[:] = IGNORE_TOKEN_ID
            rank0_print(
                    f"WARNING: tokenization mismatch: {cur_len}  vs. {total_len}."
                    f" (ignored)"
            )

    return dict(
        input_ids=input_ids,
        labels=targets,
        attention_mask=input_ids.ne(tokenizer.pad_token_id),
    )


class SupervisedDataset(Dataset):
    """Dataset for supervised fine-tuning."""

    def __init__(self, raw_data, tokenizer: transformers.PreTrainedTokenizer, conv_template = "vicuna-1.1"):
        super(SupervisedDataset, self).__init__()

        rank0_print("Formatting inputs...")
        sources = [example["conversations"] for example in raw_data]
        data_dict = preprocess(sources, tokenizer, conv_template)

        self.input_ids = data_dict["input_ids"]
        self.labels = data_dict["labels"]
        self.attention_mask = data_dict["attention_mask"]

    def __len__(self):
        return len(self.input_ids)

    def __getitem__(self, i) -> Dict[str, torch.Tensor]:
        return dict(
            input_ids=self.input_ids[i],
            labels=self.labels[i],
            attention_mask=self.attention_mask[i],
        )


class LazySupervisedDataset(Dataset):
    """Dataset for supervised fine-tuning."""

    def __init__(self, raw_data, tokenizer: transformers.PreTrainedTokenizer, conv_template = "vicuna-1.1"):
        super(LazySupervisedDataset, self).__init__()
        self.tokenizer = tokenizer

        rank0_print("Formatting inputs...Skip in lazy mode")
        self.conv_template = conv_template
        self.tokenizer = tokenizer
        self.raw_data = raw_data
        self.cached_data_dict = {}

    def __len__(self):
        return len(self.raw_data)

    def __getitem__(self, i) -> Dict[str, torch.Tensor]:
        if i in self.cached_data_dict:
            return self.cached_data_dict[i]

        ret = preprocess([self.raw_data[i]["conversations"]], self.tokenizer, self.conv_template)
        ret = dict(
            input_ids=ret["input_ids"][0],
            labels=ret["labels"][0],
            attention_mask=ret["attention_mask"][0],
        )
        self.cached_data_dict[i] = ret

        return ret


def make_supervised_data_module(
    tokenizer: transformers.PreTrainedTokenizer, data_args
) -> Dict:
    """Make dataset and collator for supervised fine-tuning."""
    conv_template = data_args.conv_template
    dataset_cls = (
        LazySupervisedDataset if data_args.lazy_preprocess else SupervisedDataset
    )
    rank0_print("Loading data...")

    try:
        train_json = json.load(open(data_args.data_path, "r"))
    except FileNotFoundError:
        train_json = load_dataset(data_args.data_path, split = "train")
        train_json = [row for row in train_json]
    
    train_dataset = dataset_cls(train_json, tokenizer=tokenizer, conv_template=conv_template)

    if hasattr(data_args, "eval_data_path") and data_args.eval_data_path:
        eval_json = json.load(open(data_args.eval_data_path, "r"))
        eval_dataset = dataset_cls(eval_json, tokenizer=tokenizer)
    else:
        eval_dataset = None

    return dict(train_dataset=train_dataset, eval_dataset=eval_dataset)

def train():
    global local_rank

    parser = transformers.HfArgumentParser(
        (ModelArguments, DataArguments, TrainingArguments)
    )
    model_args, data_args, training_args = parser.parse_args_into_dataclasses()
    training_args.do_eval = False
    local_rank = training_args.local_rank
    model = transformers.AutoModelForCausalLM.from_pretrained(
        model_args.model_name_or_path,
        cache_dir=training_args.cache_dir,
        torch_dtype=torch.bfloat16,
        attn_implementation="flash_attention_2",
    )
    model.config.use_cache = False
    tokenizer = transformers.AutoTokenizer.from_pretrained(
        model_args.model_name_or_path,
        cache_dir=training_args.cache_dir,
        model_max_length=training_args.model_max_length,
        padding_side="right",
        use_fast=False,
    )
    
    if hasattr(tokenizer, "unk_token") and tokenizer.unk_token is not None:
        tokenizer.pad_token = tokenizer.unk_token
    elif 'Llama-3' in model_args.model_name_or_path:
        tokenizer.pad_token = "<|finetune_right_pad_id|>"
    elif 'Qwen2' in model_args.model_name_or_path:
        tokenizer.pad_token = "<|endoftext|>"

    data_module = make_supervised_data_module(tokenizer=tokenizer, data_args=data_args)
    
    trainer = Trainer(
            model=model, tokenizer=tokenizer, args=training_args, **data_module
        )

    if list(pathlib.Path(training_args.output_dir).glob("checkpoint-*")):
        trainer.train(resume_from_checkpoint=True)
    else:
        trainer.train()

    trainer.save_state()
    trainer.save_model()


if __name__ == "__main__":
    train()