import logging
from pathlib import Path

import torch
from torch import nn
import torch.nn.functional as F
from torch.cuda.amp import GradScaler
import wandb
from sklearn.model_selection import train_test_split
from tqdm import trange
import hydra
from omegaconf import OmegaConf

from experiments.utils import (
    count_parameters,
    get_device,
    set_logger,
    set_seed,
)
from experiments.mixup_weights import mixup_weights_labels
from nn.models import DWSModelForClassification, MLPModelForClassification
from experiments.mnist.inr_trainer import create_object

set_logger()


@torch.no_grad()
def evaluate(model, loader, device, num_batches=None):
    model.eval()
    loss = 0.0
    correct = 0.0
    total = 0.0
    predicted, gt = [], []
    for i, batch in enumerate(loader):
        if num_batches is not None and i >= num_batches:
            break
        batch = batch.to(device)
        inputs = (batch.weights, batch.biases)
        out = model(inputs)
        loss += F.cross_entropy(out, batch.label, reduction="sum")
        total += len(batch.label)
        pred = out.argmax(1)
        correct += pred.eq(batch.label).sum()
        predicted.extend(pred.cpu().numpy().tolist())
        gt.extend(batch.label.cpu().numpy().tolist())

    model.train()
    avg_loss = loss / total
    avg_acc = correct / total

    return dict(avg_loss=avg_loss, avg_acc=avg_acc, predicted=predicted, gt=gt)


def train(cfg):
    OmegaConf.resolve(cfg)
    hydra_cfg = hydra.core.hydra_config.HydraConfig.get()
    ckpt_dir = Path(hydra_cfg.runtime.output_dir) / "checkpoints"
    ckpt_dir.mkdir(parents=True, exist_ok=True)

    if cfg.wandb.name is None:
        cfg.wandb.name = (
            f"mnist_clf_{cfg.model.name}_lr_{cfg.lr}"
            f"_bs_{cfg.batch_size}_seed_{cfg.seed}"
        )

    wandb.init(
        **cfg.wandb,
        settings=wandb.Settings(start_method="fork"),
        config=cfg,
    )

    device = get_device(gpus=cfg.gpu)

    # load dataset
    train_set = create_object(cfg.data.cls, **cfg.data.train)
    val_set = create_object(cfg.data.cls, **cfg.data.val)
    test_set = create_object(cfg.data.cls, **cfg.data.test)

    if cfg.n_samples is not None:
        _, train_indices = train_test_split(
            range(len(train_set)), test_size=cfg.n_samples
        )
        train_set = torch.utils.data.Subset(train_set, train_indices)

    train_loader = torch.utils.data.DataLoader(
        dataset=train_set,
        batch_size=cfg.batch_size,
        shuffle=True,
        num_workers=cfg.num_workers,
        pin_memory=True,
    )
    val_loader = torch.utils.data.DataLoader(
        dataset=val_set,
        batch_size=cfg.batch_size,
        num_workers=cfg.num_workers,
        shuffle=False,
        pin_memory=True,
    )
    test_loader = torch.utils.data.DataLoader(
        dataset=test_set,
        batch_size=cfg.batch_size,
        shuffle=False,
        num_workers=cfg.num_workers,
        pin_memory=True,
    )

    logging.info(
        f"train size {len(train_set)}, "
        f"val size {len(val_set)}, "
        f"test size {len(test_set)}"
    )

    point = train_set[0]
    weight_shapes = tuple(w.shape[:2] for w in point.weights)
    bias_shapes = tuple(b.shape[:1] for b in point.biases)

    layer_layout = [weight_shapes[0][0]] + [b[0] for b in bias_shapes]

    logging.info(f"weight shapes: {weight_shapes}, bias shapes: {bias_shapes}")

    # todo: make defaults for MLP so that parameters for MLP and DWS are the same
    if cfg.model.name == "mlp":
        model = MLPModelForClassification(
            in_dim=sum([w.numel() for w in weight_shapes + bias_shapes]),
            **cfg.model.kwargs,
        ).to(device)
    elif cfg.model.name == "dwsnet":
        model = DWSModelForClassification(
            weight_shapes=weight_shapes, bias_shapes=bias_shapes, **cfg.model.kwargs
        ).to(device)
    else:
        model = create_object(
            cfg.model.cls,
            layer_layout=layer_layout,
            **cfg.model.kwargs,
        ).to(device)

    logging.info(f"number of parameters: {count_parameters(model)}")

    if cfg.compile:
        model = torch.compile(model, **cfg.compile_kwargs)

    parameters = list(filter(lambda p: p.requires_grad, model.parameters()))
    optimizer = create_object(cfg.optim.cls, params=parameters, **cfg.optim.kwargs)
    if cfg.use_scheduler:
        scheduler = create_object(
            cfg.scheduler.cls,
            optimizer=optimizer,
            **cfg.scheduler.kwargs,
        )

    criterion = nn.CrossEntropyLoss()
    best_val_acc = -1
    best_test_results, best_val_results = None, None
    test_acc, test_loss = -1.0, -1.0
    global_step = 0
    start_epoch = 0

    if cfg.load_ckpt:
        ckpt = torch.load(cfg.load_ckpt)
        model.load_state_dict(ckpt["model"])
        if "optimizer" in ckpt:
            optimizer.load_state_dict(ckpt["optimizer"])
        if "scheduler" in ckpt:
            scheduler.load_state_dict(ckpt["scheduler"])
        if "epoch" in ckpt:
            start_epoch = ckpt["epoch"]
        if "global_step" in ckpt:
            global_step = ckpt["global_step"]
        logging.info(f"loaded checkpoint {cfg.load_ckpt}")

    epoch_iter = trange(start_epoch, cfg.n_epochs)
    model.train()

    if cfg.mixup:
        beta_dist = torch.distributions.beta.Beta(
            cfg.alpha * torch.ones(1, device=device), cfg.alpha * torch.ones(1, device=device))

    ckpt_dir = Path(hydra_cfg.runtime.output_dir) / wandb.run.path.split("/")[-1]
    ckpt_dir.mkdir(parents=True, exist_ok=True)

    scaler = GradScaler(**cfg.gradscaler)
    autocast_kwargs = dict(cfg.autocast)
    autocast_kwargs["dtype"] = getattr(torch, cfg.autocast.dtype, torch.float32)
    for epoch in epoch_iter:
        if cfg.mixup:
            batch_iter = enumerate(zip(train_loader, train_loader))
        else:
            batch_iter = enumerate(train_loader)
        for i, batch in batch_iter:
            optimizer.zero_grad()
            if cfg.mixup:
                inputs, label = mixup_weights_labels(
                    batch, beta_dist, device, 10,
                    optimal_perm=cfg.optimal_perm, random_perm=cfg.random_perm)
            else:
                batch = batch.to(device)
                inputs = (batch.weights, batch.biases)
                label = batch.label

            with torch.autocast(**autocast_kwargs):
                out = model(inputs)
                if cfg.mixup:
                    loss = -torch.mean(torch.sum(label * out.log_softmax(dim=-1), dim=-1))
                else:
                    loss = criterion(out, label)

            scaler.scale(loss).backward()
            scaler.unscale_(optimizer)
            log = {
                "train/loss": loss.item(),
                "global_step": global_step,
            }
            if cfg.clip_grad:
                grad_norm = torch.nn.utils.clip_grad_norm_(
                    parameters, cfg.clip_grad_max_norm
                )
                log["grad_norm"] = grad_norm
            scaler.step(optimizer)
            scaler.update()

            if cfg.use_scheduler:
                log["lr"] = scheduler.get_last_lr()[0]
                scheduler.step()
            wandb.log(log)

            epoch_iter.set_description(
                f"[{epoch} {i+1}], train loss: {loss.item():.3f}, test_loss: {test_loss:.3f}, test_acc: {test_acc:.3f}"
            )
            global_step += 1

        if (epoch + 1) % cfg.eval_every == 0 or epoch == cfg.n_epochs - 1:
            torch.save(
                {
                    "model": model.state_dict(),
                    "optimizer": optimizer.state_dict(),
                    "epoch": epoch,
                    "cfg": cfg,
                    "global_step": global_step,
                },
                ckpt_dir / "latest.ckpt",
            )

            val_loss_dict = evaluate(model, val_loader, device)
            test_loss_dict = evaluate(model, test_loader, device)
            val_loss = val_loss_dict["avg_loss"]
            val_acc = val_loss_dict["avg_acc"]
            test_loss = test_loss_dict["avg_loss"]
            test_acc = test_loss_dict["avg_acc"]

            train_loss_dict = evaluate(model, train_loader, device, num_batches=100)

            best_val_criteria = val_acc >= best_val_acc

            if best_val_criteria:
                torch.save(
                    {
                        "model": model.state_dict(),
                        "optimizer": optimizer.state_dict(),
                        "epoch": epoch,
                        "cfg": cfg,
                        "global_step": global_step,
                    },
                    ckpt_dir / "best_val.ckpt",
                )
                best_val_acc = val_acc
                best_test_results = test_loss_dict
                best_val_results = val_loss_dict

            log = {
                "train/avg_loss": train_loss_dict["avg_loss"],
                "train/acc": train_loss_dict["avg_acc"],
                "val/loss": val_loss,
                "val/acc": val_acc,
                "val/best_loss": best_val_results["avg_loss"],
                "val/best_acc": best_val_results["avg_acc"],
                "test/loss": test_loss,
                "test/acc": test_acc,
                "test/best_loss": best_test_results["avg_loss"],
                "test/best_acc": best_test_results["avg_acc"],
                # "test/conf_mat": wandb.plot.confusion_matrix(
                #     probs=None,
                #     y_true=test_loss_dict["gt"],
                #     preds=test_loss_dict["predicted"],
                #     class_names=range(10),
                # ),
                "epoch": epoch,
                "global_step": global_step,
            }

            wandb.log(log)


@hydra.main(config_path="configs", config_name="config", version_base=None)
def main(cfg):
    torch.set_float32_matmul_precision(cfg.matmul_precision)
    torch.backends.cudnn.benchmark = cfg.cudnn_benchmark
    if cfg.seed is not None:
        set_seed(cfg.seed)

    train(cfg)


if __name__ == "__main__":
    main()
