import os
import shutil
import subprocess

import numpy as np

bounds_mazda_problems = np.array([
        [0.9 , 1.5 ],
       [0.3 , 0.95],
       [1.1 , 2.1 ],
       [0.35, 1.  ],
       [1.5 , 2.3 ],
       [0.5 , 2.1 ],
       [1.3 , 2.1 ],
       [1.7 , 2.3 ],
       [1.1 , 1.7 ],
       [0.7 , 1.5 ],
       [1.7 , 2.6 ],
       [0.5 , 1.3 ],
       [1.1 , 1.7 ],
       [1.3 , 1.9 ],
       [1.1 , 1.7 ],
       [1.3 , 1.9 ],
       [0.9 , 2.1 ],
       [1.1 , 1.9 ],
       [1.5 , 2.3 ],
       [0.9 , 1.5 ],
       [0.9 , 1.5 ],
       [0.5 , 1.1 ],
       [0.5 , 1.2 ],
       [0.9 , 2.1 ],
       [0.3 , 0.9 ],
       [0.4 , 1.  ],
       [1.3 , 1.9 ],
       [0.6 , 1.3 ],
       [0.6 , 1.2 ],
       [0.4 , 1.7 ],
       [0.9 , 1.9 ],
       [0.6 , 1.3 ],
       [0.9 , 1.7 ],
       [0.7 , 1.3 ],
       [0.4 , 1.9 ],
       [1.5 , 2.3 ],
       [0.6 , 1.2 ],
       [0.3 , 0.9 ],
       [0.3 , 0.9 ],
       [0.6 , 1.2 ],
       [1.7 , 2.3 ],
       [0.9 , 1.7 ],
       [0.5 , 1.1 ],
       [1.3 , 1.9 ],
       [0.7 , 1.7 ],
       [0.7 , 2.1 ],
       [0.9 , 2.1 ],
       [1.3 , 2.1 ],
       [1.1 , 1.9 ],
       [1.3 , 2.1 ],
       [2.  , 2.6 ],
       [1.7 , 2.3 ],
       [1.1 , 1.7 ],
       [0.9 , 2.1 ],
       [0.3 , 0.9 ],
       [0.6 , 1.3 ],
       [0.9 , 1.7 ],
       [1.1 , 1.9 ],
       [1.3 , 1.9 ],
       [0.3 , 1.3 ],
       [0.3 , 1.1 ],
       [1.3 , 1.9 ],
       [0.9 , 1.5 ],
       [1.1 , 1.7 ],
       [1.7 , 2.3 ],
       [1.5 , 2.1 ],
       [1.1 , 1.6 ],
       [0.9 , 1.7 ],
       [0.7 , 2.1 ],
       [1.1 , 1.9 ],
       [0.9 , 1.7 ],
       [1.7 , 2.3 ],
       [2.  , 2.6 ],
       [0.5 , 1.3 ],
       [0.9 , 1.5 ],
       [0.3 , 0.95],
       [1.1 , 2.1 ],
       [0.35, 1.  ],
       [1.5 , 2.3 ],
       [0.5 , 2.1 ],
       [1.3 , 2.1 ],
       [1.7 , 2.3 ],
       [1.1 , 1.7 ],
       [0.7 , 1.5 ],
       [1.7 , 2.6 ],
       [0.5 , 1.3 ],
       [1.1 , 1.7 ],
       [1.3 , 1.9 ],
       [1.1 , 1.7 ],
       [1.3 , 1.9 ],
       [0.9 , 2.1 ],
       [1.1 , 1.9 ],
       [1.5 , 2.3 ],
       [0.9 , 1.5 ],
       [0.9 , 1.5 ],
       [0.5 , 1.1 ],
       [0.5 , 1.2 ],
       [0.9 , 2.1 ],
       [0.3 , 0.9 ],
       [0.4 , 1.  ],
       [1.3 , 1.9 ],
       [0.6 , 1.3 ],
       [0.6 , 1.2 ],
       [0.4 , 1.7 ],
       [0.9 , 1.9 ],
       [0.6 , 1.3 ],
       [0.9 , 1.7 ],
       [0.7 , 1.3 ],
       [0.4 , 1.9 ],
       [1.5 , 2.3 ],
       [0.6 , 1.2 ],
       [0.3 , 0.9 ],
       [0.3 , 0.9 ],
       [0.6 , 1.2 ],
       [1.7 , 2.3 ],
       [0.9 , 1.7 ],
       [0.5 , 1.1 ],
       [1.3 , 1.9 ],
       [0.7 , 1.7 ],
       [0.7 , 2.1 ],
       [0.9 , 2.1 ],
       [1.3 , 2.1 ],
       [1.1 , 1.9 ],
       [1.3 , 2.1 ],
       [2.  , 2.6 ],
       [1.7 , 2.3 ],
       [1.1 , 1.7 ],
       [0.9 , 2.1 ],
       [0.3 , 0.9 ],
       [0.6 , 1.3 ],
       [0.9 , 1.7 ],
       [1.1 , 1.9 ],
       [1.3 , 1.9 ],
       [0.3 , 1.3 ],
       [0.3 , 1.1 ],
       [1.3 , 1.9 ],
       [0.9 , 1.5 ],
       [1.1 , 1.7 ],
       [1.7 , 2.3 ],
       [1.5 , 2.1 ],
       [1.1 , 1.6 ],
       [0.9 , 1.7 ],
       [0.7 , 2.1 ],
       [1.1 , 1.9 ],
       [0.9 , 1.7 ],
       [1.7 , 2.3 ],
       [2.  , 2.6 ],
       [0.5 , 1.3 ],
       [0.9 , 1.5 ],
       [0.3 , 0.95],
       [1.1 , 2.1 ],
       [0.35, 1.  ],
       [1.5 , 2.3 ],
       [0.5 , 2.1 ],
       [1.3 , 2.1 ],
       [1.7 , 2.3 ],
       [1.1 , 1.7 ],
       [0.7 , 1.5 ],
       [1.7 , 2.6 ],
       [0.5 , 1.3 ],
       [1.1 , 1.7 ],
       [1.3 , 1.9 ],
       [1.1 , 1.7 ],
       [1.3 , 1.9 ],
       [0.9 , 2.1 ],
       [1.1 , 1.9 ],
       [1.5 , 2.3 ],
       [0.9 , 1.5 ],
       [0.9 , 1.5 ],
       [0.5 , 1.1 ],
       [0.5 , 1.2 ],
       [0.9 , 2.1 ],
       [0.3 , 0.9 ],
       [0.4 , 1.  ],
       [1.3 , 1.9 ],
       [0.6 , 1.3 ],
       [0.6 , 1.2 ],
       [0.4 , 1.7 ],
       [0.9 , 1.9 ],
       [0.6 , 1.3 ],
       [0.9 , 1.7 ],
       [0.7 , 1.3 ],
       [0.4 , 1.9 ],
       [1.5 , 2.3 ],
       [0.6 , 1.2 ],
       [0.3 , 0.9 ],
       [0.3 , 0.9 ],
       [0.6 , 1.2 ],
       [1.7 , 2.3 ],
       [0.9 , 1.7 ],
       [0.5 , 1.1 ],
       [1.3 , 1.9 ],
       [0.7 , 1.7 ],
       [0.7 , 2.1 ],
       [0.9 , 2.1 ],
       [1.3 , 2.1 ],
       [1.1 , 1.9 ],
       [1.3 , 2.1 ],
       [2.  , 2.6 ],
       [1.7 , 2.3 ],
       [1.1 , 1.7 ],
       [0.9 , 2.1 ],
       [0.3 , 0.9 ],
       [0.6 , 1.3 ],
       [0.9 , 1.7 ],
       [1.1 , 1.9 ],
       [1.3 , 1.9 ],
       [0.3 , 1.3 ],
       [0.3 , 1.1 ],
       [1.3 , 1.9 ],
       [0.9 , 1.5 ],
       [1.1 , 1.7 ],
       [1.7 , 2.3 ],
       [1.5 , 2.1 ],
       [1.1 , 1.6 ],
       [0.9 , 1.7 ],
       [0.7 , 2.1 ],
       [1.1 , 1.9 ],
       [0.9 , 1.7 ],
       [1.7 , 2.3 ],
       [2.  , 2.6 ],
       [0.5 , 1.3 ]
       ]
       )

class BaseMazdaCarsOptimization:
    '''
    Minimize the total weight and number of common parts of a car
    Constraints: g >= 0
    '''
    def __init__(self, normalize_objectives=True, **kwargs):
        self.input_dims = 222
        self.num_constraints = 54
        self.num_objectives = 2
        self.ref_point = np.array([1.1, 0.0])
        self.name = "mazda-car"
        self.executable_path = os.path.join(
            os.path.dirname(os.path.abspath(__file__)),
            "bin",
            "mazda_mop",
        )
        if not os.path.exists(self.executable_path):
            raise FileNotFoundError(f"Executable not found: {self.executable_path}")
        self.bounds = bounds_mazda_problems
        assert len(self.bounds) == self.input_dims
        self.temp_dir = os.path.join(
            os.path.dirname(os.path.abspath(__file__)),
            "temp",
        )

        # Normalize objectiv values to compute HV
        # See https://ladse.eng.isas.jaxa.jp/benchmark/
        self.normalize_objectives = normalize_objectives 

        try:
            self.initital_design = np.loadtxt('_test_functions/mazda_cars/mazda_initial_design.txt', delimiter=',', ndmin=2)
        except:
            print("Initial design not found")

    
    def __call__(self, x: np.ndarray):
        objectives, constraints = self._call(x)
        return (objectives, constraints)

    def _call(self, x: np.ndarray):
        """
        Evaluate Mazda benchmark for many designs

        Args:
            x: one input configuration

        Returns: (total_weight, number_of_common_parts, [constraint1, constraint2, ...])

        """
        if x.ndim == 1:
            assert len(x) == self.input_dims
            x = x.reshape(1, -1)
        elif x.ndim == 2:
            assert x.shape[1] == self.input_dims
        else:
            raise ValueError("x must be 1D or 2D array")
        
        # check in bound
        assert np.all(x >= self.bounds[:, 0])
        assert np.all(x <= self.bounds[:, 1])
        
        temp_dir = os.path.join(self.temp_dir, str(np.random.randint(0, 100000)))
        os.makedirs(temp_dir, exist_ok=True)

        # write input to file in dir
        input_file = os.path.join(temp_dir, "pop_vars_eval.txt")
        np.savetxt(input_file, x, fmt="%f", delimiter="\t")
        
        # pass directory as working directory to process
        cmd = f'{self.executable_path} {temp_dir}'
        res = subprocess.run(cmd, shell=True)
        # popen = subprocess.Popen(
        #     self.executable_path,
        #     stdout=subprocess.PIPE,
        #     cwd=temp_dir,
        # )
        # popen.wait()
        # read and parse output file - objective
        result_objectives = np.loadtxt(os.path.join(temp_dir, "pop_objs_eval.txt"), delimiter="\t", ndmin=2)
        result_constraints = np.loadtxt(os.path.join(temp_dir,"pop_cons_eval.txt"), delimiter="\t", ndmin=2)

        assert result_objectives.shape[0] == x.shape[0] == result_constraints.shape[0]

        # os.remove(os.path.join(temp_dir,"pop_vars_eval.txt"))
        # os.remove(os.path.join(temp_dir,"pop_objs_eval.txt"))
        # os.remove(os.path.join(temp_dir,"pop_cons_eval.txt"))
        shutil.rmtree(temp_dir)

        return_objectives = result_objectives[:, :2]
        return_constraints = result_constraints
        if self.normalize_objectives:
            return_objectives[:, 0] -= 2.0
            return_objectives[:, 1] /= 74.0
        return (return_objectives, return_constraints)
 


if __name__ == "__main__":
    mazda_cars_optimization = BaseMazdaCarsOptimization()
    x = np.random.uniform(bounds_mazda_problems[:, 0], bounds_mazda_problems[:, 1])
    results = mazda_cars_optimization(x)
    print(results)