

import argparse
import os
import random
import time
import warnings
import numpy as np
os.environ['CUDA_VISIBLE_DEVICES'] = '0,1,2,3,4'
import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets
from utils import *
from warmup_scheduler import GradualWarmupScheduler
#import bnn.models.resnet_imagenet as models
import bnn.models.resnet1 as models
from bnn.engine import BinaryChef
from bnn.models.vgg import VGG
model_names = sorted(name for name in models.__dict__
    if name.islower() and not name.startswith("__")
    and callable(models.__dict__[name]))

parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')
parser.add_argument('data', metavar='DIR', default='',
                    help='path to dataset')
parser.add_argument('-a', '--arch', metavar='ARCH', default='resnet18',
                    choices=model_names,
                    help='model architecture: ' +
                        ' | '.join(model_names) +
                        ' (default: resnet18)')
parser.add_argument('-j', '--workers', default=16, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('--epochs', default=90, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=256, type=int,
                    metavar='N',
                    help='mini-batch size (default: 256), this is the total '
                         'batch size of all GPUs on the current node when '
                         'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--lr', '--learning-rate', default=0.001, type=float,
                    metavar='LR', help='initial learning rate', dest='lr')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--wd', '--weight-decay', default=1e-5, type=float,
                    metavar='W', help='weight decay (default: 1e-4)',
                    dest='weight_decay')
parser.add_argument('-p', '--print-freq', default=100, type=int,
                    metavar='N', help='print frequency (default: 10)')
parser.add_argument('--resume', default='', type=str, metavar='PATH',
                    help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                    help='evaluate model on validation set')
parser.add_argument('--world-size', default=-1, type=int,
                    help='number of nodes for distributed training')
parser.add_argument('--rank', default=-1, type=int,
                    help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://127.0.0.1:23456', type=str,
                    help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str,
                    help='distributed backend')
parser.add_argument('--seed', default=None, type=int,
                    help='seed for initializing training. ')
parser.add_argument('--gpu', default='0,1,2,3,4', type=str,
                    help='GPU id to use.')
parser.add_argument('--multiprocessing-distributed', action='store_true',
                    help='Use multi-processing distributed training to launch '
                         'N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')
parser.add_argument('--optimizer', type=str, choices=['sgd', 'adam', 'adamw'], default='adamw')
parser.add_argument('--output-dir', type=str, default='experiments_/vit')
parser.add_argument('--scheduler', type=str, choices=['multistep', 'cosine'], default='cosine')
parser.add_argument('--warmup', type=int, default=0)
parser.add_argument('--stem-type', type=str, default='basic')
parser.add_argument('--resume-epoch', action='store_true', help='if enabled, will resume the epoch')
parser.add_argument('--step', type=int, default=2, choices=[0,1])
parser.add_argument('--pretrained', default= 'false', type= str)

best_acc1 = 0

def main():
    args = parser.parse_args()

    print(vars(args))

    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')
        
    if not os.path.isdir(args.output_dir):
        os.makedirs(args.output_dir)

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args):
    global best_acc1
    args.gpu = gpu

    if args.gpu is not None:
        print("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
    
    num_classes = 1000

    # create model
    #print("=> creating model '{}'".format(args.arch))
    model = models.__dict__[args.arch](stem_type=args.stem_type, num_classes=num_classes, block_type=models.PreBasicBlock, activation=nn.PReLU)
    if args.pretrained == 'true':
        print('##############################################')
        #checkpoint = torch.load('./checkpoint/ckpt_full.pth')
        #checkpoint = checkpoint['net']
        #checkpoint = torch.load('./checkpoint/resnet_imagenet/ckpt_epoch{}.pth'.format(20))
        #net.load_state_dict({k.replace('module.',''):v for k,v in checkpoint.items()})
        pre_weights = torch.load('./checkpoints/resnet18-f37072fd.pth', map_location='cpu')


        pre_dict = {k: v for k, v in pre_weights.items() if model.state_dict()[k].numel() == v.numel()}
        missing_keys, unexpected_keys = model.load_state_dict(pre_dict, strict=False)
    bchef = BinaryChef('examples/recepies/imagenet-baseline.yaml')
    model = bchef.run_step(model, 1)
    print(model)

    print('Num paramters: {}'.format(count_parameters(model)))
    model = nn.DataParallel(model, device_ids=[0, 1, 2, 3, 4]).cuda()
    '''
    if args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device([0,1,2,3,4])
            model.cuda([0,1,2,3,4])
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[0,1,2,3,4])
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)
    elif args.gpu is not None:
        #torch.cuda.set_device([0,1,2,3,4])
        #model = model.cuda([0,1,2,3,4])
        model = torch.nn.DataParallel(model,device_ids=[0,1,2,3,4,5]).cuda()
    else:
        # DataParallel will divide and allocate batch_size to all available GPUs
        if args.arch.startswith('alexnet') or args.arch.startswith('vgg'):
            model.features = torch.nn.DataParallel(model.features)
            model.cuda()
        else:
            model = torch.nn.DataParallel(model).cuda()
    '''
    # define loss function (criterion) and optimizer
    criterion = nn.CrossEntropyLoss().cuda()
    
    parameters = model.parameters()
    if args.optimizer == 'adamw':
        wd = args.weight_decay if args.step == 0 else 0
        optimizer = torch.optim.AdamW(parameters, args.lr, weight_decay=wd)
    elif args.optimizer == 'adam':
        optimizer = torch.optim.Adam(parameters, args.lr)
    elif args.optimizer == 'sgd':
        wd = 0 if args.step > 0 else args.weight_decay
        optimizer = torch.optim.SGD(parameters, args.lr,
                                momentum=args.momentum,
                                weight_decay=wd)
    else:
        raise ValueError('Unknown optimizer selected: {}'.format(args.optimizer))

    if args.scheduler == 'multistep':
        milestone = [25,50,75]
        lr_scheduler = torch.optim.lr_scheduler.MultiStepLR(optimizer, milestones=[x - args.warmup for x in milestone], gamma=0.1) # 
    elif args.scheduler == 'cosine':
        lr_scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(optimizer, float(args.epochs-args.warmup), eta_min=0)
    else:
        raise ValueError('Unknown schduler selected: {}'.format(args.scheduler))

    if args.warmup>0:
        print('=> Applying warmup ({} epochs)'.format(args.warmup))
        lr_scheduler = GradualWarmupScheduler(optimizer, multiplier=1, total_epoch=args.warmup, after_scheduler=lr_scheduler)

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume)
            else:
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc)
            if args.resume_epoch:
                args.start_epoch = checkpoint['epoch']
                best_acc1 = checkpoint['best_acc1']
            if args.gpu is not None:
                pass
                # best_acc1 may be from a checkpoint from a different GPU
                #best_acc1 = best_acc1.to(args.gpu)
            try:
                model.load_state_dict(checkpoint['state_dict'])
                if not ('adam' in args.optimizer and 'sgd' in args.resume):
                    print('=> Loading optimizer...')
                    #optimizer.load_state_dict(checkpoint['optimizer'])
            except:
                print('=> Warning: dict model mismatch, loading with strict = False')
                model.load_state_dict(checkpoint['state_dict'], strict=False)
            print("=> loaded checkpoint '{}' (epoch {})"
                  .format(args.resume, checkpoint['epoch']))
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))
        
        # Reset learning rate
        for g in optimizer.param_groups:
            g['lr'] = args.lr
        

    if args.start_epoch>0:
        print('Advancing the scheduler to epoch {}'.format(args.start_epoch))
        for i in range(args.start_epoch):
            lr_scheduler.step()
    cudnn.benchmark = True

    # Data loading code
    traindir = os.path.join(args.data, 'train')
    valdir = os.path.join(args.data, 'val')


    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])
    transforms_train = transforms.Compose([
                        transforms.Resize(256),
                        transforms.RandomResizedCrop(224),
                        transforms.RandomHorizontalFlip(),
                        transforms.ToTensor(),
                        normalize,
                ])
    transforms_val = transforms.Compose([
                transforms.Resize(256),
                transforms.CenterCrop(224),
                transforms.ToTensor(),
                normalize,
            ])
    
    train_dataset = datasets.ImageFolder(
                traindir,
                transforms_train)
    val_dataset = datasets.ImageFolder(valdir, transforms_val)
            

    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    else:
        train_sampler = None

    train_loader = torch.utils.data.DataLoader(
        train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
        num_workers=args.workers, pin_memory=True, sampler=train_sampler)

    val_loader = torch.utils.data.DataLoader(
        val_dataset,
        batch_size=args.batch_size, shuffle=False,
        num_workers=args.workers, pin_memory=True)

    if args.evaluate:
        validate(val_loader, model, criterion, args)
        return

    show_logs = (not args.multiprocessing_distributed) or (args.multiprocessing_distributed
                and args.rank % ngpus_per_node == 0)

    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_sampler.set_epoch(epoch)
        if args.scheduler == 'cosine':
            lr_scheduler.step(epoch)
        else:
            lr_scheduler.step()
        if show_logs:
            print('New lr: {}'.format(lr_scheduler.get_last_lr()))

        # train for one epoch
        train(train_loader, model, criterion, optimizer, epoch, args, show_logs)

        # evaluate on validation set
        acc1 = validate(val_loader, model, criterion, args, show_logs)
        

        # remember best acc@1 and save checkpoint
        is_best = acc1 > best_acc1
        best_acc1 = max(acc1, best_acc1)
        
        print('Current best: {}'.format(best_acc1))
        
        if show_logs:
            save_checkpoint({
                'epoch': epoch + 1,
                'arch': args.arch,
                'state_dict': model.state_dict(),
                'best_acc1': best_acc1,
                'optimizer' : optimizer.state_dict(),
            }, is_best, args.output_dir)

def train(train_loader, model, criterion, optimizer, epoch, args, show_logs=True):
    batch_time = AverageMeter('Time', ':6.3f')
    data_time = AverageMeter('Data', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    
    all_meters = [batch_time, data_time, losses, top1, top5]
    
    progress = ProgressMeter(
        len(train_loader),
        all_meters,
        prefix="Epoch: [{}]".format(epoch))

    # switch to train mode
    model.train()

    end = time.time()
    for i, (images, target) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)

        if args.gpu is not None:
            images = images.cuda()
        target = target.cuda()

        # compute output
        output = model(images)
        
        loss = criterion(output, target)

        # measure accuracy and record loss
        acc1, acc5 = accuracy(output, target, topk=(1, 5))
        losses.update(loss.item(), images.size(0))
        top1.update(acc1[0], images.size(0))
        top5.update(acc5[0], images.size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0 and show_logs:
            progress.display(i)


def validate(val_loader, model, criterion, args, show_logs=True):
    batch_time = AverageMeter('Time', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    progress = ProgressMeter(
        len(val_loader),
        [batch_time, losses, top1, top5],
        prefix='Test: ')

    # switch to evaluate mode
    model.eval()

    with torch.no_grad():
        end = time.time()
        for i, (images, target) in enumerate(val_loader):
            if args.gpu is not None:
                images = images.cuda()
            target = target.cuda()
            
            output = model(images)
            loss = criterion(output, target)

            # measure accuracy and record loss
            acc1, acc5 = accuracy(output, target, topk=(1, 5))
            losses.update(loss.item(), images.size(0))
            top1.update(acc1[0], images.size(0))
            top5.update(acc5[0], images.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0 and show_logs:
                progress.display(i)

        # TODO: this should also be done with the ProgressMeter
        if show_logs:
            print(' * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}'
                .format(top1=top1, top5=top5))

    return top1.avg

if __name__ == '__main__':
    main()
