# model definitions based on https://github.com/pytorch/vision/blob/master/torchvision/models/resnet.py
from typing import Type, Any, Callable, Union, List, Optional
import torch
import torch.nn as nn

from layers import BasicBlock, Bottleneck, HBlock, PreBasicBlock, PreBottleneck
from layers import conv1x1


class DaBNNStem(nn.Module):
    def __init__(self, planes: int, norm_layer: Optional[Callable[..., nn.Module]] = None,
                 activation=nn.ReLU):
        super(DaBNNStem, self).__init__()

        self.conv1 = nn.Sequential(
            nn.Conv2d(3, planes // 2, kernel_size=3, stride=2, padding=1, bias=False),
            norm_layer(planes // 2),
            activation()
        )
        self.conv2_1 = nn.Sequential(
            nn.Conv2d(planes // 2, planes // 4, 1, 1, bias=False),
            norm_layer(planes // 4),
            activation()
        )
        self.conv2_2 = nn.Sequential(
            nn.Conv2d(planes // 4, planes // 2, kernel_size=3, stride=2, padding=1, bias=False),
            norm_layer(planes // 2),
            activation()
        )
        self.conv3 = nn.Sequential(
            nn.Conv2d(planes, planes, 1, 1, bias=False),
            norm_layer(planes),
            activation()
        )
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        x = self.conv1(x)
        x = torch.cat(
            [
                self.conv2_2(self.conv2_1(x)),
                self.maxpool(x)
            ],
            dim=1
        )
        x = self.conv3(x)
        return x


class ResNet(nn.Module):

    def __init__(
        self,
        block: Type[Union[BasicBlock, Bottleneck, HBlock, PreBasicBlock, PreBottleneck]],
        layers: List[int],
        num_classes: int = 10,
        zero_init_residual: bool = False,
        groups: int = 1,
        width_per_group: int = 64,
        replace_stride_with_dilation: Optional[List[bool]] = None,
        norm_layer: Optional[Callable[..., nn.Module]] = None,
        activation: Optional[Callable[..., nn.Module]] = None,
        stem_type: str = 'basic'
    ) -> None:
        super(ResNet, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if activation is None:
            activation = nn.ReLU
        self._norm_layer = norm_layer
        self._activation = activation

        self.stem_type = stem_type
        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        if stem_type == 'basic':
            self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=3, stride=1, padding=1,
                                   bias=False)
            self.bn1 = norm_layer(self.inplanes)
        elif stem_type == 'dabnn':
            self.conv1 = DaBNNStem(self.inplanes, norm_layer=norm_layer)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=1, stride=1, padding=0)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(self.outplanes, num_classes)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)


        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)  # type: ignore[arg-type]
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)  # type: ignore[arg-type]

    def _make_layer(self, block: Type[Union[BasicBlock, Bottleneck]], planes: int, blocks: int,
                    stride: int = 1, dilate: bool = False) -> nn.Sequential:
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.AvgPool2d(kernel_size=stride, stride=stride, ceil_mode=True, count_include_pad=False),
                conv1x1(self.inplanes, planes * block.expansion, stride=1),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer, activation=self._activation))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer, activation=self._activation))
        self.outplanes = planes

        return nn.Sequential(*layers)

    def _forward_impl(self, x: torch.Tensor) -> torch.Tensor:
        # See note [TorchScript super()]
        x = self.conv1(x)
        if self.stem_type == 'basic':
            x = self.bn1(x)
            x = self.relu(x)
            x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = x.view(x.size(0), -1)
        #x = torch.flatten(x, 1)
        x = self.fc(x)

        return x

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        return self._forward_impl(x)


def _resnet(
    arch: str,
    block: Type[Union[BasicBlock, Bottleneck, HBlock, PreBottleneck, PreBasicBlock]],
    layers: List[int],
    **kwargs: Any
) -> ResNet:
    model = ResNet(block, layers, **kwargs)
    return model


def resnet18(block_type: Optional[Type[Union[BasicBlock, Bottleneck, HBlock,
                                             PreBottleneck, PreBasicBlock]]] = None, **kwargs: Any) -> ResNet:
    r"""ResNet-18 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet18', BasicBlock if block_type is None else block_type, [2, 2, 2, 2],
                   **kwargs)


def resnet34(block_type: Optional[Type[Union[BasicBlock, Bottleneck, HBlock,
                                             PreBottleneck, PreBasicBlock]]] = None, **kwargs: Any) -> ResNet:
    r"""ResNet-34 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet34', BasicBlock if block_type is None else block_type, [3, 4, 6, 3],
                   **kwargs)


def resnet50(block_type: Optional[Type[Union[BasicBlock, Bottleneck, HBlock,
                                             PreBottleneck, PreBasicBlock]]] = None, **kwargs: Any) -> ResNet:
    r"""ResNet-50 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet50', Bottleneck if block_type is None else block_type, [3, 4, 6, 3],
                   **kwargs)
