import argparse
import os
import shutil
import time

import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.optim
import torch.utils.data
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import resnet
from adabound import AdaBound

import numpy as np
import random

import pickle

model_names = sorted(name for name in resnet.__dict__
    if name.islower() and not name.startswith("__")
                     and name.startswith("resnet")
                     and callable(resnet.__dict__[name]))

print(model_names)

parser = argparse.ArgumentParser(description='Propert ResNets for CIFAR10 in pytorch')
parser.add_argument('--arch', '-a', metavar='ARCH', default='resnet32',
                    choices=model_names,
                    help='model architecture: ' + ' | '.join(model_names) +
                    ' (default: resnet32)')
parser.add_argument('-j', '--workers', default=4, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('--epochs', default=160, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=128, type=int,
                    metavar='N', help='mini-batch size (default: 128)')
parser.add_argument('--val-batch-size', type=int, default=128, metavar='N',
                        help='input batch size for validating (default: 128)')
parser.add_argument('--test-batch-size', type=int, default=128, metavar='N',
                        help='input batch size for testing (default: 128)')
parser.add_argument('--print-freq', '-p', default=50, type=int,
                    metavar='N', help='print frequency (default: 1)')
parser.add_argument('--resume', default='', type=str, metavar='PATH',
                    help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                    help='evaluate model on validation set')
parser.add_argument('--pretrained', dest='pretrained', action='store_true',
                    help='use pre-trained model')
parser.add_argument('--half', dest='half', action='store_true',help='use half-precision(16-bit) ')
parser.add_argument('--save-dir', dest='save_dir',
                    help='The directory used to save the trained models',
                    default='save_temp', type=str)
parser.add_argument('--save-every', dest='save_every',
                    help='Saves checkpoints at every specified number of epochs',
                    type=int, default=10)
parser.add_argument('--dump-data',default='output.ser',type=str,metavar='PATH',
                    help='path to save loss\correct data (default: output.ser)')
parser.add_argument('--seed', type=int, default=1, metavar='S',help='random seed (default: 1)')
parser.add_argument('--lr', '--learning-rate', default=0.1, type=float,
                    metavar='LR', help='initial learning rate')
parser.add_argument('--lr_decay', default=0.1, type=float,help='learning_rate decay')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',help='momentum')
parser.add_argument('--weight_decay', '--wd', default=1e-4, type=float,
                    metavar='W', help='weight decay (default: 1e-4)')
parser.add_argument('--beta1', default=0.9, type=float, help='Adam coefficients beta_1')
parser.add_argument('--beta2', default=0.999, type=float, help='Adam coefficients beta_2')
parser.add_argument('--eps', default=1e-8, type=float, help='eps for var adam')
parser.add_argument('--lr-gamma', default=0.1, type=float, help='learning rate')
parser.add_argument('--final_lr', default=0.1, type=float,help='final learning rate of AdaBound')
parser.add_argument('--gamma', default=1e-3, type=float,help='convergence speed term of AdaBound')
parser.add_argument('--beta', default=1.0, type=float,help='beta (default: 1.0)')
parser.add_argument('--alpha', default=1.0, type=float,help='alpha (default: 1.0)')
parser.add_argument('--damp', default=1e-2, type=float,help='damp (default: 1e-2)')
parser.add_argument('--damp1', default=1e-2, type=float,help='damp (default: 1e-2)')
parser.add_argument('--damp2', default=1e-2, type=float,help='damp (default: 1e-2)')
parser.add_argument('--period', default=1, type=int,help='period (default: 1)')
parser.add_argument('--hist_length', default=10, type=int,help='hist-length (default: 10)')
parser.add_argument('--pullback_momentum',default='reset',type=str,
                    help='pullback_momentum for lookahead(default: reset)')
parser.add_argument('--optim', default='sgdm', type=str, help='optimizer',
                        choices=['sgdm', 'adam', 'adadelta','adamw', 'adabelief','adabound','amsgrad','lookahead','sdlbfgs',
                                 'adasam','ca','rstbfgs'])
# parser.add_argument('--partition', action='store_true', default=False,
#                         help='partition training dataset into traindata and validatedata')

best_prec1 = 0

results = {'trainloss_val':[],'trainloss_avg':[],'trainprec_val':[],'trainprec_avg':[],
            'valloss_val':[],'valloss_avg':[],'valprec_val':[],'valprec_avg':[],'testprec':[],'testloss':[]}

def setup_seed(seed):
     torch.manual_seed(seed)
     torch.cuda.manual_seed(seed)
     torch.cuda.manual_seed_all(seed)
     # np.random.seed(seed)
     # random.seed(seed)
     #torch.backends.cudnn.deterministic = True

def main():
    global args, best_prec1
    args = parser.parse_args()
    setup_seed(args.seed)
    #torch.manual_seed(1)

    # Check the save_dir exists or not
    if not os.path.exists(args.save_dir):
        os.makedirs(args.save_dir)

    main_start_time = time.time()

    model = torch.nn.DataParallel(resnet.__dict__[args.arch](),device_ids=[0])
    model.cuda()

#    device = torch.device('cuda0')
#    model = resnet

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            checkpoint = torch.load(args.resume)
            args.start_epoch = checkpoint['epoch']
            best_prec1 = checkpoint['best_prec1']
            model.load_state_dict(checkpoint['state_dict'])
            print("=> loaded checkpoint '{}' (epoch {})"
                  .format(args.evaluate, checkpoint['epoch']))
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])

    train_loader = torch.utils.data.DataLoader(
        datasets.CIFAR10(root='./data', train=True, transform=transforms.Compose([
            transforms.RandomHorizontalFlip(),
            transforms.RandomCrop(32, 4),
            transforms.ToTensor(),
            normalize,
        ]), download=True),
        batch_size=args.batch_size, shuffle=True,
        num_workers=args.workers, pin_memory=True)

    val_loader = torch.utils.data.DataLoader(
        datasets.CIFAR10(root='./data', train=False, transform=transforms.Compose([
            transforms.ToTensor(),
            normalize,
        ])),
        batch_size=args.val_batch_size, shuffle=True,
        num_workers=args.workers, pin_memory=True)

    test_loader = val_loader

    print('train_loader.dataset:',len(train_loader.dataset))
    print('val_loader.dataset:',len(val_loader.dataset))

    # define loss function (criterion) and optimizer
    criterion = nn.CrossEntropyLoss(reduction='mean').cuda()

    if args.half:
        model.half()
        criterion.half()

    optimizer = getOptimizer(args,model.parameters())
    if args.optim not in ['lookahead','adasam']:
        lr_scheduler = torch.optim.lr_scheduler.MultiStepLR(optimizer,
                                                    milestones=[80, 120], last_epoch=args.start_epoch - 1,gamma=args.lr_decay)
    else:
        lr_scheduler = torch.optim.lr_scheduler.MultiStepLR(optimizer.optimizer,
                                                            milestones=[80, 120], last_epoch=args.start_epoch - 1,gamma=args.lr_decay)
    if args.arch in ['resnet1202', 'resnet110']:
        # for resnet1202 original paper uses lr=0.01 for first 400 minibatches for warm-up
        # then switch back. In this setup it will correspond for first epoch.
        for param_group in optimizer.param_groups:
            param_group['lr'] = args.lr*0.1

    if args.evaluate:
        validate(val_loader, model, criterion)
        return

    best_epoch = 0
    for epoch in range(args.start_epoch, args.epochs):
        if args.optim in ['rstbfgs']:
            if epoch == 80:
                #optimizer.alpha = optimizer.alpha * 0.1
                optimizer.beta = optimizer.beta * 0.1
            if epoch == 120:
                #optimizer.alpha = optimizer.alpha * 0.1
                optimizer.beta = optimizer.beta * 0.1

        # train for one epoch
        print('current lr {:.5e}'.format(optimizer.param_groups[0]['lr']))
        train(train_loader, model, criterion, optimizer, epoch)
        lr_scheduler.step()

        # evaluate on validation set
        prec1 = validate(val_loader, model, criterion)

        # remember best prec@1 and save checkpoint
        is_best = prec1 > best_prec1
        #best_prec1 = max(prec1, best_prec1)
        if is_best:
            best_prec1 = prec1
            best_epoch = epoch

        if epoch > 0 and epoch % args.save_every == 0:
            save_checkpoint({
                'epoch': epoch + 1,
                'state_dict': model.state_dict(),
                'best_prec1': best_prec1,
            }, is_best, filename=os.path.join(args.save_dir, 'checkpoint.th'))

        save_checkpoint({
            'state_dict': model.state_dict(),
            'best_prec1': best_prec1,
        }, is_best, filename=os.path.join(args.save_dir, 'model.th'))

        pickle.dump(results,open(args.dump_data,'wb'))

    print('best epoch: {}, best_valprec1: {:.3f}'.format(best_epoch,best_prec1))
    print('test...')
    test(test_loader, model, criterion)
    pickle.dump(results, open(args.dump_data, 'wb'))

    main_end_time = time.time()
    print('total running time: {:.3f} seconds'.format(main_end_time-main_start_time))

def train(train_loader, model, criterion, optimizer, epoch):
    """
        Run one train epoch
    """
    batch_time = AverageMeter()
    data_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()

    # switch to train mode
    model.train()

    end = time.time()
    for i, (input, target) in enumerate(train_loader):

        # measure data loading time
        data_time.update(time.time() - end)

        target = target.cuda()
        input_var = input.cuda()
        target_var = target
        if args.half:
            input_var = input_var.half()

        # compute output
        def closure():
            output = model(input_var)
            loss = criterion(output, target_var)
            optimizer.zero_grad()
            loss.backward()
            return loss,output
        loss,output = optimizer.step(closure)

        output = output.float()
        loss = loss.float()
        # measure accuracy and record loss
        prec1 = accuracy(output.data, target)[0]
        losses.update(loss.item(), input.size(0))
        top1.update(prec1.item(), input.size(0))

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            print('Epoch: [{0}][{1}/{2}]\t'
                  'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                  'Data {data_time.val:.3f} ({data_time.avg:.3f})\t'
                  'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                  'Prec@1 {top1.val:.3f} ({top1.avg:.3f})'.format(
                      epoch, i, len(train_loader), batch_time=batch_time,
                      data_time=data_time, loss=losses, top1=top1))

    results['trainloss_val'].append(losses.val)
    results['trainloss_avg'].append(losses.avg)
    results['trainprec_val'].append(top1.val)
    results['trainprec_avg'].append(top1.avg)


def validate(val_loader, model, criterion):
    """
    Run evaluation
    """
    batch_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()

    # switch to evaluate mode
    model.eval()

    end = time.time()
    with torch.no_grad():
        for i, (input, target) in enumerate(val_loader):
            target = target.cuda()
            input_var = input.cuda()
            target_var = target.cuda()

            if args.half:
                input_var = input_var.half()

            # compute output
            output = model(input_var)
            loss = criterion(output, target_var)

            output = output.float()
            loss = loss.float()

            # measure accuracy and record loss
            prec1 = accuracy(output.data, target)[0]
            losses.update(loss.item(), input.size(0))
            top1.update(prec1.item(), input.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0:
                print('Test: [{0}/{1}]\t'
                      'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                      'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                      'Prec@1 {top1.val:.3f} ({top1.avg:.3f})'.format(
                          i, len(val_loader), batch_time=batch_time, loss=losses,
                          top1=top1))

    results['valloss_val'].append(losses.val)
    results['valloss_avg'].append(losses.avg)
    results['valprec_val'].append(top1.val)
    results['valprec_avg'].append(top1.avg)


    print(' * Prec@1 {top1.avg:.3f}'
          .format(top1=top1))

    return top1.avg

def test(test_loader, model, criterion):
    """
    Run evaluation
    """
    batch_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()

    # switch to evaluate mode
    model.eval()

    end = time.time()
    with torch.no_grad():
        for i, (input, target) in enumerate(test_loader):
            target = target.cuda()
            input_var = input.cuda()
            target_var = target.cuda()

            if args.half:
                input_var = input_var.half()

            # compute output
            output = model(input_var)
            loss = criterion(output, target_var)

            output = output.float()
            loss = loss.float()

            # measure accuracy and record loss
            prec1 = accuracy(output.data, target)[0]
            losses.update(loss.item(), input.size(0))
            top1.update(prec1.item(), input.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0:
                print('Test: [{0}/{1}]\t'
                      'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                      'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                      'Prec@1 {top1.val:.3f} ({top1.avg:.3f})'.format(
                          i, len(test_loader), batch_time=batch_time, loss=losses,
                          top1=top1))

    print(' * Prec@1 {top1.avg:.3f}'
          .format(top1=top1))

    results['testloss'].append(losses.avg)
    results['testprec'].append(top1.avg)

    return top1.avg

def getOptimizer(args, model_params):
    args.optim = args.optim.lower()
    if args.optim == 'sgdm':
        return torch.optim.SGD(model_params, lr=args.lr, momentum=args.momentum,
                         weight_decay=args.weight_decay)
    elif args.optim == 'adam':
        return torch.optim.Adam(model_params, lr=args.lr, betas=(args.beta1, args.beta2),
                          weight_decay=args.weight_decay, eps=args.eps)
    elif args.optim == 'adadelta':
        return torch.optim.Adadelta(model_params,lr=args.lr,weight_decay=args.weight_decay)
    elif args.optim == 'amsgrad':
        return torch.optim.Adam(model_params, lr=args.lr, betas=(args.beta1, args.beta2),
                                weight_decay=args.weight_decay, eps=args.eps, amsgrad=True)
    elif args.optim == 'adamw':
        return torch.optim.AdamW(model_params, lr=args.lr, betas=(args.beta1, args.beta2),
                          weight_decay=args.weight_decay, eps=args.eps)
    elif args.optim == 'adabelief':
        return torch.optim.AdaBelief(model_params, args.lr, betas=(args.beta1, args.beta2),
                          weight_decay=args.weight_decay, eps=args.eps)
    elif args.optim == 'adabound':
        return AdaBound(model_params, args.lr, betas=(args.beta1, args.beta2),
                        final_lr=args.final_lr, gamma=args.gamma,
                        weight_decay=args.weight_decay)
    elif args.optim == 'lookahead':
        sgd = torch.optim.SGD(model_params, lr=args.lr, momentum=args.momentum,
                         weight_decay=args.weight_decay)
        return torch.optim.Lookahead(sgd,la_steps=args.hist_length,la_alpha=args.alpha,pullback_momentum=args.pullback_momentum)
    elif args.optim == 'sdlbfgs':
        return torch.optim.SdLBFGS(model_params,lr=args.lr,lr_decay=False,weight_decay=args.weight_decay,history_size=args.hist_length)
    elif args.optim == 'adasam':
        sgd = torch.optim.SGD(model_params,lr=args.lr,weight_decay=args.weight_decay)
        return torch.optim.AdaSAM(sgd,period=args.period,hist_length=args.hist_length,
                               damp=args.damp,beta=args.beta,alpha=args.alpha)
    elif args.optim == 'ca':
        sgd = torch.optim.SGD(model_params,lr=args.lr,weight_decay=args.weight_decay)
        return torch.optim.CA(sgd,beta=args.beta,alpha=args.alpha,
                               damp1=args.damp1,damp2=args.damp2)
    elif args.optim == 'rstbfgs':
        sgd = torch.optim.SGD(model_params,lr=args.lr,weight_decay=args.weight_decay)
        return torch.optim.RSTBFGS(sgd,beta=args.beta,alpha=args.alpha,
                               damp1=args.damp1,damp2=args.damp2,precision=0)
    else:
        print('Optimizer not found')

def save_checkpoint(state, is_best, filename='checkpoint.pth.tar'):
    """
    Save the training model
    """
    torch.save(state, filename)

class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self):
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count


def accuracy(output, target, topk=(1,)):
    """Computes the precision@k for the specified values of k"""
    maxk = max(topk)
    batch_size = target.size(0)

    _, pred = output.topk(maxk, 1, True, True)
    pred = pred.t()
    correct = pred.eq(target.view(1, -1).expand_as(pred))

    res = []
    for k in topk:
        correct_k = correct[:k].view(-1).float().sum(0)
        res.append(correct_k.mul_(100.0 / batch_size))
    return res


if __name__ == '__main__':
    main()
