import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import torch.backends.cudnn as cudnn
import torchvision
import torchvision.transforms as transforms
import argparse
import os
import sys
from models import *

seed = 0
torch.manual_seed(seed)
torch.cuda.manual_seed(seed)
torch.cuda.manual_seed_all(seed)
torch.backends.cudnn.benchmark=False
torch.backends.cudnn.deterministic=True

parser = argparse.ArgumentParser(description='CIFAR10 Adversarial Training')
parser.add_argument('--lr', default=0.01, type=float, help='learning rate')
parser.add_argument('--alpha', '-a', default=2/255, type=float, help='attack step size')
parser.add_argument('--eps', '-e', default=8/255, type=float, help='eps for attack')
parser.add_argument('--K', '-K', default=5, type=int, help='attack step')
parser.add_argument('--epoch', default=100, type=int, help='training epoch')
parser.add_argument('--model','-model', default='mobilenet_v2', help='only support mobilenet_v2 here now!')
parser.add_argument('--linbp_attack', '-lin_a', action='store_true',help='use LinBP for attack')
parser.add_argument('--linbp_train', '-lin_t', action='store_true',help='use LinBP for train')
args = parser.parse_args()

_train_acc = []
_test_acc = []
_test_acc_adv = []
learning_rate = args.lr
epsilon = args.eps
k = args.K
alpha = args.alpha
file_name = 'pgd_adversarial_training'

device = 'cuda' if torch.cuda.is_available() else 'cpu'

transform_train = transforms.Compose([
    transforms.RandomCrop(32, padding=4),
    transforms.RandomHorizontalFlip(),
    transforms.ToTensor(),
])

transform_test = transforms.Compose([
    transforms.ToTensor(),
])

train_dataset = torchvision.datasets.CIFAR10(root='./data', train=True, download=True, transform=transform_train)
test_dataset = torchvision.datasets.CIFAR10(root='./data', train=False, download=True, transform=transform_test)

train_loader = torch.utils.data.DataLoader(train_dataset, batch_size=128, shuffle=True, num_workers=4)
test_loader = torch.utils.data.DataLoader(test_dataset, batch_size=100, shuffle=False, num_workers=4)

class LinfPGDAttack(object):
    def __init__(self, model):
        self.model = model

    def perturb(self, x_natural, y, Lin = False):
        x = x_natural.detach()
        x = x + torch.zeros_like(x).uniform_(-epsilon, epsilon)
        for i in range(k):
            x.requires_grad_()
            with torch.enable_grad():
                logits = self.model(x,linbp = Lin)
                loss = F.cross_entropy(logits, y)
            grad = torch.autograd.grad(loss, [x])[0]
            x = x.detach() + alpha * torch.sign(grad.detach())
            x = torch.min(torch.max(x, x_natural - epsilon), x_natural + epsilon)
            x = torch.clamp(x, 0, 1)
        return x

def attack(x, y, model, adversary):
    model_copied = copy.deepcopy(model)
    model_copied.eval()
    adversary.model = model_copied
    adv = adversary.perturb(x, y)
    return adv

if(args.model == 'mobilenet_v2'):
    net = mobilenet_v2()

    net = net.to(device)
    net = torch.nn.DataParallel(net)
else:
    print('The model name is wrong!')
    sys.exit()

cudnn.benchmark = True

adversary = LinfPGDAttack(net)
criterion = nn.CrossEntropyLoss()
optimizer = optim.SGD(net.parameters(), lr=learning_rate, momentum=0.9, weight_decay=0.0002)

def train(epoch):
    print('\n[ Train epoch: %d ]' % epoch)
    net.train()
    train_loss = 0
    correct = 0
    total = 0
    for batch_idx, (inputs, targets) in enumerate(train_loader):
        inputs, targets = inputs.to(device), targets.to(device)
        optimizer.zero_grad()

        adv = adversary.perturb(inputs, targets,Lin=args.linbp_attack)
        adv_outputs = net(adv,linbp = args.linbp_train)
        loss = criterion(adv_outputs, targets)
        loss.backward()

        optimizer.step()
        train_loss += loss.item()
        _, predicted = adv_outputs.max(1)

        total += targets.size(0)
        correct += predicted.eq(targets).sum().item()

        if batch_idx % 10 == 0:
            print('Current batch:', str(batch_idx))
            #print('Current adversarial train accuracy:', str(predicted.eq(targets).sum().item() / targets.size(0)))
            #print('Current adversarial train loss:', loss.item())

    print('\nTotal adversarial train accuarcy:', 100. * correct / total)
    #print('Total adversarial train loss:', train_loss)
    _train_acc.append(100. * correct / total)

def test(epoch):
    print('\n[ Test epoch: %d ]' % epoch)
    net.eval()
    benign_loss = 0
    adv_loss = 0
    benign_correct = 0
    adv_correct = 0
    total = 0
    with torch.no_grad():
        for batch_idx, (inputs, targets) in enumerate(test_loader):
            inputs, targets = inputs.to(device), targets.to(device)
            total += targets.size(0)

            outputs = net(inputs)
            loss = criterion(outputs, targets)
            benign_loss += loss.item()

            _, predicted = outputs.max(1)
            benign_correct += predicted.eq(targets).sum().item()

            if batch_idx % 10 == 0:
                print('Current batch:', str(batch_idx))
                #print('Current benign test accuracy:', str(predicted.eq(targets).sum().item() / targets.size(0)))
                #print('Current benign test loss:', loss.item())

            adv = adversary.perturb(inputs, targets, Lin=False)
            adv_outputs = net(adv)
            loss = criterion(adv_outputs, targets)
            adv_loss += loss.item()

            _, predicted = adv_outputs.max(1)
            adv_correct += predicted.eq(targets).sum().item()




            '''
            if batch_idx % 10 == 0:
                print('Current adversarial test accuracy:', str(predicted.eq(targets).sum().item() / targets.size(0)))
                print('Current adversarial test loss:', loss.item())
            '''

    print('\nTotal benign test accuarcy:', 100. * benign_correct / total)
    print('Total adversarial test Accuarcy:', 100. * adv_correct / total)
    #print('Total benign test loss:', benign_loss)
    #print('Total adversarial test loss:', adv_loss)
    #print('Total adversarial test loss2:', adv_loss2)

    _test_acc.append(benign_correct / total)
    _test_acc_adv.append(100. * adv_correct / total)


    state = {
        'net': net.state_dict()
    }
    if not os.path.isdir('checkpoint'):
        os.mkdir('checkpoint')
    torch.save(state, './checkpoint/' + file_name)
    print('Model Saved!')

def adjust_learning_rate(optimizer, epoch):
    lr = learning_rate
    if epoch >= 100:
        lr /= 10
    if epoch >= 150:
        lr /= 10
    for param_group in optimizer.param_groups:
        param_group['lr'] = lr

for epoch in range(0, args.epoch):
    adjust_learning_rate(optimizer, epoch)
    train(epoch)
    test(epoch)

print(_train_acc)
print(_test_acc)
print(_test_acc_adv)
