import copy
from PIL import Image
from typing import Any, Callable, List, Optional, Tuple, Union

import numpy as np

import torch
from torchvision import datasets
import torchvision.transforms.functional as F


class ImageDataset(datasets.vision.VisionDataset):
    def __init__(self, images: List[str], labels: Union[List, np.ndarray, torch.Tensor],
                 transform: Optional[Callable] = None, 
                 target_transform: Optional[Callable] = None,
                 to_rgb: bool = False) -> None:
        self.images = images
        self.labels = labels
        self.transform = transform
        self.target_transform = target_transform
        self.to_rgb = to_rgb

        sample_image = Image.open(images[0])
        if to_rgb:
            sample_image = sample_image.convert('RGB')
        if transform:
            sample_image = transform(sample_image)
        else:
            sample_image = F.to_tensor(sample_image)

        sample_label = labels[0]
        if target_transform:
            sample_label = target_transform(sample_label)
        else:
            sample_label = torch.as_tensor(sample_label)

        self.num_outcomes = self.get_num_outcomes(sample_image)
        self.num_treatments = self.get_num_treatments(sample_label)
        self.num_covariates = self.get_num_covariates()

        self.nb_dims = sample_image.ndim

    def get_num_outcomes(self, image):
        return tuple(image.size())

    def get_num_treatments(self, label):
        return len(label)

    def get_num_covariates(self):
        return [1] # dummy covariate

    def get_cf_image(self, index: int): 
        """
        Only overridden for morphomnist.
        """
        return None

    def get_cf_label(self, index: int):
        label = self.labels[index]
        cf_index = np.random.choice(len(self.labels))
        cf_field = np.random.choice(len(label))
        cf_label = copy.deepcopy(label)
        cf_label[cf_field] = self.labels[cf_index][cf_field]

        return cf_label

    def __getitem__(self, index: int) -> Any:
        image = Image.open(self.images[index])
        if self.to_rgb:
            image = image.convert('RGB')
        if self.transform:
            image = self.transform(image)
        else:
            image = F.to_tensor(image)

        label = self.labels[index]
        cf_label = self.get_cf_label(index)

        if self.target_transform:
            label = self.target_transform(label)
            cf_label = self.target_transform(cf_label)
        else:
            label = torch.as_tensor(label)
            cf_label = torch.as_tensor(cf_label)

        cf_image = None
        if hasattr(self, "split") and self.split != "train": 
            cf_image = self.get_cf_image(index)

        return ( # Y, T, X, T', Ys under X and T'
            # image, label, [torch.tensor([0.])], cf_label, self.get_cf_image(index)
            image, label, [torch.tensor([0.])], cf_label, cf_image
        )

    def __len__(self) -> int:
        return len(self.labels)
