from onpolicy.debug import debug_print
import torch
import numpy as np
import torch.nn.functional as F
from onpolicy.utils.util import get_shape_from_obs_space, get_shape_from_act_space
from onpolicy.utils.reward_scaling import RunningRewardScaler


def _flatten(T, N, x):
    return x.reshape(T * N, *x.shape[2:])


def _cast(x):
    return x.transpose(1, 2, 0, *list(range(3, len(x.shape)))).reshape(-1, *x.shape[3:])


def _shuffle_agent_grid(x, y):
    rows = np.indices((x, y))[0]
    # cols = np.stack([np.random.permutation(y) for _ in range(x)])
    cols = np.stack([np.arange(y) for _ in range(x)])
    return rows, cols

class SharedReplayBuffer(object):
    """
    Buffer to store training data.
    :param args: (argparse.Namespace) arguments containing relevant model, policy, and env information.
    :param num_agents: (int) number of agents in the env.
    :param obs_space: (gym.Space) observation space of agents.
    :param cent_obs_space: (gym.Space) centralized observation space of agents.
    :param act_space: (gym.Space) action space for agents.
    """

    def __init__(self, args, num_agents, obs_space, cent_obs_space, act_space):
        self.episode_length = args.episode_length
        self.n_rollout_threads = args.n_rollout_threads
        self.hidden_size = args.hidden_size
        self.recurrent_N = args.recurrent_N
        self.value_dim = args.value_dim
        self.n_timesteps = args.n_timesteps
        self.gamma = args.gamma
        self.gae_lambda = args.gae_lambda
        self._use_gae = args.use_gae
        self._use_popart = args.use_popart
        self._use_valuenorm = args.use_valuenorm
        self._use_proper_time_limits = args.use_proper_time_limits
        self.algo = args.algorithm_name
        self.num_agents = num_agents
        self.rnum_agents = args.rnum_agents
        self.use_attention = args.use_attention
        self.use_symmetry = args.use_symmetry
        rnum_agents = args.rnum_agents
        self.act_step = args.act_step
        self.use_latent_actions = args.use_latent_actions
        self.sep_logprob = args.sep_logprob
        self.use_latent_prob = args.use_latent_prob
        if self.use_latent_actions:
            # self.value_dim = self.n_timesteps
            self.value_dim = 1
            
        self.running_reward_scaling = args.running_reward_scaling
        
        if args.running_reward_scaling:
            self.reward_scaler = RunningRewardScaler(self.n_rollout_threads)

        obs_shape = get_shape_from_obs_space(obs_space)
        share_obs_shape = get_shape_from_obs_space(cent_obs_space)

        if type(obs_shape[-1]) == list:
            obs_shape = obs_shape[:1]

        if type(share_obs_shape[-1]) == list:
            share_obs_shape = share_obs_shape[:1]

        self.share_obs = np.zeros((self.episode_length + 1, self.n_rollout_threads, num_agents, *share_obs_shape),
                                  dtype=np.float32)
        # if self.use_attention:
        #     self.obs = np.zeros((self.episode_length + 1, self.n_rollout_threads, rnum_agents, *obs_shape), dtype=np.float32)
        # else:
        self.obs = np.zeros((self.episode_length + 1, self.n_rollout_threads, num_agents, *obs_shape), dtype=np.float32)
        self.obs_sym = np.zeros((self.episode_length + 1, self.n_rollout_threads, rnum_agents, *obs_shape), dtype=np.float32)
        
        # if self.use_symmetry:
        #     self.share_obs = np.zeros((self.episode_length + 1, self.n_rollout_threads, rnum_agents, *share_obs_shape),
        #                           dtype=np.float32)
        #     self.obs = np.zeros((self.episode_length + 1, self.n_rollout_threads, rnum_agents, *obs_shape), dtype=np.float32)

        self.rnn_states = np.zeros(
            (self.episode_length + 1, self.n_rollout_threads, num_agents, self.recurrent_N, self.hidden_size),
            dtype=np.float32)
        self.rnn_states_critic = np.zeros_like(self.rnn_states)

        self.value_preds = np.zeros(
            (self.episode_length + 1, self.n_rollout_threads, num_agents, self.value_dim), dtype=np.float32)
        self.advantages = np.zeros(
            (self.episode_length, self.n_rollout_threads, num_agents, self.value_dim), dtype=np.float32)
        

        if act_space[0].__class__.__name__ == 'Discrete':
            self.act_size = act_space[0].n
            self.available_actions = np.ones((self.episode_length + 1, self.n_rollout_threads, 1, rnum_agents * act_space[0].n),
                                             dtype=np.float32)
        else:
            self.act_size = act_space[0].shape[0]
            self.available_actions = None
        
        # debug_print(act_space)
        act_shape, latent_act_shape = get_shape_from_act_space(act_space)
        
        self.actions = np.zeros(
            (self.episode_length, self.n_rollout_threads, num_agents, act_shape), dtype=np.float32)
        self.action_log_probs = np.zeros(
            (self.episode_length, self.n_rollout_threads, num_agents, act_shape), dtype=np.float32)
        self.rewards = np.zeros(
            (self.episode_length, self.n_rollout_threads, num_agents, 1), dtype=np.float32)
        

        if self.use_latent_actions:
            self.latent_actions = np.zeros(
                (self.episode_length, self.n_rollout_threads, num_agents, self.n_timesteps + 1, latent_act_shape), dtype=np.float32)
            self.noise = np.zeros(
                (self.episode_length, self.n_rollout_threads, num_agents, self.n_timesteps + 1, latent_act_shape), dtype=np.float32)
            self.sampled_actions = np.zeros(
                (self.episode_length, self.n_rollout_threads, num_agents, latent_act_shape), dtype=np.float32)
            self.action_log_probs = np.zeros(
                (self.episode_length, self.n_rollout_threads, num_agents, self.n_timesteps, 1), dtype=np.float32)
            if self.sep_logprob:
                self.action_log_probs_last = np.zeros(
                    (self.episode_length, self.n_rollout_threads, num_agents, rnum_agents, latent_act_shape), dtype=np.float32)
            else:
                self.action_log_probs_last = np.zeros(
                    (self.episode_length, self.n_rollout_threads, num_agents, rnum_agents), dtype=np.float32)
            self.rewards = np.zeros(
                (self.episode_length, self.n_rollout_threads, rnum_agents, 1), dtype=np.float32)
            self.value_preds = np.zeros(
                (self.episode_length + 1, self.n_rollout_threads, rnum_agents, self.value_dim), dtype=np.float32)
            # self.value_preds = np.zeros(
            #     (self.episode_length + 1, self.n_rollout_threads, num_agents, self.n_timesteps, self.value_dim), dtype=np.float32)
        else:
            self.latent_actions = np.zeros(
                (self.episode_length, self.n_rollout_threads, num_agents, self.n_timesteps + 1, latent_act_shape), dtype=np.float32) # To be fixed
        # debug_print(self.latent_actions.size, self.sampled_actions.size, self.action_log_probs.size, self.value_preds.size, self.returns.size, self.advantages.size, self.actions.size, self.action_log_probs.size, self.rewards.size)
        self.returns = np.zeros_like(self.value_preds)

        self.masks = np.ones((self.episode_length + 1, self.n_rollout_threads, num_agents, 1), dtype=np.float32)
        self.bad_masks = np.ones_like(self.masks)
        # self.active_masks = np.ones_like(self.masks)
        self.active_masks = np.ones((self.episode_length + 1, self.n_rollout_threads, rnum_agents, 1), dtype=np.float32)

        self.step = 0

    def insert(self, share_obs, obs, obs_sym, rnn_states_actor, rnn_states_critic, actions, action_log_probs, action_log_probs_last,
               value_preds, rewards, masks, bad_masks=None, active_masks=None, available_actions=None, latent_actions=None, sampled_actions=None, noise=None):
        """
        Insert data into the buffer.
        :param share_obs: (argparse.Namespace) arguments containing relevant model, policy, and env information.
        :param obs: (np.ndarray) local agent observations.
        :param rnn_states_actor: (np.ndarray) RNN states for actor network.
        :param rnn_states_critic: (np.ndarray) RNN states for critic network.
        :param actions:(np.ndarray) actions taken by agents.
        :param action_log_probs:(np.ndarray) log probs of actions taken by agents
        :param value_preds: (np.ndarray) value function prediction at each step.
        :param rewards: (np.ndarray) reward collected at each step.
    :param masks: (np.ndarray) denotes whether the environment has terminated or not.
        :param bad_masks: (np.ndarray) action space for agents.
        :param active_masks: (np.ndarray) denotes whether an agent is active or dead in the env.
        :param available_actions: (np.ndarray) actions available to each agent. If None, all actions are available.
        """
        self.share_obs[self.step + 1] = share_obs.copy()
        self.obs[self.step + 1] = obs.copy()
        self.obs_sym[self.step + 1] = obs_sym.copy()
        self.rnn_states[self.step + 1] = rnn_states_actor.copy()
        self.rnn_states_critic[self.step + 1] = rnn_states_critic.copy()
        self.actions[self.step] = actions.copy()
        self.action_log_probs[self.step] = action_log_probs.copy()
        self.action_log_probs_last[self.step] = action_log_probs_last.copy()
        if self.use_latent_actions:
            self.latent_actions[self.step] = latent_actions.copy()
            self.sampled_actions[self.step] = sampled_actions.copy()
        # debug_print('fa', value_preds.shape)
        self.value_preds[self.step] = value_preds.copy()
        self.rewards[self.step] = rewards.copy()
        self.masks[self.step + 1] = masks.copy()
        if bad_masks is not None:
            self.bad_masks[self.step + 1] = bad_masks.copy()
        if active_masks is not None:
            self.active_masks[self.step + 1] = active_masks.copy()
        if available_actions is not None:
            self.available_actions[self.step + 1] = available_actions.copy()
        if noise is not None:
            self.noise[self.step] = noise.copy()

        self.step = (self.step + 1) % self.episode_length

    def chooseinsert(self, share_obs, obs, rnn_states, rnn_states_critic, actions, action_log_probs,
                     value_preds, rewards, masks, bad_masks=None, active_masks=None, available_actions=None):
        """
        Insert data into the buffer. This insert function is used specifically for Hanabi, which is turn based.
        :param share_obs: (argparse.Namespace) arguments containing relevant model, policy, and env information.
        :param obs: (np.ndarray) local agent observations.
        :param rnn_states_actor: (np.ndarray) RNN states for actor network.
        :param rnn_states_critic: (np.ndarray) RNN states for critic network.
        :param actions:(np.ndarray) actions taken by agents.
        :param action_log_probs:(np.ndarray) log probs of actions taken by agents
        :param value_preds: (np.ndarray) value function prediction at each step.
        :param rewards: (np.ndarray) reward collected at each step.
        :param masks: (np.ndarray) denotes whether the environment has terminated or not.
        :param bad_masks: (np.ndarray) denotes indicate whether whether true terminal state or due to episode limit
        :param active_masks: (np.ndarray) denotes whether an agent is active or dead in the env.
        :param available_actions: (np.ndarray) actions available to each agent. If None, all actions are available.
        """
        self.share_obs[self.step] = share_obs.copy()
        self.obs[self.step] = obs.copy()
        self.rnn_states[self.step + 1] = rnn_states.copy()
        self.rnn_states_critic[self.step + 1] = rnn_states_critic.copy()
        self.actions[self.step] = actions.copy()
        self.action_log_probs[self.step] = action_log_probs.copy()
        self.value_preds[self.step] = value_preds.copy()
        self.rewards[self.step] = rewards.copy()
        self.masks[self.step + 1] = masks.copy()
        if bad_masks is not None:
            self.bad_masks[self.step + 1] = bad_masks.copy()
        if active_masks is not None:
            self.active_masks[self.step] = active_masks.copy()
        if available_actions is not None:
            self.available_actions[self.step] = available_actions.copy()

        self.step = (self.step + 1) % self.episode_length

    def after_update(self):
        """Copy last timestep data to first index. Called after update to model."""
        self.share_obs[0] = self.share_obs[-1].copy()
        self.obs[0] = self.obs[-1].copy()
        self.rnn_states[0] = self.rnn_states[-1].copy()
        self.rnn_states_critic[0] = self.rnn_states_critic[-1].copy()
        self.masks[0] = self.masks[-1].copy()
        self.bad_masks[0] = self.bad_masks[-1].copy()
        self.active_masks[0] = self.active_masks[-1].copy()
        if self.available_actions is not None:
            self.available_actions[0] = self.available_actions[-1].copy()

    def chooseafter_update(self):
        """Copy last timestep data to first index. This method is used for Hanabi."""
        self.rnn_states[0] = self.rnn_states[-1].copy()
        self.rnn_states_critic[0] = self.rnn_states_critic[-1].copy()
        self.masks[0] = self.masks[-1].copy()
        self.bad_masks[0] = self.bad_masks[-1].copy()

    def compute_returns(self, next_value, value_normalizer=None):
        """
        Compute returns either as discounted sum of rewards, or using GAE.
        :param next_value: (np.ndarray) value predictions for the step after the last episode step.
        :param value_normalizer: (PopArt) If not None, PopArt value normalizer instance.
        """
        if self._use_proper_time_limits:
            if self._use_gae:
                self.value_preds[-1] = next_value
                gae = 0
                for step in reversed(range(self.rewards.shape[0])):
                    if self._use_popart or self._use_valuenorm:
                        # step + 1
                        delta = self.rewards[step] + self.gamma * value_normalizer.denormalize(
                            self.value_preds[step + 1]) * self.masks[step + 1] \
                                - value_normalizer.denormalize(self.value_preds[step])
                        gae = delta + self.gamma * self.gae_lambda * gae * self.masks[step + 1]
                        gae = gae * self.bad_masks[step + 1]
                        self.returns[step] = gae + value_normalizer.denormalize(self.value_preds[step])
                    else:
                        delta = self.rewards[step] + self.gamma * self.value_preds[step + 1] * self.masks[step + 1] - \
                                self.value_preds[step]
                        gae = delta + self.gamma * self.gae_lambda * self.masks[step + 1] * gae
                        gae = gae * self.bad_masks[step + 1]
                        self.returns[step] = gae + self.value_preds[step]
            else:
                self.returns[-1] = next_value
                for step in reversed(range(self.rewards.shape[0])):
                    if self._use_popart or self._use_valuenorm:
                        self.returns[step] = (self.returns[step + 1] * self.gamma * self.masks[step + 1] + self.rewards[
                            step]) * self.bad_masks[step + 1] \
                                             + (1 - self.bad_masks[step + 1]) * value_normalizer.denormalize(
                            self.value_preds[step])
                    else:
                        self.returns[step] = (self.returns[step + 1] * self.gamma * self.masks[step + 1] + self.rewards[
                            step]) * self.bad_masks[step + 1] \
                                             + (1 - self.bad_masks[step + 1]) * self.value_preds[step]
        else:
            if self._use_gae:
                self.value_preds[-1] = next_value
                gae = 0
                for step in reversed(range(self.rewards.shape[0])):
                    if self._use_popart or self._use_valuenorm:
                        if self.algo == "mat" or self.algo == "mat_dec":
                            value_t = value_normalizer.denormalize(self.value_preds[step])
                            value_t_next = value_normalizer.denormalize(self.value_preds[step + 1])
                            rewards_t = self.rewards[step]

                            # mean_v_t = np.mean(value_t, axis=-2, keepdims=True)
                            # mean_v_t_next = np.mean(value_t_next, axis=-2, keepdims=True)
                            # delta = rewards_t + self.gamma * self.masks[step + 1] * mean_v_t_next - mean_v_t

                            delta = rewards_t + self.gamma * self.masks[step + 1] * value_t_next - value_t
                            gae = delta + self.gamma * self.gae_lambda * self.masks[step + 1] * gae
                            self.advantages[step] = gae
                            self.returns[step] = gae + value_t
                        else:
                            delta = self.rewards[step] + self.gamma * value_normalizer.denormalize(
                                self.value_preds[step + 1]) * self.masks[step + 1] \
                                    - value_normalizer.denormalize(self.value_preds[step])
                            gae = delta + self.gamma * self.gae_lambda * self.masks[step + 1] * gae
                            self.returns[step] = gae + value_normalizer.denormalize(self.value_preds[step])
                    else:
                        if self.algo == "mat" or self.algo == "mat_dec":
                            rewards_t = self.rewards[step]
                            mean_v_t = np.mean(self.value_preds[step], axis=-2, keepdims=True)
                            mean_v_t_next = np.mean(self.value_preds[step + 1], axis=-2, keepdims=True)
                            delta = rewards_t + self.gamma * self.masks[step + 1] * mean_v_t_next - mean_v_t

                            # delta = rewards_t + self.gamma * self.value_preds[step + 1] * \
                            #         self.masks[step + 1] - self.value_preds[step]
                            gae = delta + self.gamma * self.gae_lambda * self.masks[step + 1] * gae
                            self.advantages[step] = gae
                            self.returns[step] = gae + self.value_preds[step]

                        else:
                            delta = self.rewards[step] + self.gamma * self.value_preds[step + 1] * \
                                    self.masks[step + 1] - self.value_preds[step]
                            gae = delta + self.gamma * self.gae_lambda * self.masks[step + 1] * gae
                            self.returns[step] = gae + self.value_preds[step]
            else:
                self.returns[-1] = next_value
                for step in reversed(range(self.rewards.shape[0])):
                    self.returns[step] = self.returns[step + 1] * self.gamma * self.masks[step + 1] + self.rewards[step]
    
    def scale_reward(self):
        if self.running_reward_scaling:
            print(self.rewards.shape, self.masks.shape)
            self.rewards[:, :, 0, 0] = self.reward_scaler(self.rewards[:, :, 0, 0].T, 1 - self.masks[:-1, :, 0, 0].T).T

    def feed_forward_generator_transformer(self, advantages, num_mini_batch=None, mini_batch_size=None):
        """
        Yield training data for MLP policies.
        :param advantages: (np.ndarray) advantage estimates.
        :param num_mini_batch: (int) number of minibatches to split the batch into.
        :param mini_batch_size: (int) number of samples in each minibatch.
        """
        episode_length, n_rollout_threads, num_agents = self.rewards.shape[0:3]
        batch_size = n_rollout_threads * episode_length

        if mini_batch_size is None:
            assert batch_size >= num_mini_batch, (
                "PPO requires the number of processes ({}) "
                "* number of steps ({}) = {} "
                "to be greater than or equal to the number of PPO mini batches ({})."
                "".format(n_rollout_threads, episode_length,
                          n_rollout_threads * episode_length,
                          num_mini_batch))
            mini_batch_size = batch_size // num_mini_batch

        rand = torch.randperm(batch_size).numpy()
        sampler = [rand[i * mini_batch_size:(i + 1) * mini_batch_size] for i in range(num_mini_batch)]
        rows, cols = _shuffle_agent_grid(batch_size, num_agents)

        # keep (num_agent, dim)
        share_obs = self.share_obs[:-1].reshape(-1, *self.share_obs.shape[2:])
        share_obs = share_obs[rows, cols]
        obs = self.obs[:-1].reshape(-1, *self.obs.shape[2:])
        obs = obs[rows, cols]
        rnn_states = self.rnn_states[:-1].reshape(-1, *self.rnn_states.shape[2:])
        rnn_states = rnn_states[rows, cols]
        rnn_states_critic = self.rnn_states_critic[:-1].reshape(-1, *self.rnn_states_critic.shape[2:])
        rnn_states_critic = rnn_states_critic[rows, cols]
        actions = self.actions.reshape(-1, *self.actions.shape[2:])
        actions = actions[rows, cols]
        if self.available_actions is not None:
            available_actions = self.available_actions[:-1].reshape(-1, *self.available_actions.shape[2:])
            available_actions = available_actions[rows, cols]
        value_preds = self.value_preds[:-1].reshape(-1, *self.value_preds.shape[2:])
        value_preds = value_preds[rows, cols]
        returns = self.returns[:-1].reshape(-1, *self.returns.shape[2:])
        returns = returns[rows, cols]
        masks = self.masks[:-1].reshape(-1, *self.masks.shape[2:])
        masks = masks[rows, cols]
        active_masks = self.active_masks[:-1].reshape(-1, *self.active_masks.shape[2:])
        active_masks = active_masks[rows, cols]
        action_log_probs = self.action_log_probs.reshape(-1, *self.action_log_probs.shape[2:])
        action_log_probs = action_log_probs[rows, cols]
        advantages = advantages.reshape(-1, *advantages.shape[2:])
        advantages = advantages[rows, cols]

        for indices in sampler:
            # [L,T,N,Dim]-->[L*T,N,Dim]-->[index,N,Dim]-->[index*N, Dim]
            share_obs_batch = share_obs[indices].reshape(-1, *share_obs.shape[2:])
            obs_batch = obs[indices].reshape(-1, *obs.shape[2:])
            rnn_states_batch = rnn_states[indices].reshape(-1, *rnn_states.shape[2:])
            rnn_states_critic_batch = rnn_states_critic[indices].reshape(-1, *rnn_states_critic.shape[2:])
            actions_batch = actions[indices].reshape(-1, *actions.shape[2:])
            if self.available_actions is not None:
                available_actions_batch = available_actions[indices].reshape(-1, *available_actions.shape[2:])
            else:
                available_actions_batch = None
            value_preds_batch = value_preds[indices].reshape(-1, *value_preds.shape[2:])
            return_batch = returns[indices].reshape(-1, *returns.shape[2:])
            masks_batch = masks[indices].reshape(-1, *masks.shape[2:])
            active_masks_batch = active_masks[indices].reshape(-1, *active_masks.shape[2:])
            old_action_log_probs_batch = action_log_probs[indices].reshape(-1, *action_log_probs.shape[2:])
            if advantages is None:
                adv_targ = None
            else:
                adv_targ = advantages[indices].reshape(-1, *advantages.shape[2:])

            yield share_obs_batch, obs_batch, rnn_states_batch, rnn_states_critic_batch, actions_batch, \
                  value_preds_batch, return_batch, masks_batch, active_masks_batch, old_action_log_probs_batch, \
                  adv_targ, available_actions_batch

    def feed_forward_generator(self, advantages, num_mini_batch=None, mini_batch_size=None):
        """
        Yield training data for MLP policies.
        :param advantages: (np.ndarray) advantage estimates.
        :param num_mini_batch: (int) number of minibatches to split the batch into.
        :param mini_batch_size: (int) number of samples in each minibatch.
        """
        episode_length, n_rollout_threads, num_agents = self.rewards.shape[0:3]
        batch_size = n_rollout_threads * episode_length * num_agents

        if mini_batch_size is None:
            assert batch_size >= num_mini_batch, (
                "PPO requires the number of processes ({}) "
                "* number of steps ({}) * number of agents ({}) = {} "
                "to be greater than or equal to the number of PPO mini batches ({})."
                "".format(n_rollout_threads, episode_length, num_agents,
                          n_rollout_threads * episode_length * num_agents,
                          num_mini_batch))
            mini_batch_size = batch_size // num_mini_batch

        rand = torch.randperm(batch_size).numpy()
        sampler = [rand[i * mini_batch_size:(i + 1) * mini_batch_size] for i in range(num_mini_batch)]

        share_obs = self.share_obs[:-1].reshape(-1, *self.share_obs.shape[3:])
        obs = self.obs[:-1].reshape(-1, *self.obs.shape[3:])
        rnn_states = self.rnn_states[:-1].reshape(-1, *self.rnn_states.shape[3:])
        rnn_states_critic = self.rnn_states_critic[:-1].reshape(-1, *self.rnn_states_critic.shape[3:])
        actions = self.actions.reshape(-1, self.actions.shape[-1])
        latent_actions = self.latent_actions.reshape(-1, self.n_timesteps + 1, self.actions.shape[-1])
        if self.available_actions is not None:
            available_actions = self.available_actions[:-1].reshape(-1, self.available_actions.shape[-1])
        value_preds = self.value_preds[:-1].reshape(-1, self.value_preds.shape[-1])
        returns = self.returns[:-1].reshape(-1, self.returns.shape[-1])
        masks = self.masks[:-1].reshape(-1, 1)
        active_masks = self.active_masks[:-1].reshape(-1, 1)
        action_log_probs = self.action_log_probs.reshape(-1, *self.action_log_probs.shape[3:])
        advantages = advantages.reshape(-1, *advantages.shape[3:])

        for indices in sampler:
            # obs size [T+1 N M Dim]-->[T N M Dim]-->[T*N*M,Dim]-->[index,Dim]
            share_obs_batch = share_obs[indices]
            obs_batch = obs[indices]
            rnn_states_batch = rnn_states[indices]
            rnn_states_critic_batch = rnn_states_critic[indices]
            actions_batch = actions[indices]
            latent_actions_batch = latent_actions[indices]
            if self.available_actions is not None:
                available_actions_batch = available_actions[indices]
            else:
                available_actions_batch = None
            value_preds_batch = value_preds[indices]
            return_batch = returns[indices]
            masks_batch = masks[indices]
            active_masks_batch = active_masks[indices]
            old_action_log_probs_batch = action_log_probs[indices]
            if advantages is None:
                adv_targ = None
            else:
                adv_targ = advantages[indices]

            yield share_obs_batch, obs_batch, rnn_states_batch, rnn_states_critic_batch, actions_batch, latent_actions_batch,\
                  value_preds_batch, return_batch, masks_batch, active_masks_batch, old_action_log_probs_batch,\
                  adv_targ, available_actions_batch

    def naive_recurrent_generator(self, advantages, num_mini_batch):
        """
        Yield training data for non-chunked RNN training.
        :param advantages: (np.ndarray) advantage estimates.
        :param num_mini_batch: (int) number of minibatches to split the batch into.
        """
        episode_length, n_rollout_threads, num_agents = self.rewards.shape[0:3]
        batch_size = n_rollout_threads * num_agents
        assert n_rollout_threads * num_agents >= num_mini_batch, (
            "PPO requires the number of processes ({})* number of agents ({}) "
            "to be greater than or equal to the number of "
            "PPO mini batches ({}).".format(n_rollout_threads, num_agents, num_mini_batch))
        num_envs_per_batch = batch_size // num_mini_batch
        perm = torch.randperm(batch_size).numpy()

        share_obs = self.share_obs.reshape(-1, batch_size, *self.share_obs.shape[3:])
        obs = self.obs.reshape(-1, batch_size, *self.obs.shape[3:])
        rnn_states = self.rnn_states.reshape(-1, batch_size, *self.rnn_states.shape[3:])
        rnn_states_critic = self.rnn_states_critic.reshape(-1, batch_size, *self.rnn_states_critic.shape[3:])
        actions = self.actions.reshape(-1, batch_size, self.actions.shape[-1])
        if self.available_actions is not None:
            available_actions = self.available_actions.reshape(-1, batch_size, self.available_actions.shape[-1])
        value_preds = self.value_preds.reshape(-1, batch_size, 1)
        returns = self.returns.reshape(-1, batch_size, 1)
        masks = self.masks.reshape(-1, batch_size, 1)
        active_masks = self.active_masks.reshape(-1, batch_size, 1)
        action_log_probs = self.action_log_probs.reshape(-1, batch_size, self.action_log_probs.shape[-1])
        advantages = advantages.reshape(-1, batch_size, 1)

        for start_ind in range(0, batch_size, num_envs_per_batch):
            share_obs_batch = []
            obs_batch = []
            rnn_states_batch = []
            rnn_states_critic_batch = []
            actions_batch = []
            available_actions_batch = []
            value_preds_batch = []
            return_batch = []
            masks_batch = []
            active_masks_batch = []
            old_action_log_probs_batch = []
            adv_targ = []

            for offset in range(num_envs_per_batch):
                ind = perm[start_ind + offset]
                share_obs_batch.append(share_obs[:-1, ind])
                obs_batch.append(obs[:-1, ind])
                rnn_states_batch.append(rnn_states[0:1, ind])
                rnn_states_critic_batch.append(rnn_states_critic[0:1, ind])
                actions_batch.append(actions[:, ind])
                if self.available_actions is not None:
                    available_actions_batch.append(available_actions[:-1, ind])
                value_preds_batch.append(value_preds[:-1, ind])
                return_batch.append(returns[:-1, ind])
                masks_batch.append(masks[:-1, ind])
                active_masks_batch.append(active_masks[:-1, ind])
                old_action_log_probs_batch.append(action_log_probs[:, ind])
                adv_targ.append(advantages[:, ind])

            # [N[T, dim]]
            T, N = self.episode_length, num_envs_per_batch
            # These are all from_numpys of size (T, N, -1)
            share_obs_batch = np.stack(share_obs_batch, 1)
            obs_batch = np.stack(obs_batch, 1)
            actions_batch = np.stack(actions_batch, 1)
            if self.available_actions is not None:
                available_actions_batch = np.stack(available_actions_batch, 1)
            value_preds_batch = np.stack(value_preds_batch, 1)
            return_batch = np.stack(return_batch, 1)
            masks_batch = np.stack(masks_batch, 1)
            active_masks_batch = np.stack(active_masks_batch, 1)
            old_action_log_probs_batch = np.stack(old_action_log_probs_batch, 1)
            adv_targ = np.stack(adv_targ, 1)

            # States is just a (N, dim) from_numpy [N[1,dim]]
            rnn_states_batch = np.stack(rnn_states_batch).reshape(N, *self.rnn_states.shape[3:])
            rnn_states_critic_batch = np.stack(rnn_states_critic_batch).reshape(N, *self.rnn_states_critic.shape[3:])

            # Flatten the (T, N, ...) from_numpys to (T * N, ...)
            share_obs_batch = _flatten(T, N, share_obs_batch)
            obs_batch = _flatten(T, N, obs_batch)
            actions_batch = _flatten(T, N, actions_batch)
            if self.available_actions is not None:
                available_actions_batch = _flatten(T, N, available_actions_batch)
            else:
                available_actions_batch = None
            value_preds_batch = _flatten(T, N, value_preds_batch)
            return_batch = _flatten(T, N, return_batch)
            masks_batch = _flatten(T, N, masks_batch)
            active_masks_batch = _flatten(T, N, active_masks_batch)
            old_action_log_probs_batch = _flatten(T, N, old_action_log_probs_batch)
            adv_targ = _flatten(T, N, adv_targ)

            yield share_obs_batch, obs_batch, rnn_states_batch, rnn_states_critic_batch, actions_batch,\
                  value_preds_batch, return_batch, masks_batch, active_masks_batch, old_action_log_probs_batch,\
                  adv_targ, available_actions_batch

    def recurrent_generator(self, advantages, num_mini_batch, data_chunk_length):
        """
        Yield training data for chunked RNN training.
        :param advantages: (np.ndarray) advantage estimates.
        :param num_mini_batch: (int) number of minibatches to split the batch into.
        :param data_chunk_length: (int) length of sequence chunks with which to train RNN.
        """
        data_chunk_length = self.act_step  # self.episode_length * self.n_rollout_threads * self.action_log_probs.shape[2] #self.act_step
        episode_length, n_rollout_threads, num_agents = self.rewards.shape[0:3]
        assert episode_length % self.act_step == 0
        num_agents = self.action_log_probs.shape[2]
        batch_size = n_rollout_threads * episode_length * num_agents
        data_chunks = batch_size // data_chunk_length  # [C=r*T*M/L]
        mini_batch_size = data_chunks // num_mini_batch

        rand = torch.randperm(data_chunks).numpy()
        sampler = [rand[i * mini_batch_size:(i + 1) * mini_batch_size] for i in range(num_mini_batch)]
        # debug_print(batch_size, num_agents, data_chunk_length, data_chunks, num_mini_batch, batch_size)
        
        if len(self.share_obs.shape) > 4:
            share_obs = self.share_obs[:-1].transpose(1, 2, 0, 3, 4, 5).reshape(-1, *self.share_obs.shape[3:])
            if self.use_symmetry:
                obs_sym = self.obs_sym[:-1].transpose(1, 2, 0, 3, 4, 5).reshape(-1, *self.obs_sym.shape[2:])
            # if self.use_attention:
            #     obs = self.obs[:-1].transpose(1, 2, 0, 3, 4, 5).reshape(-1, *self.obs.shape[2:])
            # else:
            obs = self.obs[:-1].transpose(1, 2, 0, 3, 4, 5).reshape(-1, *self.obs.shape[3:])
        else:
            share_obs = _cast(self.share_obs[:-1])
            obs = _cast(self.obs[:-1])
            # obs = _cast(self.obs[:-1].reshape(self.obs.shape[0] - 1, self.obs.shape[1], 1, -1))
            # obs = obs.reshape(obs.shape[0], self.rnum_agents, -1)
            if self.use_symmetry:
                obs_sym = _cast(self.obs_sym[:-1])
        # debug_print('shp', obs.shape)

        actions = _cast(self.actions)
        latent_actions = _cast(self.latent_actions)
        # debug_print('noise', self.noise.shape)
        # debug_print('noises', self.noise.transpose(1, 2, 0, *list(range(3, len(self.noise.shape)))).shape)
        noises = _cast(self.noise)
        # debug_print('noise', noises.shape)
        sampled_actions = _cast(self.sampled_actions)
        # debug_print(sampled_actions)
        # debug_print(self.action_log_probs.shape, self.action_log_probs_last.shape, advantages.shape)
        action_log_probs = _cast(self.action_log_probs)
        action_log_probs_last = _cast(self.action_log_probs_last)
        # debug_print(action_log_probs.shape, action_log_probs_last.shape)
        advantages = _cast(advantages[:, :, None])
        # debug_print(self.value_preds.shape, self.action_log_probs.shape)
        value_preds = _cast(self.value_preds[:-1, :, :, :, None].transpose(0, 1, 4, 3, 2))
        returns = _cast(self.returns[:-1, :, :, :, None].transpose(0, 1, 4, 3, 2))
        # debug_print(returns.shape, self.returns.shape)
        masks = _cast(self.masks[:-1])
        active_masks = _cast(self.active_masks[:-1, :, None])
        # debug_print('buffer', action_log_probs.shape, action_log_probs_last.shape, value_preds.shape, returns.shape)
        # debug_print('gen:', self.active_masks.shape, active_masks.shape, latent_actions.shape, self.latent_actions.shape)
        # rnn_states = _cast(self.rnn_states[:-1])
        # rnn_states_critic = _cast(self.rnn_states_critic[:-1])
        rnn_states = self.rnn_states[:-1].transpose(1, 2, 0, 3, 4).reshape(-1, *self.rnn_states.shape[3:])
        rnn_states_critic = self.rnn_states_critic[:-1].transpose(1, 2, 0, 3, 4).reshape(-1,
                                                                                         *self.rnn_states_critic.shape[
                                                                                          3:])
        # debug_print(advantages.shape, action_log_probs.shape)

        if self.available_actions is not None:
            available_actions = _cast(self.available_actions[:-1])

        for indices in sampler:
            share_obs_batch = []
            obs_batch = []
            obs_sym_batch = []
            rnn_states_batch = []
            rnn_states_critic_batch = []
            actions_batch = []
            latent_actions_batch = []
            noises_batch = []
            sampled_actions_batch = []
            available_actions_batch = []
            value_preds_batch = []
            return_batch = []
            masks_batch = []
            active_masks_batch = []
            old_action_log_probs_batch = []
            action_log_probs_last_batch = []
            adv_targ = []
            
            # debug_print(indices)
            for index in indices:

                ind = index * data_chunk_length
                # debug_print(data_chunk_length)
                # size [T+1 N M Dim]-->[T N M Dim]-->[N,M,T,Dim]-->[N*M*T,Dim]-->[L,Dim]
                share_obs_batch.append(share_obs[ind:ind + data_chunk_length])
                obs_batch.append(obs[ind:ind + data_chunk_length])
                if self.use_symmetry:
                    obs_sym_batch.append(obs_sym[ind:ind + data_chunk_length])
                actions_batch.append(actions[ind:ind + data_chunk_length])
                latent_actions_batch.append(latent_actions[ind:ind+data_chunk_length])
                # debug_print(ind, ind+data_chunk_length)
                noises_batch.append(noises[ind:ind+data_chunk_length])
                # assert latent_actions_batch[-1].shape[-1] % self.rnum_agents == 0
                # latent_actions_batch[-1] = latent_actions_batch[-1].reshape(*latent_actions_batch[-1].shape[:-1], latent_actions_batch[-1].shape[-1]//self.rnum_agents, self.rnum_agents)
                sampled_actions_batch.append(sampled_actions[ind:ind+data_chunk_length])
                # sampled_actions_batch[-1] = sampled_actions_batch[-1].reshape(*sampled_actions_batch[-1].shape[:-1], sampled_actions_batch[-1].shape[-1]//self.rnum_agents, self.rnum_agents)
                if self.available_actions is not None:
                    available_actions_batch.append(available_actions[ind:ind + data_chunk_length])
                value_preds_batch.append(value_preds[ind:ind + data_chunk_length])
                return_batch.append(returns[ind:ind + data_chunk_length])
                masks_batch.append(masks[ind:ind + data_chunk_length])
                active_masks_batch.append(active_masks[ind:ind + data_chunk_length])
                old_action_log_probs_batch.append(action_log_probs[ind:ind + data_chunk_length])
                action_log_probs_last_batch.append(action_log_probs_last[ind:ind + data_chunk_length])
                adv_targ.append(advantages[ind:ind + data_chunk_length])
                # debug_print(value_preds.shape, masks.shape)
                # size [T+1 N M Dim]-->[T N M Dim]-->[N M T Dim]-->[N*M*T,Dim]-->[1,Dim]
                rnn_states_batch.append(rnn_states[ind])
                rnn_states_critic_batch.append(rnn_states_critic[ind])

            L, N = data_chunk_length, mini_batch_size
            # debug_print(data_chunk_length, data_chunks, indices)
            # debug_print(sampled_actions_batch)

            # These are all from_numpys of size (L, N, Dim)  
            # debug_print('share_obs_batch', share_obs_batch[0].shape, 'obs_batch', obs_batch[0].shape, 'actions_batch', actions_batch[0].shape, 'latent_actions_batch', latent_actions_batch[0].shape, 'noises_batch', noises_batch[0].shape, 'sampled_actions_batch', sampled_actions_batch[0].shape)
            share_obs_batch = np.stack(share_obs_batch, axis=0)
            obs_batch = np.stack(obs_batch, axis=0)
            # debug_print(obs_sym_batch[0].shape, len(obs_sym_batch))

            actions_batch = np.stack(actions_batch, axis=0)
            latent_actions_batch = np.stack(latent_actions_batch, axis=0)
            # debug_print('noises_batch', noises_batch[0].shape)
            noises_batch = np.stack(noises_batch, axis=0)
            # debug_print('noises_batch', noises_batch.shape)
            sampled_actions_batch = np.stack(sampled_actions_batch, axis=0)
            if self.available_actions is not None:
                available_actions_batch = np.stack(available_actions_batch, axis=0)
            value_preds_batch = np.stack(value_preds_batch, axis=0)
            return_batch = np.stack(return_batch, axis=0)
            masks_batch = np.stack(masks_batch, axis=0)
            active_masks_batch = np.stack(active_masks_batch, axis=0)
            old_action_log_probs_batch = np.stack(old_action_log_probs_batch, axis=0)
            action_log_probs_last_batch = np.stack(action_log_probs_last_batch, axis=0)
            adv_targ = np.stack(adv_targ, axis=0)

            # States is just a (N, -1) from_numpy
            rnn_states_batch = np.stack(rnn_states_batch).reshape(N, *self.rnn_states.shape[3:])
            rnn_states_critic_batch = np.stack(rnn_states_critic_batch).reshape(N, *self.rnn_states_critic.shape[3:])

            # Flatten the (L, N, ...) from_numpys to (L * N, ...)
            share_obs_batch = _flatten(L, N, share_obs_batch)
            obs_batch = _flatten(L, N, obs_batch)
            # debug_print(obs_sym_batch.shape)
            actions_batch = _flatten(L, N, actions_batch)
            latent_actions_batch = _flatten(L, N, latent_actions_batch)
            noises_batch = _flatten(L, N, noises_batch)
            sampled_actions_batch = _flatten(L, N, sampled_actions_batch)
            if self.available_actions is not None:
                available_actions_batch = _flatten(L, N, available_actions_batch)
            else:
                available_actions_batch = None
            value_preds_batch = _flatten(L, N, value_preds_batch)
            return_batch = _flatten(L, N, return_batch)
            masks_batch = _flatten(L, N, masks_batch)
            active_masks_batch = _flatten(L, N, active_masks_batch)
            old_action_log_probs_batch = _flatten(L, N, old_action_log_probs_batch)
            action_log_probs_last_batch = _flatten(L, N, action_log_probs_last_batch)
            # debug_print(type(old_action_log_probs_batch), type(action_log_probs_last_batch))
            adv_targ = _flatten(L, N, adv_targ)
            # debug_print(sampled_actions_batch)

            if self.use_symmetry:
                obs_sym_batch = np.stack(obs_sym_batch, axis=1)
                obs_sym_batch = _flatten(L, N, obs_sym_batch)
                yield share_obs_batch, obs_batch, obs_sym, rnn_states_batch, rnn_states_critic_batch, actions_batch, latent_actions_batch, sampled_actions_batch,\
                    value_preds_batch, return_batch, masks_batch, active_masks_batch, old_action_log_probs_batch, action_log_probs_last_batch,\
                    adv_targ, available_actions_batch
            else:
                yield share_obs_batch, obs_batch, rnn_states_batch, rnn_states_critic_batch, actions_batch, latent_actions_batch, sampled_actions_batch,\
                    value_preds_batch, return_batch, masks_batch, active_masks_batch, old_action_log_probs_batch, action_log_probs_last_batch,\
                    adv_targ, available_actions_batch, noises_batch