import os
from datetime import datetime

import torch
import torch.nn as nn
import numpy as np
import gpytorch

from data import uci_woval
from toy import x3_gap_toy, sin_toy
from toy import polynomial_toy, polynomial_gap_toy, polynomial_gap_toy2, g2_toy, g3_toy

from ifbde_std_softplus import *

import matplotlib
matplotlib.use('Agg')
from matplotlib import pyplot as plt

from utils.utils import default_plotting_new as init_plotting

from utils.utils import device


################################### Hyper-parameters ###################################

lr_bnn = 0.01
lr_gp0 = 0.1

prior_coeff = 10
random_seed = 127

save_model_freq = 5000
log_model_freq = 3000

test_interval = 1000
num_sample = 1000
epochs = 2001
n_step_prior_pretraining = 100

# add_w2_freq = 128
# add_w2_len = 1


PRIOR_PRETRAIN = True
ADD_IN_W2 = False

################################### Network Architecture ###################################

n_units = 10
n_hidden = 2
hidden_dims = [n_units] * n_hidden
activation_fn = 'tanh'

######################################################################


datasetstring = 'concrete' #'protein' # 'concrete' # yacht # boston kin8nm
bnn_name_string = 'BDE-uci-'+activation_fn+'-'+datasetstring+'-'+str(prior_coeff)+'-'
# bnn_name_string = 'gp-prior-only'


torch.manual_seed(random_seed)
torch.cuda.manual_seed(random_seed)
np.random.seed(random_seed)
# torch.use_deterministic_algorithms(True)
torch.backends.cudnn.deterministic=True
torch.autograd.set_detect_anomaly(True)


print("============================================================================================")
################################## set device ##################################

# # set device to cpu or cuda
# device = torch.device('cpu')
# print((torch.cuda.is_available()))
# if (torch.cuda.is_available()):
#     device = torch.device('cuda:0')
#     torch.cuda.empty_cache()
#     print("Device set to : " + str(torch.cuda.get_device_name(device)))
# else:
#     print("Device set to : cpu")

print("============================================================================================")


################### savefigures ###################

results_folder = "./results"
if not os.path.exists(results_folder):
    os.makedirs(results_folder)

figures_folder = results_folder + '/' + bnn_name_string + '/' + 'sin_tt_0.1_mset' + '/'
if not os.path.exists(figures_folder):
    os.makedirs(figures_folder)



###################### logging ######################
run_num = 0
#### create new log file for each run
log_f_name = figures_folder + bnn_name_string + '_sin_tt_0.1_mset' + "_" + str(run_num) + ".csv"

print("current logging run number for " + bnn_name_string + " : ", run_num)
print("logging at : " + log_f_name)


print("============================================================================================")
################### checkpointing ###################

run_num_pretrained = 0  #### change this to prevent overwriting weights in same env_name folder

checkpoint_path = figures_folder + "{}_{}_{}.pth".format(bnn_name_string, random_seed, run_num_pretrained)
print("save checkpoint path : " + checkpoint_path)

print("============================================================================================")
############# print all hyperparameters #############

print("learning rate: ", lr_bnn)
print("coefficient of prior regularization: ", prior_coeff)
print("random seed: ", random_seed)


print("============================================================================================")
############################## load and normalize data ##############################


train_x, test_x, train_y, test_y = uci_woval(datasetstring, seed=random_seed)
# print(test_y)

training_num, input_dim = train_x.shape
test_num = test_x.shape[0]
output_dim = 1
is_continuous = True

print("training_num = ", training_num, " input_dim = ", input_dim, " output_dim = ", output_dim)

original_x_train = torch.from_numpy(train_x).float().to(device)
original_y_train = torch.from_numpy(train_y).float().to(device)
original_x_test = torch.from_numpy(test_x).float().to(device)
original_y_test = torch.from_numpy(test_y).float().to(device)

original_x_train_test = torch.concat((original_x_train, original_x_test))
original_y_train_test = torch.concat((original_y_train, original_y_test))

num_train_test = original_y_train_test.shape[0]

print("num_train_test = ", num_train_test)


print("============================================================================================")
################################## define model ##################################

bnn = ifBDE(input_dim, output_dim, hidden_dims, activation_fn, is_continuous, num_ensemble=5).to(device)
# bnn_optimizer = torch.optim.Adam([{'params': bnn.parameters(), 'lr': lr_bnn}])
bnn_optimizer = torch.optim.Adam([{'params': bnn.parameters(), 'lr': lr_bnn}])


# loss = nn.MSELoss(reduction='mean')

print("============================================================================================")
################################## prior pre-training ##################################

# TODO: pre-train BNN prior
#

print("============================================================================================")
################################## start training ##################################

# track total training time
start_time = datetime.now().replace(microsecond=0)
print("Started training at (GMT) : ", start_time)

# logging file
log_f = open(log_f_name, "w+")
log_f.write('epoch,training_loss\n')
# scheduler = torch.optim.lr_scheduler.StepLR(bnn_optimizer, 5000, gamma=0.9, last_epoch=-1)

loss = nn.MSELoss(reduction='mean')

w2_all = []
likelihood_loss = []
all_loss = []

weight_dis_normal = torch.distributions.normal.Normal(torch.tensor([0.0]).to(device), torch.tensor([1.0]).to(device))
# weight_dis_normal.to(device)

for epoch in range(epochs):

    x_num = 100
    idx = torch.randperm(training_num).to(device)
    original_x_train_batch = original_x_train[idx[:x_num], :]
    original_y_train_batch = original_y_train[idx[:x_num]]

    original_x_train_tmp = torch.zeros(bnn.num_ensemble, x_num, input_dim).to(device)
    original_y_train_tmp = torch.zeros(bnn.num_ensemble, x_num).to(device)

    for i in range(bnn.num_ensemble):
        # idx = torch.randperm(training_num)
        original_x_train_tmp[i, :, :] = original_x_train_batch
        original_y_train_tmp[i, :] = original_y_train_batch

    # 3. obtain likelihood gradients
    bnn.train()

    train_loss = 0
    likehd_loss = 0

    pred_y = bnn.forward_de(original_x_train_tmp)

    for i in range(bnn.num_ensemble):
        likehd_loss = likehd_loss + loss(pred_y[i, :].squeeze().view(-1), original_y_train_tmp[i, :])
        likehd_loss = likehd_loss - prior_coeff * (torch.mean(weight_dis_normal.log_prob(bnn.get_ensemble_weights(i))))

    likehd_loss = likehd_loss / bnn.num_ensemble
    likelihood_loss.append(likehd_loss.item())
    train_loss = train_loss + likehd_loss

    wdist_objective = 0
    w2_all.append(wdist_objective)

    # train_loss = train_loss + 0 * wdist_objective
    all_loss.append(train_loss.item())

    # print("epoch : {} \t mean_term \t : {} \t trace_term \t : {} \t hard_trace \t : {} \t".format(
    #     epoch, mean_term, trace_term, hard_trace),
    #     datetime.now().replace(microsecond=0) - start_time)

    print("epoch : {} \t training loss \t : {} \t likelihood loss \t : {} \t wdist loss \t : {} \t".format(
        epoch, all_loss[-1], likelihood_loss[-1], w2_all[-1]),
        datetime.now().replace(microsecond=0) - start_time)

    # print(torch.cuda.memory_summary())

    # 5. optimisation
    bnn_optimizer.zero_grad()
    train_loss.backward()  # obtain p.grad
    bnn_optimizer.step()

    # 8. log in logging file
    if epoch % log_model_freq == 0:
        print("--------------------------------------------------------------------------------------------")
        log_f.write('{},{}\n'.format(epoch, train_loss))
        log_f.flush()
        print("log saved")
        print("--------------------------------------------------------------------------------------------")

    # 9. save model weights
    if epoch % save_model_freq == 0:
        print("--------------------------------------------------------------------------------------------")
        print("saving model at : " + checkpoint_path)
        bnn.save(checkpoint_path)
        print("model saved")
        print("--------------------------------------------------------------------------------------------")

    # 10. test bnn
    if epoch >= (epochs-1) or epoch % test_interval == 0:

        bnn.eval()

        # mean_y_pred, std_y_pred, _ = bnn.forward_wb_gaus(original_x_test)
        # mean_y_pred = mean_y_pred.detach().squeeze().view(-1)
        # std_y_pred = std_y_pred.detach().squeeze().view(-1)

        samples_pred_test_y = bnn.forward_eval(original_x_test, num_sample).squeeze().detach()
        # mean_y_pred = pred_test_y
        mean_y_pred = torch.mean(samples_pred_test_y, 0)
        std_y_pred = torch.std(samples_pred_test_y, 0)

        y_pred = torch.mean(samples_pred_test_y, 0)

        rmse_test_loss = torch.sqrt(loss(y_pred, original_y_test))
        print('rmse_test_loss: ', rmse_test_loss)

        variance_y_pred = std_y_pred ** 2
        log_std = torch.log(std_y_pred + 1e-4).sum()
        vec = 0.5 * (original_y_test - mean_y_pred) ** 2 / variance_y_pred
        vec = vec.sum()
        const = 0.5 * np.log(np.pi * 2)
        nll = test_num * const + log_std + vec
        nll_mean = nll / test_num


        print('nll_mean: ', nll_mean)


log_f.close()

print("============================================================================================")

# final test
bnn.eval()

# mean_y_pred, std_y_pred, _ = bnn.forward_wb_gaus(original_x_test)
# mean_y_pred = mean_y_pred.detach().squeeze().view(-1)
# std_y_pred = std_y_pred.detach().squeeze().view(-1)

samples_pred_test_y = bnn.forward_eval(original_x_test, num_sample).squeeze().detach()
# mean_y_pred = pred_test_y
mean_y_pred = torch.mean(samples_pred_test_y, 0)
std_y_pred = torch.std(samples_pred_test_y, 0)

y_pred = torch.mean(samples_pred_test_y, 0)

rmse_test_loss = torch.sqrt(loss(y_pred, original_y_test))
print('rmse_test_loss: ', rmse_test_loss)

variance_y_pred = std_y_pred ** 2
log_std = torch.log(std_y_pred + 1e-4).sum()
vec = 0.5 * (original_y_test - mean_y_pred) ** 2 / variance_y_pred
vec = vec.sum()
const = 0.5 * np.log(np.pi * 2)
nll = test_num * const + log_std + vec
nll_mean = nll / test_num

print('nll_mean: ', nll_mean)


#
plt.figure()
indices = np.arange(epochs)
loss_all = np.array(all_loss)
plt.plot(indices, loss_all[indices], '-ko', ms=3)
plt.ylabel(r'All loss')
plt.tight_layout()
plt.xlabel('Iteration')
plt.tight_layout()
plt.savefig(figures_folder + '/all_loss.pdf')

plt.figure()
w2_all = np.array(w2_all)
plt.plot(w2_all, 'r-')
plt.title('w2_all')
plt.savefig(figures_folder + '/w2_all.pdf')

plt.figure()
likelihood_loss = np.array(likelihood_loss)
plt.plot(likelihood_loss, 'r-')
plt.title('likelihood_loss')
plt.savefig(figures_folder + '/likelihood_loss.pdf')

# print total training time
print("============================================================================================")
end_time = datetime.now().replace(microsecond=0)
print("Started training at (GMT) : ", start_time)
print("Finished training at (GMT) : ", end_time)
print("Total training time  : ", end_time - start_time)
print("============================================================================================")

