import os
from datetime import datetime

import torch
import torch.nn as nn
import numpy as np
import gpytorch

from data import uci_woval
from toy import x3_gap_toy, sin_toy
from toy import polynomial_toy, polynomial_gap_toy, polynomial_gap_toy2, g2_toy, g3_toy

from ifbnn_std_softplus_exactgrad import *

import matplotlib
matplotlib.use('Agg')
from matplotlib import pyplot as plt

from torch.func import functional_call, vmap, grad

from utils.utils import default_plotting_new as init_plotting

from utils.utils import device

# set device to cpu or cuda
device = torch.device('cpu')
if (torch.cuda.is_available()):
    device = torch.device('cuda:0')
    torch.cuda.empty_cache()
    print("Device set to : " + str(torch.cuda.get_device_name(device)))
else:
    print("Device set to : cpu")

################################### Hyper-parameters ###################################

lr_bnn = 0.01
lr_gp0 = 0.1 #1e-3

prior_coeff = 10
random_seed = 127

save_model_freq = 5000
log_model_freq = 3000

test_interval = 1000
num_sample = 1000
epochs = 2001

n_step_prior_pretraining = 100 #3000

PRIOR_PRETRAIN = True
ADD_W2 = True
# num_xmset = 40

################################### Network Architecture ###################################

n_units = 10
n_hidden = 2
hidden_dims = [n_units] * n_hidden
activation_fn = 'tanh'

######################################################################


datasetstring = 'kin8nm' #'protein' # 'concrete' #
bnn_name_string = 'FSVI-uci-'+activation_fn+'-'+datasetstring+'-'+str(prior_coeff)+'-'


torch.manual_seed(random_seed)
torch.cuda.manual_seed(random_seed)
np.random.seed(random_seed)
# torch.use_deterministic_algorithms(True)
torch.backends.cudnn.deterministic=True
torch.autograd.set_detect_anomaly(True)

print("============================================================================================")


################### savefigures ###################

results_folder = "./results"
if not os.path.exists(results_folder):
    os.makedirs(results_folder)

figures_folder = results_folder + '/' + bnn_name_string + '/' + 'sin_tt_0.1_mset' + '/'
if not os.path.exists(figures_folder):
    os.makedirs(figures_folder)



###################### logging ######################
run_num = 0
#### create new log file for each run
log_f_name = figures_folder + bnn_name_string + '_sin_tt_0.1_mset' + "_" + str(run_num) + ".csv"

print("current logging run number for " + bnn_name_string + " : ", run_num)
print("logging at : " + log_f_name)


print("============================================================================================")
################### checkpointing ###################

run_num_pretrained = 0  #### change this to prevent overwriting weights in same env_name folder

checkpoint_path = figures_folder + "{}_{}_{}.pth".format(bnn_name_string, random_seed, run_num_pretrained)
print("save checkpoint path : " + checkpoint_path)

print("============================================================================================")
############# print all hyperparameters #############

print("learning rate: ", lr_bnn)
print("coefficient of prior regularization: ", prior_coeff)
print("random seed: ", random_seed)


print("============================================================================================")
############################## load and normalize data ##############################


train_x, test_x, train_y, test_y = uci_woval(datasetstring, seed=random_seed)
# print(test_y)

training_num, input_dim = train_x.shape
test_num = test_x.shape[0]
output_dim = 1
is_continuous = True

print("training_num = ", training_num, " input_dim = ", input_dim, " output_dim = ", output_dim)

original_x_train = torch.from_numpy(train_x).float().to(device)
original_y_train = torch.from_numpy(train_y).float().to(device)
original_x_test = torch.from_numpy(test_x).float().to(device)
original_y_test = torch.from_numpy(test_y).float().to(device)

original_x_train_test = torch.concat((original_x_train, original_x_test))
original_y_train_test = torch.concat((original_y_train, original_y_test))

num_train_test = original_y_train_test.shape[0]

print("num_train_test = ", num_train_test)


print("============================================================================================")
################################## define model ##################################

bnn = ifBNN(input_dim, output_dim, hidden_dims, activation_fn, is_continuous, scaled_variance=True).to(device)
# bnn_optimizer = torch.optim.Adam([{'params': bnn.parameters(), 'lr': lr_bnn}])
bnn_optimizer = torch.optim.Adam([{'params': bnn.parameters(), 'lr': lr_bnn}])

w_size = len(bnn.get_variance().squeeze())

# loss = nn.MSELoss(reduction='mean')

print("============================================================================================")
################################## prior pre-training ##################################

# TODO: pre-train BNN prior

# prior = ifBNN(input_dim, output_dim, hidden_dims, activation_fn, is_continuous, scaled_variance=True).to(device)
# prior_optimizer = torch.optim.Adam([{'params': prior.parameters(), 'lr': lr_bnn}])
# prior.eval()

if PRIOR_PRETRAIN:

    # pre-train GP prior
    # data_x_for_prior = original_x_train
    # data_y_for_prior = original_y_train

    data_x_for_prior = original_x_test
    data_y_for_prior = original_y_test

    # idx = torch.randperm(num_train_test)[:10000]
    # data_x_for_prior = original_x_train_test[idx, :]
    # data_y_for_prior = original_y_train_test[idx]

    likelihood = gpytorch.likelihoods.GaussianLikelihood()
    # prior = ExactGPModel(original_x_train, original_y_train, likelihood, input_dim).to(device)
    # prior = ExactGPModel(original_x_test, original_y_test, likelihood, input_dim).to(device)
    prior = ExactGPModel(data_x_for_prior, data_y_for_prior, likelihood, input_dim).to(device)

    # idx = torch.randperm(num_train_test)[:10000]
    # prior = ExactGPModel(original_x_train_test[idx], original_y_train_test[idx], likelihood, input_dim).to(device)

    prior.train()
    likelihood.train()

    # Use the adam optimizer
    gp_optimizer = torch.optim.Adam(prior.parameters(), lr=lr_gp0)  # Includes GaussianLikelihood parameters

    # "Loss" for GPs - the marginal log likelihood
    mll = gpytorch.mlls.ExactMarginalLogLikelihood(likelihood, prior)

    for i in range(n_step_prior_pretraining):
        # Zero gradients from previous iteration
        gp_optimizer.zero_grad()
        # Output from model
        output = prior(data_x_for_prior)
        # idx = torch.randperm(num_train_test)[:1000]
        # output = prior(original_x_train_test[idx, :])
        # Calc loss and backprop gradients
        loss_gp = -mll(output, data_y_for_prior)

        # gp_optimizer.zero_grad()
        loss_gp.backward()
        print('Iter %d/%d - Loss: %.3f     noise: %.3f' % (
            i + 1, n_step_prior_pretraining, loss_gp.item(),
            prior.likelihood.noise.item()
        ))
        gp_optimizer.step()

    prior.eval()
    likelihood.eval()
    sigma = prior.likelihood.noise.item()

    # del data_x_for_prior
    # del data_y_for_prior
    del output
    del gp_optimizer


print("============================================================================================")
################################## start training ##################################

# track total training time
start_time = datetime.now().replace(microsecond=0)
print("Started training at (GMT) : ", start_time)

# logging file
log_f = open(log_f_name, "w+")
log_f.write('epoch,training_loss\n')
# scheduler = torch.optim.lr_scheduler.StepLR(bnn_optimizer, 5000, gamma=0.9, last_epoch=-1)

loss = nn.MSELoss(reduction='mean')



print("============================================================================================")

# first test before training
bnn.eval()

# mean_y_pred, std_y_pred, _ = bnn.forward_wb_gaus(original_x_test)
# mean_y_pred = mean_y_pred.detach().squeeze().view(-1)
# std_y_pred = std_y_pred.detach().squeeze().view(-1)

samples_pred_test_y = bnn.forward_eval(original_x_test, num_sample).squeeze().detach()
# mean_y_pred = pred_test_y
mean_y_pred = torch.mean(samples_pred_test_y, 0)
std_y_pred = torch.std(samples_pred_test_y, 0)

y_pred = torch.mean(samples_pred_test_y, 0)

rmse_test_loss = torch.sqrt(loss(y_pred, original_y_test))
print('rmse_test_loss: ', rmse_test_loss)

variance_y_pred = torch.std(samples_pred_test_y, 0) ** 2 + sigma
log_std = torch.log(std_y_pred + 1e-4).sum()
vec = 0.5 * (original_y_test - mean_y_pred) ** 2 / variance_y_pred
vec = vec.sum()
const = 0.5 * np.log(np.pi * 2) + np.log(sigma + 1e-4)
nll = test_num * const + log_std + vec
nll_mean = nll / test_num

print('nll_mean: ', nll_mean)


print("============================================================================================")


kl_all = []
likelihood_loss = []
all_loss = []


def compute_loss(params, buffers, sample):
    samples = sample.unsqueeze(0)

    predictions = functional_call(bnn, (params, buffers), (samples,))
    loss = predictions.squeeze()
    return loss

for epoch in range(epochs):

    # bnn.eval()
    x_mset_num = 40

    # 1. sample measurement set
    x_mset_random = sample_measurement_set(original_x_train_test, num_data=x_mset_num) # original_x_train_add

    x_num = 100
    idx = torch.randperm(training_num)
    original_x_train_batch = original_x_train[idx[:x_num], :]
    original_y_train_batch = original_y_train[idx[:x_num]]

    # 2. measurement set include training data and random data
    x_mset = torch.cat((original_x_train_batch, x_mset_random), 0)
    num_xmset = x_mset.shape[0]

    # calcultate gradient before update
    grads_x_mset = torch.zeros(num_xmset, w_size).to(device)

    # i = 0
    # for x_i in x_mset:
    #     pred_y_i = bnn.forward(x_i)
    #     pred_y_i = pred_y_i.squeeze().flatten()
    #
    #     # evaluate gradient
    #     bnn_optimizer.zero_grad()
    #     pred_y_i.backward()
    #
    #     # output gradient
    #     grad_i = torch.cat([p.grad.view(-1) for name, p in bnn.named_parameters() if "mu" in name])
    #     grad_i = grad_i[:, None]
    #
    #     grads_x_mset[i, :] = torch.transpose(grad_i, 0, 1)

    #
    params = {k: v.detach() for k, v in bnn.named_parameters() if "_mu" in k}
    buffers = {k: v.detach() for k, v in bnn.named_buffers()}

    ft_compute_grad = grad(compute_loss)
    ft_compute_sample_grad = vmap(ft_compute_grad, in_dims=(None, None, 0))
    ft_per_sample_grads = ft_compute_sample_grad(params, buffers, x_mset)

    i = 0
    pre_idx = 0
    for key in ft_per_sample_grads:
        ft_grad = ft_per_sample_grads[key].reshape(num_xmset, -1)
        idx = ft_grad.shape[1]
        grads_x_mset[:, pre_idx:(pre_idx + idx)] = ft_grad
        pre_idx = pre_idx + idx

    # print(' sum_ft_grad = ', grads_x_mset.sum())

    # 3. obtain likelihood gradients
    bnn.train()

    train_loss = 0
    likehd_loss = 0

    # pred_y, Wb = bnn.forward_wb(original_x_train_batch)
    pred_y = bnn.forward_normal(original_x_train_batch)
    pred_y = pred_y.squeeze().view(-1)
    likehd_loss = loss(pred_y, original_y_train_batch)
    likelihood_loss.append(likehd_loss.item())
    train_loss = train_loss + likehd_loss

    # 4. evaluate w2
    if ADD_W2:
        eps = 1e-5

        # # obtain prior
        prior_marginal = likelihood(prior(x_mset))
        mean_prior = prior_marginal.mean.unsqueeze(1)
        # mean_prior = mean_prior.squeeze()
        K_prior = prior_marginal.covariance_matrix

        # # obtain gradients
        pred_y_mset_mean = bnn(x_mset)

        variances = bnn.get_variance().squeeze()
        grad_mset_var = grads_x_mset * variances

        #
        mean_posterior = pred_y_mset_mean
        K_posterior = torch.matmul(grad_mset_var, grads_x_mset.t())
        K_posterior = (K_posterior + K_posterior.t()) / 2

        # KL divergence
        inv_K_prior = torch.pinverse(K_prior)

        diff = mean_posterior - mean_prior
        mean_term = diff.t() @ inv_K_prior @ diff
        trace_term1 = torch.trace(inv_K_prior @ K_posterior)
        det_prior = torch.logdet(K_prior)
        det_posterior = torch.logdet(K_posterior + torch.eye(num_xmset).to(device))
        det_term = det_posterior - det_prior

        kldist_objective = 0.5 * (mean_term + trace_term1 - det_term - num_xmset)

        # prior_dis = MultivariateNormal(mean_prior.squeeze(), K_prior + torch.eye(num_xmset).to(device))
        # posterior_dis = MultivariateNormal(mean_posterior.squeeze(), K_posterior + torch.eye(num_xmset).to(device))
        # kldist_objective = torch.distributions.kl.kl_divergence(prior_dis,posterior_dis)

        kl_all.append(kldist_objective.item())
    else:
        kldist_objective = 0
        kl_all.append(kldist_objective)

    if torch.isnan(kldist_objective) or torch.isinf(kldist_objective):
        continue

    train_loss = train_loss + prior_coeff * kldist_objective
    all_loss.append(train_loss.item())

    # print("epoch : {} \t mean_term \t : {} \t trace_term \t : {} \t hard_trace \t : {} \t".format(
    #     epoch, mean_term, trace_term, hard_trace),
    #     datetime.now().replace(microsecond=0) - start_time)

    print("epoch : {} \t training loss \t : {} \t likelihood loss \t : {} \t kldist loss \t : {} \t".format(
        epoch, all_loss[-1], likelihood_loss[-1], kl_all[-1]),
        datetime.now().replace(microsecond=0) - start_time)

    # print(torch.cuda.memory_summary())

    # 5. optimisation
    bnn_optimizer.zero_grad()
    train_loss.backward()  # obtain p.grad
    bnn_optimizer.step()

    for name, para in bnn.named_parameters():
        if "_std" in name:
            # para = para.clip(-0.5, 0.5)
            para.data = para.data.clamp(min=-0.5, max=0.5)

    # 8. log in logging file
    if epoch % log_model_freq == 0:
        print("--------------------------------------------------------------------------------------------")
        log_f.write('{},{}\n'.format(epoch, train_loss))
        log_f.flush()
        print("log saved")
        print("--------------------------------------------------------------------------------------------")

    # 9. save model weights
    if epoch % save_model_freq == 0:
        print("--------------------------------------------------------------------------------------------")
        print("saving model at : " + checkpoint_path)
        bnn.save(checkpoint_path)
        print("model saved")
        print("--------------------------------------------------------------------------------------------")

    # 10. test bnn
    if epoch >= 0 and epoch % test_interval == 0:

        bnn.eval()

        # mean_y_pred, std_y_pred, _ = bnn.forward_wb_gaus(original_x_test)
        # mean_y_pred = mean_y_pred.detach().squeeze().view(-1)
        # std_y_pred = std_y_pred.detach().squeeze().view(-1)

        samples_pred_test_y = bnn.forward_eval(original_x_test, num_sample).squeeze().detach()
        # mean_y_pred = pred_test_y
        mean_y_pred = torch.mean(samples_pred_test_y, 0)
        std_y_pred = torch.std(samples_pred_test_y, 0)

        y_pred = torch.mean(samples_pred_test_y, 0)

        rmse_test_loss = torch.sqrt(loss(y_pred, original_y_test))
        print('rmse_test_loss: ', rmse_test_loss)

        variance_y_pred = torch.std(samples_pred_test_y, 0) ** 2 + sigma
        log_std = torch.log(std_y_pred + 1e-4).sum()
        vec = 0.5 * (original_y_test - mean_y_pred) ** 2 / variance_y_pred
        vec = vec.sum()
        const = 0.5 * np.log(np.pi * 2) + np.log(sigma + 1e-4)
        nll = test_num * const + log_std + vec
        nll_mean = nll / test_num

        print('nll_mean: ', nll_mean)



log_f.close()
print("============================================================================================")


# final test
bnn.eval()

# mean_y_pred, std_y_pred, _ = bnn.forward_wb_gaus(original_x_test)
# mean_y_pred = mean_y_pred.detach().squeeze().view(-1)
# std_y_pred = std_y_pred.detach().squeeze().view(-1)

samples_pred_test_y = bnn.forward_eval(original_x_test, num_sample).squeeze().detach()
# mean_y_pred = pred_test_y
mean_y_pred = torch.mean(samples_pred_test_y, 0)
std_y_pred = torch.std(samples_pred_test_y, 0)

y_pred = torch.mean(samples_pred_test_y, 0)

rmse_test_loss = torch.sqrt(loss(y_pred, original_y_test))
print('rmse_test_loss: ', rmse_test_loss)

variance_y_pred = torch.std(samples_pred_test_y, 0) ** 2 + sigma
log_std = torch.log(std_y_pred + 1e-4).sum()
vec = 0.5 * (original_y_test - mean_y_pred) ** 2 / variance_y_pred
vec = vec.sum()
const = 0.5 * np.log(np.pi * 2) + np.log(sigma + 1e-4)
nll = test_num * const + log_std + vec
nll_mean = nll / test_num

print('nll_mean: ', nll_mean)

#
plt.figure()
indices = np.arange(len(all_loss))
loss_all = np.array(all_loss)
plt.plot(indices, loss_all[indices], '-ko', ms=3)
plt.ylabel(r'All loss')
plt.tight_layout()
plt.xlabel('Iteration')
plt.tight_layout()
plt.savefig(figures_folder + '/all_loss.pdf')

plt.figure()
kl_all = np.array(kl_all)
plt.plot(kl_all, 'r-')
plt.title('kl_all')
plt.savefig(figures_folder + '/kl_all.pdf')

plt.figure()
likelihood_loss = np.array(likelihood_loss)
plt.plot(likelihood_loss, 'r-')
plt.title('likelihood_loss')
plt.savefig(figures_folder + '/likelihood_loss.pdf')

# print total training time
print("============================================================================================")
end_time = datetime.now().replace(microsecond=0)
print("Started training at (GMT) : ", start_time)
print("Finished training at (GMT) : ", end_time)
print("Total training time  : ", end_time - start_time)
print("============================================================================================")

