import torch
from torch import nn
from einops import rearrange, repeat
from einops.layers.torch import Rearrange

# helpers

def pair(t):
    return t if isinstance(t, tuple) else (t, t)

# classes

class AttentionOnly(nn.Module):
    def __init__(self, dim, heads=8, dim_head=64, dropout=0.):
        super().__init__()
        inner_dim = dim_head * heads
        project_out = not (heads == 1 and dim_head == dim)

        self.heads = heads
        self.scale = dim_head ** -0.5

        self.attend = nn.Softmax(dim=-1)
        self.to_qkv = nn.Linear(dim, inner_dim * 3, bias=False)

        self.to_out = nn.Sequential(
            nn.Linear(inner_dim, dim),
            nn.Dropout(dropout)
        ) if project_out else nn.Identity()

    def forward(self, x):
        qkv = self.to_qkv(x).chunk(3, dim=-1)
        q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> b h n d', h=self.heads), qkv)

        dots = torch.matmul(q, k.transpose(-1, -2)) * self.scale
        attn = self.attend(dots)

        out = torch.matmul(attn, v)
        out = rearrange(out, 'b h n d -> b n (h d)')
        return self.to_out(out)

class AttentionOnlyTransformer(nn.Module):
    def __init__(self, dim, depth, heads, dim_head, dropout=0.):
        super().__init__()
        self.layers = nn.ModuleList([])
        self.depth = depth
        self.heads = heads
        for _ in range(depth):
            self.layers.append(
                AttentionOnly(dim, heads=heads, dim_head=dim_head, dropout=dropout)
            )

    def forward(self, x, return_intermediates=False):
        intermediate_activations = []
        for attn in self.layers:
            x = attn(x)
            if return_intermediates:
                intermediate_activations.append(x.clone())  # Store intermediate activations

        if return_intermediates:
            return x, intermediate_activations
        return x

class AttentionOnlyViT(nn.Module):
    def __init__(self, *, image_size, patch_size, num_classes, dim, depth, heads, dim_head=512, channels=3, dropout=0., emb_dropout=0.):
        super().__init__()
        image_height, image_width = pair(image_size)
        patch_height, patch_width = pair(patch_size)

        assert image_height % patch_height == 0 and image_width % patch_width == 0, 'Image dimensions must be divisible by the patch size.'

        num_patches = (image_height // patch_height) * (image_width // patch_width)
        patch_dim = channels * patch_height * patch_width

        self.to_patch_embedding = nn.Sequential(
            Rearrange('b c (h p1) (w p2) -> b (h w) (p1 p2 c)', p1=patch_height, p2=patch_width),
            nn.Linear(patch_dim, dim),
        )

        self.pos_embedding = nn.Parameter(torch.randn(1, num_patches, dim))  # Positional embedding without cls_token
        self.dropout = nn.Dropout(emb_dropout)

        # Only attention-based transformer without feedforward layers
        self.transformer = AttentionOnlyTransformer(dim, depth, heads, dim_head, dropout)

        # Instead of a pooling layer or classification token, we just use the output
        self.mlp_head = nn.Sequential(
            nn.LayerNorm(dim),
            nn.Linear(dim, num_classes)
        )

    def forward(self, img, return_intermediates=False):
        x = self.to_patch_embedding(img)
        b, n, _ = x.shape

        x += self.pos_embedding[:, :n]
        x = self.dropout(x)

        if return_intermediates:
            x, intermediate_activations = self.transformer(x, return_intermediates=True)
        else:
            x = self.transformer(x)

        x = x.mean(dim=1)

        x = self.mlp_head(x)
        if return_intermediates:
            intermediate_activations.append(x)
            return x, intermediate_activations
        return x

