from __future__ import print_function

import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import torch.backends.cudnn as cudnn
import numpy as np

import torchvision
import torchvision.transforms as transforms

import os
import argparse
import pandas as pd
import csv
import time

from models import *
from utils import progress_bar
from randomaug import RandAugment
from models.vit import ViT
from models.convmixer import ConvMixer

import matplotlib.pyplot as plt
from ripser import ripser
from persim import plot_diagrams
import umap
import warnings
import skdim
from gudhi import bottleneck_distance
from gudhi.hera import wasserstein_distance
from persim import wasserstein

warnings.simplefilter("ignore", UserWarning)

# parsers
parser = argparse.ArgumentParser(description='PyTorch CIFAR10 Training')
parser.add_argument('--lr', default=1e-4, type=float, help='learning rate') # resnets.. 1e-3, Vit..1e-4
parser.add_argument('--opt', default="adam")
parser.add_argument('--resume', '-r', action='store_true', help='resume from checkpoint')
parser.add_argument('--noaug', action='store_false', help='disable use randomaug')
parser.add_argument('--noamp', action='store_true', help='disable mixed precision training. for older pytorch versions')
parser.add_argument('--nowandb', action='store_true', help='disable wandb')
parser.add_argument('--mixup', action='store_true', help='add mixup augumentations')
parser.add_argument('--net', default='vit')
parser.add_argument('--dp', action='store_true', help='use data parallel')
parser.add_argument('--bs', default='512')
parser.add_argument('--size', default="32")
parser.add_argument('--n_epochs', type=int, default='200')
parser.add_argument('--patch', default='4', type=int, help="patch for ViT")
parser.add_argument('--dimhead', default="512", type=int)
parser.add_argument('--convkernel', default='8', type=int, help="parameter for convmixer")
parser.add_argument('--cla', default='2', type=int, help="class of data")
parser.add_argument('--head', default="8", type=int)
parser.add_argument('--depth', default="10", type=int)

args = parser.parse_args()

device_type = 'cuda' if torch.cuda.is_available() else 'cpu'

# take in args
usewandb = not args.nowandb
if usewandb:
    import wandb
    watermark = "{}_lr{}_{}epochs".format(args.net, args.lr, args.n_epochs)
    wandb.init(project="cifar10-challange",
            name=watermark)
    wandb.config.update(args)

bs = int(args.bs)
imsize = int(args.size)

use_amp = not args.noamp
aug = args.noaug

device = 'cuda' if torch.cuda.is_available() else 'cpu'
best_acc = 0  # best test accuracy
start_epoch = 0  # start from epoch 0 or last checkpoint epoch

# Data
print('==> Preparing data..')
if args.net=="vit_timm":
    size = 384
else:
    size = imsize

transform_train = transforms.Compose([
    transforms.RandomCrop(32, padding=4),
    transforms.Resize(size),
    transforms.RandomHorizontalFlip(),
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
])

transform_test = transforms.Compose([
    transforms.Resize(size),
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
])

# Add RandAugment with N, M(hyperparameter)
if aug:  
    N = 2; M = 14;
    transform_train.transforms.insert(0, RandAugment(N, M))

# Prepare dataset
trainset = torchvision.datasets.CIFAR10(root='./data', train=True, download=True, transform=transform_train)
trainloader = torch.utils.data.DataLoader(trainset, batch_size=bs, shuffle=True, num_workers=8)

testset = torchvision.datasets.CIFAR10(root='./data', train=False, download=True, transform=transform_test)
testloader = torch.utils.data.DataLoader(testset, batch_size=100, shuffle=False, num_workers=8)

classes = ('plane', 'car', 'bird', 'cat', 'deer', 'dog', 'frog', 'horse', 'ship', 'truck')

# Model factory..
print('==> Building model..')
# net = VGG('VGG19')
if args.net=='res18':
    net = ResNet18()
elif args.net=='vgg':
    net = VGG('VGG19')
elif args.net=='convbase':
    net = ConvOnlyNetLayersBase()
elif args.net=='convsmall':
    net = ConvOnlyNetLayersSmall()
elif args.net=="attn":
    from models.attn import AttentionOnlyViT
    net = AttentionOnlyViT(
    image_size = size,
    patch_size = args.patch,
    num_classes = 10,
    dim = int(args.dimhead),
    depth = 10,
    heads = args.head,
    dim_head = int(args.dimhead),
    dropout = 0.1,
    emb_dropout = 0.1
)
elif args.net=='res34':
    net = ResNet34()
elif args.net=='res50':
    net = ResNet50()
elif args.net=='res101':
    net = ResNet101()
elif args.net=="convmixer":
    # from paper, accuracy >96%. you can tune the depth and dim to scale accuracy and speed.
    net = ConvMixer(256, 16, kernel_size=args.convkernel, patch_size=1, n_classes=10)
elif args.net=="mlpmixer":
    from models.mlpmixer import MLPMixer
    net = MLPMixer(
    image_size = 32,
    channels = 3,
    patch_size = args.patch,
    dim = 512,
    depth = 6,
    num_classes = 10
)
elif args.net=="vit_small":
    from models.vit_small import ViT
    net = ViT(
    image_size = size,
    patch_size = args.patch,
    num_classes = 10,
    dim = int(args.dimhead),
    depth = args.depth,
    heads = 8,
    mlp_dim = 512,
    dropout = 0.1,
    emb_dropout = 0.1
)
elif args.net=="vit_tiny":
    from models.vit_small import ViT
    net = ViT(
    image_size = size,
    patch_size = args.patch,
    num_classes = 10,
    dim = int(args.dimhead),
    depth = 4,
    heads = 6,
    mlp_dim = 256,
    dropout = 0.1,
    emb_dropout = 0.1
)
elif args.net=="simplevit":
    from models.simplevit import SimpleViT
    net = SimpleViT(
    image_size = size,
    patch_size = args.patch,
    num_classes = 10,
    dim = int(args.dimhead),
    depth = 6,
    heads = 8,
    mlp_dim = 512
)
elif args.net=="vit":
    # ViT for cifar10
    net = ViT(
    image_size = size,
    patch_size = args.patch,
    num_classes = 10,
    dim = int(args.dimhead),
    depth = args.depth,
    heads = args.head,
    mlp_dim = 512,
    dropout = 0.1,
    emb_dropout = 0.1
)
elif args.net=="vit_timm":
    import timm
    net = timm.create_model("vit_base_patch16_384", pretrained=True)
    net.head = nn.Linear(net.head.in_features, 10)
elif args.net=="cait":
    from models.cait import CaiT
    net = CaiT(
    image_size = size,
    patch_size = args.patch,
    num_classes = 10,
    dim = int(args.dimhead),
    depth = 6,   # depth of transformer for patch to patch attention only
    cls_depth=2, # depth of cross attention of CLS tokens to patch
    heads = 8,
    mlp_dim = 512,
    dropout = 0.1,
    emb_dropout = 0.1,
    layer_dropout = 0.05
)
elif args.net=="cait_small":
    from models.cait import CaiT
    net = CaiT(
    image_size = size,
    patch_size = args.patch,
    num_classes = 10,
    dim = int(args.dimhead),
    depth = 6,   # depth of transformer for patch to patch attention only
    cls_depth=2, # depth of cross attention of CLS tokens to patch
    heads = 6,
    mlp_dim = 256,
    dropout = 0.1,
    emb_dropout = 0.1,
    layer_dropout = 0.05
)
elif args.net=="swin":
    from models.swin import swin_t
    net = swin_t(window_size=args.patch,
                num_classes=10,
                downscaling_factors=(2,2,2,1))

# For Multi-GPU
if 'cuda' in device:
    print(device)
    if args.dp:
        print("using data parallel")
        net = torch.nn.DataParallel(net) # make parallel
        cudnn.benchmark = True

if args.resume:
    # Load checkpoint.
    print('==> Resuming from checkpoint..')
    assert os.path.isdir('checkpoint'), 'Error: no checkpoint directory found!'
    if 'vit' in args.net:
        checkpoint = torch.load(f'./checkpoint/{args.net}-h{net.transformer.heads}-dim{args.dimhead}-4-ckpt.t7')
    else:
        checkpoint = torch.load(f'./checkpoint/{args.net}-4-ckpt.t7')
    net.load_state_dict(checkpoint['model'])
    # best_acc = checkpoint['acc']
    # start_epoch = checkpoint['epoch']

# Loss is CE
criterion = nn.CrossEntropyLoss()

if args.opt == "adam":
    optimizer = optim.Adam(net.parameters(), lr=args.lr)
elif args.opt == "sgd":
    optimizer = optim.SGD(net.parameters(), lr=args.lr)  
    
# use cosine scheduling
scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(optimizer, args.n_epochs)

##### Training
def compute_tda(intermediate_output):
    if len(intermediate_output.shape) > 2:
        reshaped_output = intermediate_output.reshape((intermediate_output.shape[0], -1))
        diagrams = ripser(reshaped_output)['dgms']
    else:
        diagrams = ripser(intermediate_output)['dgms']
    return diagrams

def dim_reduction(x):
    reducer = umap.UMAP(n_neighbors=50, min_dist=0.3, n_components=3, random_state=42)
    activations = []
    for layer_idx, layer_activations in enumerate(x):
        layer_activations_np = layer_activations.view(layer_activations.size(0), -1).cpu().numpy()
        reduced_activations = reducer.fit_transform(layer_activations_np)
        activations.append(reduced_activations)
    all_activations = np.vstack(activations)  # Stack all activations to get the overall min and max
    min_val = all_activations.min(axis=0)
    max_val = all_activations.max(axis=0)

    fig = plt.figure(figsize=(20, 10))
    for i in range(10):
        ax = plt.subplot(2, 5, i+1, projection='3d')
        reduced_activations = activations[i]
        sc = ax.scatter(reduced_activations[:, 0], reduced_activations[:, 1], reduced_activations[:, 2], 
                        s=5, c='lightseagreen', cmap='Spectral', alpha=0.8)
        ax.set_xlim([min_val[0], max_val[0]])
        ax.set_ylim([min_val[1], max_val[1]])
        ax.set_zlim([min_val[2], max_val[2]])
    if 'vit' in args.net:
        plt.savefig(f'figs/umap/umap_{args.net}_h{args.head}_dim{args.dimhead}_cifar10_5_epoch{epoch+1}.png')
    elif 'attn' in args.net:
        plt.savefig(f'figs/umap/umap_{args.net}_h{args.head}_dim{args.dimhead}_cifar10_5_epoch{epoch+1}.png')
    else:
        plt.savefig(f'figs/umap/umap_{args.net}_cifar10_5_epoch{epoch+1}.png')
    plt.close()
    print(f'Done for dim reduction (epoch {epoch+1})')

def train(epoch):
    print('\nEpoch: %d / %d' % (epoch, args.n_epochs))
    net.train()
    train_loss = 0
    correct = 0
    total = 0
    for batch_idx, (inputs, targets) in enumerate(trainloader):
        inputs, targets = inputs.to(device), targets.to(device)
        with torch.amp.autocast(device_type=device_type, enabled=use_amp):
            outputs = net(inputs)
            loss = criterion(outputs, targets)
        scaler.scale(loss).backward()
        scaler.step(optimizer)
        scaler.update()
        optimizer.zero_grad()

        train_loss += loss.item()
        _, predicted = outputs.max(1)
        total += targets.size(0)
        correct += predicted.eq(targets).sum().item()

        progress_bar(batch_idx, len(trainloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
            % (train_loss/(batch_idx+1), 100.*correct/total, correct, total))
    acc = 100.*correct/total
        
    net.eval()
    beta1_values = []
    variances = []
    local_dim_values = []
    bottleneck_distances = []
    w1_distances = []
    w2_distances = []
    bottleneck_change_rates = []
    w1_change_rates = []
    w2_change_rates = []
    
    if epoch in [0, args.n_epochs-1]:
        with torch.no_grad():
            indices = np.where(np.array(trainset.targets) == args.cla)[0]
            filtered_data = torch.stack([trainset[i][0] for i in indices]).to(device)

            outputs, intermediate_activations = net(filtered_data, return_intermediates=True)

            outputs_part, intermediate_activations_part = net(filtered_data[:1000], return_intermediates=True)
            dim_reduction(intermediate_activations_part)

            for layer_idx, layer_activations in enumerate(intermediate_activations):
                layer_variance = np.var(layer_activations.cpu().numpy())
                variances.append(layer_variance)

                class_token_embeddings = layer_activations.view(layer_activations.size(0), -1).cpu().numpy()
                diagram = compute_tda(class_token_embeddings)
                beta1_values.append(diagram[1])
                # diagram_data_beta1 = pd.DataFrame(diagram[1], columns=["birth", "death"])
                # if args.net in ['vit', 'attn']:
                #     diagram_data_beta1.to_csv(f"pd/csv/{args.net}_h{net.transformer.heads}_dim{args.dimhead}_layer{layer_idx+1}_beta1_cifar10_5.csv", index=False)
                # else:
                #     diagram_data_beta1.to_csv(f"pd/csv/{args.net}_layer{layer_idx+1}_beta1_cifar10_5.csv", index=False)
                print(f'Finish computing Var. and PD for layer {layer_idx}. var:{layer_variance}')

                start_time = time.time()
                lpca = skdim.id.lPCA().fit_pw(class_token_embeddings, n_neighbors=50, n_jobs=1)
                local_dim = np.mean(lpca.dimension_pw_)
                local_dim_values.append(local_dim)
                end_time = time.time()
                elapsed_time = end_time - start_time
                print(f"Spent {elapsed_time:.2f} seconds for layer {layer_idx} in {args.net}. Local dimension: {local_dim}")
            for i in range(len(beta1_values) - 1):
                diag1 = beta1_values[i]
                diag2 = beta1_values[i + 1]

                start_time = time.time()
                bottleneck_dist = bottleneck_distance(diag1, diag2)
                bottleneck_distances.append(bottleneck_dist)
                end_time = time.time()
                elapsed_time = end_time - start_time
                print(f"Spent {elapsed_time:.2f} seconds. bottleneck({i}):{bottleneck_dist}")

                if i > 0:
                    bottleneck_change_rate = (bottleneck_dist - bottleneck_distances[i - 1]) / bottleneck_distances[i - 1]
                    bottleneck_change_rates.append(bottleneck_change_rate)
                    print(f"Bottleneck change rate: {bottleneck_change_rate:.4f}")

                start_time = time.time()
                w1_dist = wasserstein_distance(diag1, diag2, order=1)
                w1_distances.append(w1_dist)
                end_time = time.time()
                elapsed_time = end_time - start_time
                print(f"Spent {elapsed_time:.2f} seconds. w1:{w1_dist}")

                if i > 0:
                    w1_change_rate = (w1_dist - w1_distances[i - 1]) / w1_distances[i - 1]
                    w1_change_rates.append(w1_change_rate)
                    print(f"W1 change rate: {w1_change_rate:.4f}")

                start_time = time.time()
                w2_dist = wasserstein_distance(diag1, diag2, order=2)
                w2_distances.append(w2_dist)
                end_time = time.time()
                elapsed_time = end_time - start_time
                print(f"Spent {elapsed_time:.2f} seconds. w2:{w2_dist}")

                if i > 0:
                    w2_change_rate = (w2_dist - w2_distances[i - 1]) / w2_distances[i - 1]
                    w2_change_rates.append(w2_change_rate)
                    print(f"W2 change rate: {w2_change_rate:.4f}")

            if 'vit' in args.net or 'attn' in args.net:
                var_csv_file = f"pd/csv/Var_{args.net}_h{args.head}_dim{args.dimhead}_epoch{epoch+1}_cifar10_5.csv"
                dist_csv_file = f"pd/csv/Dist_{args.net}_h{args.head}_dim{args.dimhead}_epoch{epoch+1}_cifar10_5.csv"
                change_csv_file = f"pd/csv/Change_{args.net}_h{args.head}_dim{args.dimhead}_epoch{epoch+1}_cifar10_5.csv"
                dim_csv_file = f"pd/csv/Dim_{args.net}_h{args.head}_dim{args.dimhead}_epoch{epoch+1}_cifar10_5.csv"
            else:
                var_csv_file = f"pd/csv/Var_{args.net}_epoch{epoch+1}_cifar10_5.csv"
                dist_csv_file = f"pd/csv/Dist_{args.net}_epoch{epoch+1}_cifar10_5.csv"
                change_csv_file = f"pd/csv/Change_{args.net}_epoch{epoch+1}_cifar10_5.csv"
                dim_csv_file = f"pd/csv/Dim_{args.net}_epoch{epoch+1}_cifar10_5.csv"

            if os.path.exists(var_csv_file):
                df_var = pd.read_csv(var_csv_file)
            else:
                df_var = pd.DataFrame(columns=["Variance"])

            if os.path.exists(dist_csv_file):
                df_dist = pd.read_csv(dist_csv_file)
            else:
                df_dist = pd.DataFrame(columns=["Bottleneck Distance", "Wasserstein-1 Distance", "Wasserstein-2 Distance"])

            if os.path.exists(change_csv_file):
                df_change = pd.read_csv(change_csv_file)
            else:
                df_change = pd.DataFrame(columns=["BD Rate", "W1 Rate", "W2 Rate"])
            
            if os.path.exists(var_csv_file):
                df_dim = pd.read_csv(var_csv_file)
            else:
                df_dim = pd.DataFrame(columns=["Intrinsic Dimension"])

            df_var["Variance"] = variances
            df_dist["Bottleneck Distance"] = bottleneck_distances
            df_dist["Wasserstein-1 Distance"] = w1_distances
            df_dist["Wasserstein-2 Distance"] = w2_distances
            df_change["BD Rate"] = bottleneck_change_rates
            df_change["W1 Rate"] = w1_change_rates
            df_change["W2 Rate"] = w2_change_rates
            df_dim["Intrinsic Dimension"] = local_dim_values
            df_var.to_csv(var_csv_file, index=True)
            df_dist.to_csv(dist_csv_file, index=True)
            df_change.to_csv(change_csv_file, index=True)
            df_dim.to_csv(dim_csv_file, index=True)

            print("Variance, Distance, Change Rate, and Intrinsic Dimendion data have been saved to separate files.")


    net.train()
    return acc, train_loss/(batch_idx+1), variances

if usewandb:
    wandb.watch(net)

scaler = torch.amp.GradScaler(enabled=use_amp)
    
net.cuda()

for epoch in range(start_epoch, args.n_epochs):
    train(epoch)
