#!/usr/bin/env python3
"""
벤치마크 로그에서 swap out/swap in latency의 총합을 계산하는 스크립트
"""

import re
import argparse
import os

def parse_swap_latencies(log_file):
    """로그 파일에서 swap latency 데이터를 파싱합니다."""
    swap_out_latencies = []
    swap_in_latencies = []
    
    try:
        with open(log_file, 'r', encoding='utf-8') as f:
            for line_num, line in enumerate(f, 1):
                # SWAP_OUT latency 패턴 매칭
                swap_out_match = re.search(r'\[SWAP_OUT\] Total swap out latency: ([\d.]+) ms', line)
                if swap_out_match:
                    latency = float(swap_out_match.group(1))
                    swap_out_latencies.append({
                        'line': line_num,
                        'latency_ms': latency,
                        'text': line.strip()
                    })
                
                # SWAP_IN latency 패턴 매칭
                swap_in_match = re.search(r'\[SWAP_IN\] Total swap in latency: ([\d.]+) ms', line)
                if swap_in_match:
                    latency = float(swap_in_match.group(1))
                    swap_in_latencies.append({
                        'line': line_num,
                        'latency_ms': latency,
                        'text': line.strip()
                    })
                    
    except Exception as e:
        print(f"파일 읽기 오류: {e}")
        return None, None
    
    return swap_out_latencies, swap_in_latencies

def calculate_stats(latencies, operation_type):
    """latency 통계를 계산합니다."""
    if not latencies:
        return None
    
    latency_values = [item['latency_ms'] for item in latencies]
    
    stats = {
        'operation': operation_type,
        'count': len(latencies),
        'total_ms': sum(latency_values),
        'total_seconds': sum(latency_values) / 1000,
        'average_ms': sum(latency_values) / len(latencies),
        'min_ms': min(latency_values),
        'max_ms': max(latency_values)
    }
    
    return stats

def print_detailed_results(swap_out_data, swap_in_data, show_details=False):
    """결과를 상세히 출력합니다."""
    print("=== SWAP LATENCY 분석 결과 ===\n")
    
    # SWAP OUT 통계
    if swap_out_data:
        swap_out_stats = calculate_stats(swap_out_data, "SWAP OUT")
        print(f"📤 SWAP OUT 통계:")
        print(f"  - 총 횟수: {swap_out_stats['count']}회")
        print(f"  - 총 latency: {swap_out_stats['total_ms']:.2f} ms ({swap_out_stats['total_seconds']:.3f} 초)")
        print(f"  - 평균 latency: {swap_out_stats['average_ms']:.2f} ms")
        print(f"  - 최소 latency: {swap_out_stats['min_ms']:.2f} ms")
        print(f"  - 최대 latency: {swap_out_stats['max_ms']:.2f} ms")
        print()
        
        if show_details:
            print("상세 SWAP OUT 기록:")
            for i, item in enumerate(swap_out_data, 1):
                print(f"  {i:3d}. 라인 {item['line']:4d}: {item['latency_ms']:8.2f} ms")
            print()
    else:
        print("📤 SWAP OUT: 데이터 없음\n")
    
    # SWAP IN 통계
    if swap_in_data:
        swap_in_stats = calculate_stats(swap_in_data, "SWAP IN")
        print(f"📥 SWAP IN 통계:")
        print(f"  - 총 횟수: {swap_in_stats['count']}회")
        print(f"  - 총 latency: {swap_in_stats['total_ms']:.2f} ms ({swap_in_stats['total_seconds']:.3f} 초)")
        print(f"  - 평균 latency: {swap_in_stats['average_ms']:.2f} ms")
        print(f"  - 최소 latency: {swap_in_stats['min_ms']:.2f} ms")
        print(f"  - 최대 latency: {swap_in_stats['max_ms']:.2f} ms")
        print()
        
        if show_details:
            print("상세 SWAP IN 기록:")
            for i, item in enumerate(swap_in_data, 1):
                print(f"  {i:3d}. 라인 {item['line']:4d}: {item['latency_ms']:8.2f} ms")
            print()
    else:
        print("📥 SWAP IN: 데이터 없음\n")
    
    # 전체 통계
    if swap_out_data or swap_in_data:
        total_swap_out = sum(item['latency_ms'] for item in swap_out_data) if swap_out_data else 0
        total_swap_in = sum(item['latency_ms'] for item in swap_in_data) if swap_in_data else 0
        total_operations = (len(swap_out_data) if swap_out_data else 0) + (len(swap_in_data) if swap_in_data else 0)
        
        print("🔄 전체 SWAP 통계:")
        print(f"  - 총 SWAP 작업 횟수: {total_operations}회")
        print(f"  - 총 SWAP OUT latency: {total_swap_out:.2f} ms ({total_swap_out/1000:.3f} 초)")
        print(f"  - 총 SWAP IN latency: {total_swap_in:.2f} ms ({total_swap_in/1000:.3f} 초)")
        print(f"  - 전체 SWAP latency 합계: {total_swap_out + total_swap_in:.2f} ms ({(total_swap_out + total_swap_in)/1000:.3f} 초)")
        
        if total_operations > 0:
            avg_latency = (total_swap_out + total_swap_in) / total_operations
            print(f"  - 평균 SWAP latency: {avg_latency:.2f} ms")

def main():
    parser = argparse.ArgumentParser(description='벤치마크 로그에서 swap latency 총합 계산')
    parser.add_argument('log_file', help='벤치마크 로그 파일 경로')
    parser.add_argument('-d', '--details', action='store_true', help='상세 기록 표시')
    
    args = parser.parse_args()
    
    # 파일 존재 확인
    if not os.path.exists(args.log_file):
        print(f"오류: 파일 '{args.log_file}'이 존재하지 않습니다.")
        return
    
    print(f"로그 파일 분석 중: {args.log_file}")
    
    # swap latency 데이터 파싱
    swap_out_data, swap_in_data = parse_swap_latencies(args.log_file)
    
    if swap_out_data is None and swap_in_data is None:
        print("swap latency 데이터를 파싱할 수 없습니다.")
        return
    
    # 결과 출력
    print_detailed_results(swap_out_data, swap_in_data, args.details)

if __name__ == "__main__":
    main() 