#include <torch/all.h>
#include <ATen/cuda/CUDAContext.h>
#include <c10/cuda/CUDAGuard.h>

#include "cuda_compat.h"
#include "dispatch_utils.h"

#ifdef USE_ROCM
  #include "quantization/fp8/amd/quant_utils.cuh"
#else
  #include "quantization/fp8/nvidia/quant_utils.cuh"
#endif

#include <algorithm>
#include <cassert>
#include <map>
#include <vector>

#ifdef USE_ROCM
  #include <hip/hip_bf16.h>
typedef __hip_bfloat16 __nv_bfloat16;
#endif

// For flash cache with GDS
// #ifdef USE_CUFILE
#include <cuda_runtime.h>
#include <cuda.h>
#include <cufile.h>
#include <pthread.h>
#include <queue>
#include <atomic>
#include <iostream>

// cuFile constants
#ifndef CU_FILE_READ_WRITE
#define CU_FILE_READ_WRITE 3  // Read and write permissions
#endif

#ifndef CU_FILE_STREAM_FIXED_AND_ALIGNED
#define CU_FILE_STREAM_FIXED_AND_ALIGNED 1
#endif

#ifndef CU_FILE_BUF_FIXED_AND_ALIGNED  
#define CU_FILE_BUF_FIXED_AND_ALIGNED 1
#endif
// #endif

// File system related headers
#include <fcntl.h>      // O_RDWR, O_DIRECT macro definitions
#include <unistd.h>     // open() function definition

// Error handling related headers
#include <errno.h>      // errno, strerror() functions
#include <string.h>     // strerror() function

void swap_blocks(torch::Tensor& src, torch::Tensor& dst,
                 const torch::Tensor& block_mapping) {
  torch::Device src_device = src.device();
  torch::Device dst_device = dst.device();
  cudaMemcpyKind memcpy_type;
  if (src_device.is_cuda() && dst_device.is_cuda()) {
    TORCH_CHECK(src_device.index() == dst_device.index(),
                "src and dst must be on the same GPU");
    memcpy_type = cudaMemcpyDeviceToDevice;
  } else if (src_device.is_cuda() && dst_device.is_cpu()) {
    memcpy_type = cudaMemcpyDeviceToHost;
  } else if (src_device.is_cpu() && dst_device.is_cuda()) {
    memcpy_type = cudaMemcpyHostToDevice;
  } else {
    TORCH_CHECK(false, "Invalid device combination");
  }

  // NOTE(youkaichao): keep in mind that `block_mapping` should be
  // a cpu tensor, otherwise every `item` call will require a gpu-cpu
  // synchronization.
  TORCH_CHECK(block_mapping.device().is_cpu(), "block_mapping must be on CPU");

  char* src_ptr = static_cast<char*>(src.data_ptr());
  char* dst_ptr = static_cast<char*>(dst.data_ptr());

  const int64_t block_size_in_bytes = src.element_size() * src[0].numel();
  const at::cuda::OptionalCUDAGuard device_guard(
      src_device.is_cuda() ? src_device : dst_device);
  const cudaStream_t stream = at::cuda::getCurrentCUDAStream();
  // NOTE(woosuk): This can be slow if the number of blocks is large.
  const int64_t num_blocks = block_mapping.size(0);
  for (size_t i = 0; i < num_blocks; i++) {
    int64_t src_block_number = block_mapping[i][0].item<int64_t>();
    int64_t dst_block_number = block_mapping[i][1].item<int64_t>();
    int64_t src_offset = src_block_number * block_size_in_bytes;
    int64_t dst_offset = dst_block_number * block_size_in_bytes;
    cudaMemcpyAsync(dst_ptr + dst_offset, src_ptr + src_offset,
                    block_size_in_bytes, memcpy_type, stream);
  }
}

namespace vllm {

// Grid: (num_layers, num_pairs)
template <typename scalar_t>
__global__ void copy_blocks_kernel(int64_t* key_cache_ptrs,
                                   int64_t* value_cache_ptrs,
                                   const int64_t* __restrict__ block_mapping,
                                   const int numel_per_block) {
  const int layer_idx = blockIdx.x;
  const int pair_idx = blockIdx.y;

  scalar_t* key_cache = reinterpret_cast<scalar_t*>(key_cache_ptrs[layer_idx]);
  scalar_t* value_cache =
      reinterpret_cast<scalar_t*>(value_cache_ptrs[layer_idx]);
  int64_t src_block_number = block_mapping[2 * pair_idx];
  int64_t dst_block_number = block_mapping[2 * pair_idx + 1];

  const int64_t src_block_offset = src_block_number * numel_per_block;
  const int64_t dst_block_offset = dst_block_number * numel_per_block;
  for (int i = threadIdx.x; i < numel_per_block; i += blockDim.x) {
    int64_t src_offset = src_block_offset + i;
    int64_t dst_offset = dst_block_offset + i;
    key_cache[dst_offset] = key_cache[src_offset];
  }
  for (int i = threadIdx.x; i < numel_per_block; i += blockDim.x) {
    int64_t src_offset = src_block_offset + i;
    int64_t dst_offset = dst_block_offset + i;
    value_cache[dst_offset] = value_cache[src_offset];
  }
}

}  // namespace vllm

// Note: the key_caches and value_caches vectors are constant but
// not the Tensors they contain. The vectors need to be const refs
// in order to satisfy pytorch's C++ operator registration code.
void copy_blocks(std::vector<torch::Tensor> const& key_caches,
                 std::vector<torch::Tensor> const& value_caches,
                 const torch::Tensor& block_mapping) {
  int num_layers = key_caches.size();
  TORCH_CHECK(num_layers == value_caches.size());
  if (num_layers == 0) {
    return;
  }
  torch::Device cache_device = key_caches[0].device();
  TORCH_CHECK(cache_device.is_cuda());

  // Create data structures for the kernel.
  // Create an array of pointers to the key and value caches.
  int64_t key_cache_ptrs[num_layers];
  int64_t value_cache_ptrs[num_layers];
  for (int layer_idx = 0; layer_idx < num_layers; ++layer_idx) {
    key_cache_ptrs[layer_idx] =
        reinterpret_cast<int64_t>(key_caches[layer_idx].data_ptr());
    value_cache_ptrs[layer_idx] =
        reinterpret_cast<int64_t>(value_caches[layer_idx].data_ptr());
  }

  // block_mapping is a 2D tensor with shape (num_pairs, 2).
  int num_pairs = block_mapping.size(0);

  // Move the data structures to the GPU.
  // NOTE: This synchronizes the CPU and GPU.
  torch::Tensor key_cache_ptrs_tensor =
      torch::from_blob(key_cache_ptrs, {num_layers}, torch::kInt64)
          .to(cache_device);
  torch::Tensor value_cache_ptrs_tensor =
      torch::from_blob(value_cache_ptrs, {num_layers}, torch::kInt64)
          .to(cache_device);

  // Launch the kernel.
  const int numel_per_block = key_caches[0][0].numel();
  dim3 grid(num_layers, num_pairs);
  dim3 block(std::min(1024, numel_per_block));
  const at::cuda::OptionalCUDAGuard device_guard(cache_device);
  const cudaStream_t stream = at::cuda::getCurrentCUDAStream();
  VLLM_DISPATCH_FLOATING_AND_BYTE_TYPES(
      key_caches[0].scalar_type(), "copy_blocks_kernel", ([&] {
        vllm::copy_blocks_kernel<scalar_t><<<grid, block, 0, stream>>>(
            key_cache_ptrs_tensor.data_ptr<int64_t>(),
            value_cache_ptrs_tensor.data_ptr<int64_t>(),
            block_mapping.data_ptr<int64_t>(), numel_per_block);
      }));
}

namespace vllm {

template <typename scalar_t, typename cache_t, Fp8KVCacheDataType kv_dt>
__global__ void reshape_and_cache_kernel(
    const scalar_t* __restrict__ key,    // [num_tokens, num_heads, head_size]
    const scalar_t* __restrict__ value,  // [num_tokens, num_heads, head_size]
    cache_t* __restrict__ key_cache,     // [num_blocks, num_heads, head_size/x,
                                         // block_size, x]
    cache_t* __restrict__ value_cache,   // [num_blocks, num_heads, head_size,
                                         // block_size]
    const int64_t* __restrict__ slot_mapping,  // [num_tokens]
    const int key_stride, const int value_stride, const int num_heads,
    const int head_size, const int block_size, const int x, const float k_scale,
    const float v_scale) {
  const int64_t token_idx = blockIdx.x;
  const int64_t slot_idx = slot_mapping[token_idx];
  if (slot_idx < 0) {
    // Padding token that should be ignored.
    return;
  }

  const int64_t block_idx = slot_idx / block_size;
  const int64_t block_offset = slot_idx % block_size;

  const int n = num_heads * head_size;
  for (int i = threadIdx.x; i < n; i += blockDim.x) {
    const int64_t src_key_idx = token_idx * key_stride + i;
    const int64_t src_value_idx = token_idx * value_stride + i;

    const int head_idx = i / head_size;
    const int head_offset = i % head_size;
    const int x_idx = head_offset / x;
    const int x_offset = head_offset % x;

    const int64_t tgt_key_idx =
        block_idx * num_heads * (head_size / x) * block_size * x +
        head_idx * (head_size / x) * block_size * x + x_idx * block_size * x +
        block_offset * x + x_offset;
    const int64_t tgt_value_idx =
        block_idx * num_heads * head_size * block_size +
        head_idx * head_size * block_size + head_offset * block_size +
        block_offset;
    scalar_t tgt_key = key[src_key_idx];
    scalar_t tgt_value = value[src_value_idx];
    if constexpr (kv_dt == Fp8KVCacheDataType::kAuto) {
      key_cache[tgt_key_idx] = tgt_key;
      value_cache[tgt_value_idx] = tgt_value;
    } else {
      key_cache[tgt_key_idx] =
          fp8::scaled_convert<cache_t, scalar_t, kv_dt>(tgt_key, k_scale);
      value_cache[tgt_value_idx] =
          fp8::scaled_convert<cache_t, scalar_t, kv_dt>(tgt_value, v_scale);
    }
  }
}

template <typename scalar_t, typename cache_t, Fp8KVCacheDataType kv_dt>
__global__ void reshape_and_cache_flash_kernel(
    const scalar_t* __restrict__ key,    // [num_tokens, num_heads, head_size]
    const scalar_t* __restrict__ value,  // [num_tokens, num_heads, head_size]
    cache_t* __restrict__ key_cache,     // [num_blocks, block_size, num_heads,
                                         // head_size]
    cache_t* __restrict__ value_cache,   // [num_blocks, block_size, num_heads,
                                         // head_size]
    const int64_t* __restrict__ slot_mapping,  // [num_tokens]
    const int block_stride, const int key_stride, const int value_stride,
    const int num_heads, const int head_size, const int block_size,
    const float k_scale, const float v_scale) {
  const int64_t token_idx = blockIdx.x;
  const int64_t slot_idx = slot_mapping[token_idx];
  // NOTE: slot_idx can be -1 if the token is padded
  if (slot_idx < 0) {
    return;
  }
  const int64_t block_idx = slot_idx / block_size;
  const int64_t block_offset = slot_idx % block_size;
  const int n = num_heads * head_size;
  for (int i = threadIdx.x; i < n; i += blockDim.x) {
    const int64_t src_key_idx = token_idx * key_stride + i;
    const int64_t src_value_idx = token_idx * value_stride + i;
    const int head_idx = i / head_size;
    const int head_offset = i % head_size;
    const int64_t tgt_key_value_idx = block_idx * block_stride +
                                      block_offset * num_heads * head_size +
                                      head_idx * head_size + head_offset;
    scalar_t tgt_key = key[src_key_idx];
    scalar_t tgt_value = value[src_value_idx];
    if constexpr (kv_dt == Fp8KVCacheDataType::kAuto) {
      key_cache[tgt_key_value_idx] = tgt_key;
      value_cache[tgt_key_value_idx] = tgt_value;
    } else {
      key_cache[tgt_key_value_idx] =
          fp8::scaled_convert<cache_t, scalar_t, kv_dt>(tgt_key, k_scale);
      value_cache[tgt_key_value_idx] =
          fp8::scaled_convert<cache_t, scalar_t, kv_dt>(tgt_value, v_scale);
    }
  }
}
}  // namespace vllm

// KV_T is the stored data type of kv-cache.
// CACHE_T is the data type of key and value tensors.
// KV_DTYPE is the real data type of kv-cache.
#define CALL_RESHAPE_AND_CACHE(KV_T, CACHE_T, KV_DTYPE)               \
  vllm::reshape_and_cache_kernel<KV_T, CACHE_T, KV_DTYPE>             \
      <<<grid, block, 0, stream>>>(                                   \
          reinterpret_cast<KV_T*>(key.data_ptr()),                    \
          reinterpret_cast<KV_T*>(value.data_ptr()),                  \
          reinterpret_cast<CACHE_T*>(key_cache.data_ptr()),           \
          reinterpret_cast<CACHE_T*>(value_cache.data_ptr()),         \
          slot_mapping.data_ptr<int64_t>(), key_stride, value_stride, \
          num_heads, head_size, block_size, x, k_scale, v_scale);

void reshape_and_cache(
    torch::Tensor& key,    // [num_tokens, num_heads, head_size]
    torch::Tensor& value,  // [num_tokens, num_heads, head_size]
    torch::Tensor&
        key_cache,  // [num_blocks, num_heads, head_size/x, block_size, x]
    torch::Tensor&
        value_cache,  // [num_blocks, num_heads, head_size, block_size]
    torch::Tensor& slot_mapping,  // [num_tokens]
    const std::string& kv_cache_dtype, const double k_scale,
    const double v_scale) {
  int num_tokens = key.size(0);
  int num_heads = key.size(1);
  int head_size = key.size(2);
  int block_size = key_cache.size(3);
  int x = key_cache.size(4);

  int key_stride = key.stride(0);
  int value_stride = value.stride(0);

  dim3 grid(num_tokens);
  dim3 block(std::min(num_heads * head_size, 512));
  const at::cuda::OptionalCUDAGuard device_guard(device_of(key));
  const cudaStream_t stream = at::cuda::getCurrentCUDAStream();

  DISPATCH_BY_KV_CACHE_DTYPE(key.dtype(), kv_cache_dtype,
                             CALL_RESHAPE_AND_CACHE)
}

// KV_T is the stored data type of kv-cache.
// CACHE_T is the data type of key and value tensors.
// KV_DTYPE is the real data type of kv-cache.
#define CALL_RESHAPE_AND_CACHE_FLASH(KV_T, CACHE_T, KV_DTYPE)         \
  vllm::reshape_and_cache_flash_kernel<KV_T, CACHE_T, KV_DTYPE>       \
      <<<grid, block, 0, stream>>>(                                   \
          reinterpret_cast<KV_T*>(key.data_ptr()),                    \
          reinterpret_cast<KV_T*>(value.data_ptr()),                  \
          reinterpret_cast<CACHE_T*>(key_cache.data_ptr()),           \
          reinterpret_cast<CACHE_T*>(value_cache.data_ptr()),         \
          slot_mapping.data_ptr<int64_t>(), block_stride, key_stride, \
          value_stride, num_heads, head_size, block_size, k_scale, v_scale);

void reshape_and_cache_flash(
    torch::Tensor& key,        // [num_tokens, num_heads, head_size]
    torch::Tensor& value,      // [num_tokens, num_heads, head_size]
    torch::Tensor& key_cache,  // [num_blocks, block_size, num_heads, head_size]
    torch::Tensor&
        value_cache,  // [num_blocks, block_size, num_heads, head_size]
    torch::Tensor& slot_mapping,  // [num_tokens] or [num_actual_tokens]
    const std::string& kv_cache_dtype, const double k_scale,
    const double v_scale) {
  // NOTE(woosuk): In vLLM V1, key.size(0) can be different from
  // slot_mapping.size(0) because of padding for CUDA graphs.
  // In vLLM V0, key.size(0) is always equal to slot_mapping.size(0) because
  // both include padding.
  // In vLLM V1, however, key.size(0) can be larger than slot_mapping.size(0)
  // since key includes padding for CUDA graphs, while slot_mapping does not.
  // In this case, slot_mapping.size(0) represents the actual number of tokens
  // before padding.
  // For compatibility with both cases, we use slot_mapping.size(0) as the
  // number of tokens.
  int num_tokens = slot_mapping.size(0);
  int num_heads = key.size(1);
  int head_size = key.size(2);
  int block_size = key_cache.size(1);

  int key_stride = key.stride(0);
  int value_stride = value.stride(0);
  int block_stride = key_cache.stride(0);
  TORCH_CHECK(key_cache.stride(0) == value_cache.stride(0));

  dim3 grid(num_tokens);
  dim3 block(std::min(num_heads * head_size, 512));
  const at::cuda::OptionalCUDAGuard device_guard(device_of(key));
  const cudaStream_t stream = at::cuda::getCurrentCUDAStream();

  DISPATCH_BY_KV_CACHE_DTYPE(key.dtype(), kv_cache_dtype,
                             CALL_RESHAPE_AND_CACHE_FLASH);
}

namespace vllm {

template <typename Tout, typename Tin, Fp8KVCacheDataType kv_dt>
__global__ void convert_fp8_kernel(const Tin* __restrict__ src_cache,
                                   Tout* __restrict__ dst_cache,
                                   const float scale,
                                   const int64_t block_stride) {
  const int64_t block_idx = blockIdx.x;
  for (int i = threadIdx.x; i < block_stride; i += blockDim.x) {
    int64_t idx = block_idx * block_stride + i;
    dst_cache[idx] =
        fp8::scaled_convert<Tout, Tin, kv_dt>(src_cache[idx], scale);
  }
}

}  // namespace vllm

#define CALL_CONVERT_FP8(Tout, Tin, KV_DTYPE)                                \
  vllm::convert_fp8_kernel<Tout, Tin, KV_DTYPE><<<grid, block, 0, stream>>>( \
      reinterpret_cast<Tin*>(src_cache.data_ptr()),                          \
      reinterpret_cast<Tout*>(dst_cache.data_ptr()), scale, block_stride);

// Only for testing.
void convert_fp8(torch::Tensor& dst_cache, torch::Tensor& src_cache,
                 const double scale, const std::string& kv_cache_dtype) {
  torch::Device src_device = src_cache.device();
  torch::Device dst_device = dst_cache.device();
  TORCH_CHECK(src_device.is_cuda(), "src must be on a GPU")
  TORCH_CHECK(dst_device.is_cuda(), "dst must be on a GPU")
  TORCH_CHECK(src_device.index() == dst_device.index(),
              "src and dst must be on the same GPU");
  at::cuda::OptionalCUDAGuard device_guard(src_device);

  int64_t num_blocks = src_cache.size(0);
  int64_t block_stride = src_cache.stride(0);

  dim3 grid(num_blocks);
  dim3 block(std::min(block_stride, int64_t(512)));
  const cudaStream_t stream = at::cuda::getCurrentCUDAStream();

  if (kv_cache_dtype == "auto") {
    if (src_cache.dtype() == at::ScalarType::Float) {
      CALL_CONVERT_FP8(uint8_t, float, vllm::Fp8KVCacheDataType::kAuto);
    } else if (src_cache.dtype() == at::ScalarType::Half) {
      CALL_CONVERT_FP8(uint8_t, uint16_t, vllm::Fp8KVCacheDataType::kAuto);
    } else if (src_cache.dtype() == at::ScalarType::BFloat16) {
      CALL_CONVERT_FP8(uint8_t, __nv_bfloat16, vllm::Fp8KVCacheDataType::kAuto);
    } else if (dst_cache.dtype() == at::ScalarType::Float) {
      CALL_CONVERT_FP8(float, uint8_t, vllm::Fp8KVCacheDataType::kAuto);
    } else if (dst_cache.dtype() == at::ScalarType::Half) {
      CALL_CONVERT_FP8(uint16_t, uint8_t, vllm::Fp8KVCacheDataType::kAuto);
    } else if (dst_cache.dtype() == at::ScalarType::BFloat16) {
      CALL_CONVERT_FP8(__nv_bfloat16, uint8_t, vllm::Fp8KVCacheDataType::kAuto);
    }
  } else if (kv_cache_dtype == "fp8" || kv_cache_dtype == "fp8_e4m3") {
    if (src_cache.dtype() == at::ScalarType::Float) {
      CALL_CONVERT_FP8(uint8_t, float, vllm::Fp8KVCacheDataType::kFp8E4M3);
    } else if (src_cache.dtype() == at::ScalarType::Half) {
      CALL_CONVERT_FP8(uint8_t, uint16_t, vllm::Fp8KVCacheDataType::kFp8E4M3);
    } else if (src_cache.dtype() == at::ScalarType::BFloat16) {
      CALL_CONVERT_FP8(uint8_t, __nv_bfloat16,
                       vllm::Fp8KVCacheDataType::kFp8E4M3);
    } else if (dst_cache.dtype() == at::ScalarType::Float) {
      CALL_CONVERT_FP8(float, uint8_t, vllm::Fp8KVCacheDataType::kFp8E4M3);
    } else if (dst_cache.dtype() == at::ScalarType::Half) {
      CALL_CONVERT_FP8(uint16_t, uint8_t, vllm::Fp8KVCacheDataType::kFp8E4M3);
    } else if (dst_cache.dtype() == at::ScalarType::BFloat16) {
      CALL_CONVERT_FP8(__nv_bfloat16, uint8_t,
                       vllm::Fp8KVCacheDataType::kFp8E4M3);
    }
  } else {
    TORCH_CHECK(false, "Unsupported data type: ", kv_cache_dtype);
  }
}

#define MAX_NUM_THREADS (16)
#define MAX_THREAD_IO_SIZE_MB (16)
#define MAX_THREAD_IO_SIZE_BYTES (MAX_THREAD_IO_SIZE_MB * 1024 * 1024)
struct ThreadData {
    CUfileHandle_t handle;
    char* src_addr;
    char* dst_addr;
    int64_t total_size;
    int64_t dst_offset;
    int64_t src_offset;
    int64_t buf_offset;
};

static pthread_t g_pool_threads[MAX_NUM_THREADS];
static pthread_mutex_t g_task_queue_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t  g_task_queue_cond  = PTHREAD_COND_INITIALIZER;

// Thread Pool shutdown flag
static bool g_shutdown = false;

// Track remaining tasks count
static std::atomic<int> g_tasks_in_progress{0};

// Queue to hold tasks
struct Task {
    bool is_gpu_to_flash;
    ThreadData data; // Information needed for cuFileWrite/read
};

// Queue to hold actual tasks
static std::queue<Task> g_task_queue;

// Initialize cuFile driver
int64_t init_gds_for_flash_cache(const std::string& file_path, 
                                  torch::Tensor& dummy_tensor) {
  CUfileHandle_t file_handle;
  CUfileDescr_t descr;
  int64_t fd = -1;

  try {
    // Initialize GDS driver
    CUfileError_t status = cuFileDriverOpen();
    TORCH_CHECK(status.err == CU_FILE_SUCCESS, "cuFileDriverOpen failed");
    
    // Set file options
    int64_t flags = O_RDWR | O_DIRECT;

    // Open file
    fd = ::open(file_path.c_str(), flags, 0666);
    TORCH_CHECK(fd >= 0, "Failed to open file: " + std::string(strerror(errno)));

    memset(&descr, 0, sizeof(CUfileDescr_t));
    descr.handle.fd = fd;
    descr.type = CU_FILE_HANDLE_TYPE_OPAQUE_FD;
    
    // Register GDS handle
    status = cuFileHandleRegister(&file_handle, &descr);
    TORCH_CHECK(status.err == CU_FILE_SUCCESS, "cuFileHandleRegister failed");

    return reinterpret_cast<int64_t>(file_handle);
  } catch (...) {
    // Cleanup on error
    if (fd >= 0) {
      ::close(fd);
    }
    cuFileDriverClose();
    throw;
  }
}

void reset_gds_for_flash_cache(int64_t file_handle, torch::Tensor& dummy_tensor) {
  CUfileHandle_t handle = reinterpret_cast<CUfileHandle_t>(file_handle);
  
  cuFileHandleDeregister(handle);
  cuFileDriverClose();
}

void register_gpu_cache_buffer(torch::Tensor& gpu_kv_cache) {
  char* gpu_kv_cache_ptr = static_cast<char*>(gpu_kv_cache.data_ptr());
  size_t total_cache_size = gpu_kv_cache.numel() * gpu_kv_cache.element_size();
  
  // Synchronize for GPU memory access optimization
  cudaError_t cuda_status = cudaDeviceSynchronize();
  TORCH_CHECK(cuda_status == cudaSuccess, "cudaDeviceSynchronize failed: ", cudaGetErrorString(cuda_status));
  
  // Register GDS buffer
  //CUfileError_t status = cuFileBufRegister(gpu_kv_cache_ptr, total_cache_size, CU_FILE_READ_WRITE);
  CUfileError_t status = cuFileBufRegister(gpu_kv_cache_ptr, total_cache_size, 0);
  TORCH_CHECK(status.err == CU_FILE_SUCCESS, "cuFileBufRegister failed: ", status.err);
}

void deregister_gpu_cache_buffer(torch::Tensor& gpu_kv_cache) {
  char* gpu_kv_cache_ptr = static_cast<char*>(gpu_kv_cache.data_ptr());
  
  // Wait for GPU operations to complete
  cudaError_t cuda_status = cudaDeviceSynchronize();
  TORCH_CHECK(cuda_status == cudaSuccess, "cudaDeviceSynchronize failed: ", cudaGetErrorString(cuda_status));
  
  // Deregister buffer
  CUfileError_t status = cuFileBufDeregister(gpu_kv_cache_ptr);
  TORCH_CHECK(status.err == CU_FILE_SUCCESS, "cuFileBufDeregister failed: ", status.err);
}

void* thread_pool_worker(void* arg)
{
    while (true) {
        pthread_mutex_lock(&g_task_queue_mutex);

        // Wait while queue is empty and no shutdown command
        while (g_task_queue.empty() && !g_shutdown) {
            pthread_cond_wait(&g_task_queue_cond, &g_task_queue_mutex);
        }

        // Exit thread if shutdown is required
        if (g_shutdown && g_task_queue.empty()) {
            pthread_mutex_unlock(&g_task_queue_mutex);
            break;
        }

        // Get one task from queue
        Task current_task = g_task_queue.front();
        g_task_queue.pop();

        pthread_mutex_unlock(&g_task_queue_mutex);

        // Perform actual work
        if (current_task.is_gpu_to_flash) {
            // Same logic as thread_write
            const auto& data = current_task.data;
            ssize_t bytes_transferred = cuFileWrite(
                data.handle,
                data.src_addr,
                data.total_size,
                data.dst_offset,
                data.buf_offset
            );
            TORCH_CHECK(bytes_transferred >= 0, "cuFileWrite failed in thread");
        } else {
            // Same logic as thread_read
            const auto& data = current_task.data;
            ssize_t bytes_transferred = cuFileRead(
                data.handle,
                data.dst_addr,
                data.total_size,
                data.src_offset,
                data.buf_offset
            );
            TORCH_CHECK(bytes_transferred >= 0, "cuFileRead failed in thread");
        }

        // One task completed -> decrease tasks in progress count
        g_tasks_in_progress.fetch_sub(1, std::memory_order_relaxed);
    }

    return nullptr;
}

void init_thread_pool(int num_threads)
{
    // Create num_threads threads
    g_shutdown = false;

    for (int i = 0; i < num_threads; i++) {
        pthread_create(&g_pool_threads[i], nullptr, thread_pool_worker, nullptr);
    }
}

void shutdown_thread_pool(int num_threads)
{
    // Send thread pool shutdown signal
    pthread_mutex_lock(&g_task_queue_mutex);
    g_shutdown = true;
    pthread_cond_broadcast(&g_task_queue_cond);
    pthread_mutex_unlock(&g_task_queue_mutex);

    // Join all threads
    for (int i = 0; i < num_threads; i++) {
        pthread_join(g_pool_threads[i], nullptr);
    }
}

void enqueue_task(const Task& new_task)
{
    // Increase tasks in progress count when task arrives
    g_tasks_in_progress.fetch_add(1, std::memory_order_relaxed);

    // Protect queue
    pthread_mutex_lock(&g_task_queue_mutex);

    g_task_queue.push(new_task);

    // Notify that new task has arrived
    pthread_cond_signal(&g_task_queue_cond);
    pthread_mutex_unlock(&g_task_queue_mutex);
}

void wait_for_all_tasks_done()
{
    // Simple spin/polling
    while (g_tasks_in_progress.load(std::memory_order_relaxed) > 0) {
        // Small sleep or yield
        sched_yield(); 
    }
}

void swap_flash_blocks(int64_t file_handle,
                      torch::Tensor& gpu_kv_cache,
                      const torch::Tensor& block_mapping, 
                      const torch::Tensor& tuning_params,
                      bool is_gpu_to_flash) {
  CUfileHandle_t handle = reinterpret_cast<CUfileHandle_t>(file_handle);
  torch::Device gpu_kv_cache_device = gpu_kv_cache.device();

  char* gpu_kv_cache_ptr = static_cast<char*>(gpu_kv_cache.data_ptr());
  
  const int64_t num_blocks = block_mapping.size(0);
  const int64_t block_size_in_bytes = gpu_kv_cache.element_size() * gpu_kv_cache[0].numel();

  // tuning_params has "num_threads" and "io_size_bytes"
  const int64_t num_threads = tuning_params[0].item<int64_t>();
  TORCH_CHECK(num_threads <= MAX_NUM_THREADS, "num_threads must be less than or equal to MAX_NUM_THREADS");
  init_thread_pool(num_threads);

  const int64_t io_size_bytes = tuning_params[1].item<int64_t>();
  TORCH_CHECK(io_size_bytes <= MAX_THREAD_IO_SIZE_BYTES, "io_size_bytes must be less than or equal to MAX_THREAD_IO_SIZE_BYTES");
  
  const int64_t fc_cache_offset_bytes = tuning_params[2].item<int64_t>();

  auto block_mapping_cpu = block_mapping.to(torch::kCPU);
  auto src_blocks = block_mapping_cpu.select(1, 0);
  auto dst_blocks = block_mapping_cpu.select(1, 1);

  // Process each block individually
  for (int64_t i = 0; i < num_blocks; i++) {
      int64_t src_block = src_blocks[i].item<int64_t>();
      int64_t dst_block = dst_blocks[i].item<int64_t>();

      ThreadData tdata;
      tdata.handle     = handle;
      tdata.total_size = block_size_in_bytes;
      tdata.buf_offset = 0; 

      if (is_gpu_to_flash) {
          tdata.src_addr   = gpu_kv_cache_ptr;
          tdata.buf_offset = src_block * block_size_in_bytes;
          tdata.dst_offset = dst_block * block_size_in_bytes + fc_cache_offset_bytes;
      } else {
          tdata.dst_addr = gpu_kv_cache_ptr;
          tdata.buf_offset = dst_block * block_size_in_bytes;
          tdata.src_offset = src_block * block_size_in_bytes + fc_cache_offset_bytes;
      }

      Task new_task;
      new_task.is_gpu_to_flash = is_gpu_to_flash;
      new_task.data = tdata;

      enqueue_task(new_task);
  }
  wait_for_all_tasks_done();
  shutdown_thread_pool(num_threads);
}