#!/usr/bin/env python3
"""
NVIDIA GPU 0번 전력 소모량 모니터링 스크립트
"""

import time
import subprocess
import argparse
from datetime import datetime
import csv
import os

def get_gpu_power(gpu_id=0):
    """지정된 GPU의 현재 전력 소모량을 가져옵니다 (mW 단위)"""
    try:
        # nvidia-smi 명령어로 GPU 정보 가져오기
        result = subprocess.run(
            ['nvidia-smi', '--query-gpu=power.draw', '--format=csv,noheader,nounits', '-i', str(gpu_id)],
            capture_output=True, text=True, check=True
        )
        power_str = result.stdout.strip()
        # "123 W" 형태에서 숫자만 추출
        power_mw = float(power_str.replace(' W', '')) * 1000
        return power_mw
    except subprocess.CalledProcessError as e:
        print(f"nvidia-smi 실행 오류: {e}")
        return None
    except ValueError as e:
        print(f"전력 값 파싱 오류: {e}")
        return None

def monitor_gpu_power(gpu_id=0, interval=1, duration=None, output_file=None):
    """
    GPU 전력 소모량을 모니터링합니다.
    
    Args:
        gpu_id (int): 모니터링할 GPU ID
        interval (int): 모니터링 간격 (초)
        duration (int): 모니터링 지속 시간 (초), None이면 무한
        output_file (str): CSV 파일로 저장할 파일명
    """
    print(f"GPU {gpu_id}번 전력 소모량 모니터링 시작")
    print(f"모니터링 간격: {interval}초")
    if duration:
        print(f"모니터링 지속 시간: {duration}초")
    if output_file:
        print(f"결과 저장 파일: {output_file}")
    print("-" * 50)
    
    # CSV 파일 초기화
    csv_writer = None
    csv_file = None
    if output_file:
        csv_file = open(output_file, 'w', newline='')
        csv_writer = csv.writer(csv_file)
        csv_writer.writerow(['시간', '전력(mW)', '전력(W)'])
    
    start_time = time.time()
    count = 0
    
    try:
        while True:
            current_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            power_mw = get_gpu_power(gpu_id)
            
            if power_mw is not None:
                power_w = power_mw / 1000
                print(f"[{current_time}] GPU {gpu_id}번 전력: {power_mw:.1f} mW ({power_w:.3f} W)")
                
                if csv_writer:
                    csv_writer.writerow([current_time, f"{power_mw:.1f}", f"{power_w:.3f}"])
                    csv_file.flush()  # 즉시 파일에 쓰기
            else:
                print(f"[{current_time}] 전력 정보를 가져올 수 없습니다.")
            
            count += 1
            
            # 지속 시간 체크
            if duration and (time.time() - start_time) >= duration:
                break
                
            time.sleep(interval)
            
    except KeyboardInterrupt:
        print("\n모니터링이 중단되었습니다.")
    finally:
        if csv_file:
            csv_file.close()
            print(f"\n총 {count}개의 데이터 포인트가 {output_file}에 저장되었습니다.")

def get_available_gpus():
    """사용 가능한 GPU 목록을 가져옵니다."""
    try:
        result = subprocess.run(['nvidia-smi', '--list-gpus'], capture_output=True, text=True, check=True)
        gpu_lines = result.stdout.strip().split('\n')
        available_gpus = []
        for line in gpu_lines:
            if line.startswith('GPU '):
                # "GPU 0: NVIDIA GeForce RTX 4090" 형태에서 GPU ID 추출
                gpu_id = int(line.split(':')[0].split()[1])
                available_gpus.append(gpu_id)
        return available_gpus
    except (subprocess.CalledProcessError, FileNotFoundError):
        return []

def main():
    parser = argparse.ArgumentParser(description='NVIDIA GPU 전력 소모량 모니터링')
    parser.add_argument('-g', '--gpu', type=int, default=0, 
                       help='모니터링할 GPU ID (기본값: 0)')
    parser.add_argument('-i', '--interval', type=int, default=1, 
                       help='모니터링 간격 (초, 기본값: 1)')
    parser.add_argument('-d', '--duration', type=int, 
                       help='모니터링 지속 시간 (초, 기본값: 무한)')
    parser.add_argument('-o', '--output', type=str, 
                       help='결과를 저장할 CSV 파일명')
    parser.add_argument('-l', '--list', action='store_true',
                       help='사용 가능한 GPU 목록을 표시합니다')
    
    args = parser.parse_args()
    
    # nvidia-smi 설치 확인
    try:
        subprocess.run(['nvidia-smi', '--version'], capture_output=True, check=True)
    except (subprocess.CalledProcessError, FileNotFoundError):
        print("오류: nvidia-smi가 설치되지 않았거나 NVIDIA 드라이버가 로드되지 않았습니다.")
        print("NVIDIA 드라이버와 CUDA를 설치해주세요.")
        return
    
    # 사용 가능한 GPU 목록 가져오기
    available_gpus = get_available_gpus()
    
    if args.list:
        if available_gpus:
            print("사용 가능한 GPU 목록:")
            for gpu_id in available_gpus:
                print(f"  GPU {gpu_id}")
        else:
            print("사용 가능한 GPU가 없습니다.")
        return
    
    # GPU ID 유효성 검사
    if args.gpu not in available_gpus:
        print(f"오류: GPU {args.gpu}번이 존재하지 않습니다.")
        if available_gpus:
            print(f"사용 가능한 GPU: {', '.join(map(str, available_gpus))}")
            print("사용 가능한 GPU 목록을 보려면: python3 gpu_power_monitor.py -l")
        else:
            print("사용 가능한 GPU가 없습니다.")
        return
    
    monitor_gpu_power(args.gpu, args.interval, args.duration, args.output)

if __name__ == "__main__":
    main() 