#!/usr/bin/env python3
"""
로그 파일에서 시간별 전력 데이터를 읽어서 그래프를 생성하는 스크립트
"""

import pandas as pd
import matplotlib.pyplot as plt
import matplotlib.dates as mdates
from datetime import datetime
import argparse
import os

def load_power_data(log_file):
    """로그 파일에서 전력 데이터를 로드합니다."""
    try:
        # CSV 파일로 읽기
        df = pd.read_csv(log_file)
        
        # 시간 컬럼을 datetime으로 변환
        df['시간'] = pd.to_datetime(df['시간'])
        
        # 전력(W) 컬럼을 float로 변환
        df['전력(W)'] = df['전력(W)'].astype(float)
        
        return df
    except Exception as e:
        print(f"파일 읽기 오류: {e}")
        return None

def plot_power_graph(df, output_file=None, title=None):
    """전력 그래프를 생성합니다."""
    plt.figure(figsize=(15, 8))
    
    # 그래프 그리기
    plt.plot(df['시간'], df['전력(W)'], linewidth=1, alpha=0.8)
    
    # 그래프 스타일 설정
    plt.grid(True, alpha=0.3)
    plt.xlabel('시간', fontsize=12)
    plt.ylabel('전력 (W)', fontsize=12)
    
    if title:
        plt.title(title, fontsize=14, fontweight='bold')
    else:
        plt.title('GPU 전력 소모량 시간별 변화', fontsize=14, fontweight='bold')
    
    # x축 시간 포맷 설정
    plt.gca().xaxis.set_major_formatter(mdates.DateFormatter('%H:%M'))
    plt.gca().xaxis.set_major_locator(mdates.MinuteLocator(interval=5))
    plt.xticks(rotation=45)
    
    # 통계 정보 추가
    avg_power = df['전력(W)'].mean()
    max_power = df['전력(W)'].max()
    min_power = df['전력(W)'].min()
    
    stats_text = f'평균: {avg_power:.1f}W\n최대: {max_power:.1f}W\n최소: {min_power:.1f}W'
    plt.text(0.02, 0.98, stats_text, transform=plt.gca().transAxes, 
             verticalalignment='top', bbox=dict(boxstyle='round', facecolor='wheat', alpha=0.8))
    
    plt.tight_layout()
    
    if output_file:
        plt.savefig(output_file, dpi=300, bbox_inches='tight')
        print(f"그래프가 {output_file}에 저장되었습니다.")
    
    plt.show()

def main():
    parser = argparse.ArgumentParser(description='로그 파일에서 전력 그래프 생성')
    parser.add_argument('log_file', help='전력 로그 파일 경로')
    parser.add_argument('-o', '--output', help='저장할 이미지 파일명 (예: power_graph.png)')
    parser.add_argument('-t', '--title', help='그래프 제목')
    
    args = parser.parse_args()
    
    # 파일 존재 확인
    if not os.path.exists(args.log_file):
        print(f"오류: 파일 '{args.log_file}'이 존재하지 않습니다.")
        return
    
    # 데이터 로드
    print(f"로그 파일을 읽는 중: {args.log_file}")
    df = load_power_data(args.log_file)
    
    if df is None:
        print("데이터를 로드할 수 없습니다.")
        return
    
    print(f"총 {len(df)}개의 데이터 포인트를 로드했습니다.")
    print(f"시간 범위: {df['시간'].min()} ~ {df['시간'].max()}")
    print(f"전력 범위: {df['전력(W)'].min():.1f}W ~ {df['전력(W)'].max():.1f}W")
    print(f"평균 전력: {df['전력(W)'].mean():.1f}W")
    
    # 그래프 생성
    plot_power_graph(df, args.output, args.title)

if __name__ == "__main__":
    main() 