#!/usr/bin/env python3
"""
로그 파일에서 "Running:" 값의 평균을 계산하는 스크립트
"""

import re
import argparse
import os
from statistics import mean, median, stdev

def parse_running_values(log_file):
    """로그 파일에서 Running 값을 파싱합니다."""
    running_values = []
    
    try:
        with open(log_file, 'r', encoding='utf-8') as f:
            for line_num, line in enumerate(f, 1):
                # Running: X reqs 패턴 매칭
                running_match = re.search(r'Running: (\d+) reqs', line)
                if running_match:
                    value = int(running_match.group(1))
                    running_values.append({
                        'line': line_num,
                        'value': value,
                        'text': line.strip()
                    })
                    
    except Exception as e:
        print(f"파일 읽기 오류: {e}")
        return None
    
    return running_values

def calculate_stats(values):
    """통계를 계산합니다."""
    if not values:
        return None
    
    value_list = [item['value'] for item in values]
    
    stats = {
        'count': len(values),
        'total': sum(value_list),
        'average': mean(value_list),
        'median': median(value_list),
        'min': min(value_list),
        'max': max(value_list),
        'stdev': stdev(value_list) if len(value_list) > 1 else 0
    }
    
    return stats

def print_detailed_results(running_data, show_details=False):
    """결과를 상세히 출력합니다."""
    print("=== Running 값 분석 결과 ===\n")
    
    if not running_data:
        print("Running 값 데이터가 없습니다.")
        return
    
    stats = calculate_stats(running_data)
    
    print(f"📊 Running 값 통계:")
    print(f"  - 총 데이터 포인트: {stats['count']}개")
    print(f"  - 평균: {stats['average']:.2f} reqs")
    print(f"  - 중앙값: {stats['median']:.2f} reqs")
    print(f"  - 최소값: {stats['min']} reqs")
    print(f"  - 최대값: {stats['max']} reqs")
    print(f"  - 표준편차: {stats['stdev']:.2f}")
    print(f"  - 총합: {stats['total']} reqs")
    print()
    
    # 값별 분포 계산
    value_counts = {}
    for item in running_data:
        value = item['value']
        value_counts[value] = value_counts.get(value, 0) + 1
    
    print("📈 값별 분포:")
    for value in sorted(value_counts.keys()):
        count = value_counts[value]
        percentage = (count / stats['count']) * 100
        print(f"  - {value} reqs: {count}회 ({percentage:.1f}%)")
    print()
    
    if show_details:
        print("상세 Running 기록:")
        for i, item in enumerate(running_data, 1):
            print(f"  {i:3d}. 라인 {item['line']:4d}: {item['value']:2d} reqs")
        print()

def main():
    parser = argparse.ArgumentParser(description='로그 파일에서 Running 값 평균 계산')
    parser.add_argument('log_file', help='벤치마크 로그 파일 경로')
    parser.add_argument('-d', '--details', action='store_true', help='상세 기록 표시')
    
    args = parser.parse_args()
    
    # 파일 존재 확인
    if not os.path.exists(args.log_file):
        print(f"오류: 파일 '{args.log_file}'이 존재하지 않습니다.")
        return
    
    print(f"로그 파일 분석 중: {args.log_file}")
    
    # Running 값 파싱
    running_data = parse_running_values(args.log_file)
    
    if running_data is None:
        print("Running 값 데이터를 파싱할 수 없습니다.")
        return
    
    # 결과 출력
    print_detailed_results(running_data, args.details)

if __name__ == "__main__":
    main() 