#pragma once

#include <torch/all.h>

#include <map>
#include <vector>

// For flash cache with GDS
#include <cuda_runtime.h>
#include <cuda.h>
#include <cufile.h>

void swap_blocks(torch::Tensor& src, torch::Tensor& dst,
                 const torch::Tensor& block_mapping);

// Note: the key_caches and value_caches vectors are constant but
// not the Tensors they contain. The vectors need to be const refs
// in order to satisfy pytorch's C++ operator registration code.
void copy_blocks(std::vector<torch::Tensor> const& key_caches,
                 std::vector<torch::Tensor> const& value_caches,
                 const torch::Tensor& block_mapping);

void reshape_and_cache(torch::Tensor& key, torch::Tensor& value,
                       torch::Tensor& key_cache, torch::Tensor& value_cache,
                       torch::Tensor& slot_mapping,
                       const std::string& kv_cache_dtype, const double k_scale,
                       const double v_scale);

void reshape_and_cache_flash(torch::Tensor& key, torch::Tensor& value,
                             torch::Tensor& key_cache,
                             torch::Tensor& value_cache,
                             torch::Tensor& slot_mapping,
                             const std::string& kv_cache_dtype,
                             const double k_scale, const double v_scale);

// Just for unittest
void convert_fp8(torch::Tensor& dst_cache, torch::Tensor& src_cache,
                 const double scale, const std::string& kv_cache_dtype);

// Initialize GDS for flash cache
int64_t init_gds_for_flash_cache(const std::string& file_path, torch::Tensor& dummy_tensor);

// Reset GDS for flash cache
void reset_gds_for_flash_cache(int64_t file_handle, torch::Tensor& dummy_tensor);

// Swap flash blocks between GPU and flash memory
void swap_flash_blocks(int64_t file_handle,
                        torch::Tensor& gpu_kv_cache,
                        const torch::Tensor& block_mapping,                       
                        const torch::Tensor& tuning_params,
                        bool is_gpu_to_flash);

// Register/deregister GPU cache buffer
void register_gpu_cache_buffer(torch::Tensor& gpu_kv_cache);
void deregister_gpu_cache_buffer(torch::Tensor& gpu_kv_cache);