#!/usr/bin/env python3
"""
로그 파일에서 특정 시간 구간의 전력 통계를 계산하는 스크립트
"""

import pandas as pd
import numpy as np
from datetime import datetime, timedelta
import argparse
import os

def load_power_data(log_file):
    """로그 파일에서 전력 데이터를 로드합니다."""
    try:
        # CSV 파일로 읽기
        df = pd.read_csv(log_file)
        
        # 시간 컬럼을 datetime으로 변환
        df['시간'] = pd.to_datetime(df['시간'])
        
        # 전력(W) 컬럼을 float로 변환
        df['전력(W)'] = df['전력(W)'].astype(float)
        
        return df
    except Exception as e:
        print(f"파일 읽기 오류: {e}")
        return None

def calculate_power_stats(df, start_seconds=0, end_seconds=1700):
    """지정된 시간 구간의 전력 통계를 계산합니다."""
    if df.empty:
        return None
    
    # 시작 시간과 종료 시간 계산
    start_time = df['시간'].iloc[0] + timedelta(seconds=start_seconds)
    end_time = df['시간'].iloc[0] + timedelta(seconds=end_seconds)
    
    # 지정된 시간 구간의 데이터 필터링
    mask = (df['시간'] >= start_time) & (df['시간'] <= end_time)
    filtered_df = df[mask]
    
    if filtered_df.empty:
        print(f"경고: {start_seconds}~{end_seconds}초 구간에 데이터가 없습니다.")
        return None
    
    # 통계 계산
    stats = {
        '최소 전력 (W)': filtered_df['전력(W)'].min(),
        '최대 전력 (W)': filtered_df['전력(W)'].max(),
        '평균 전력 (W)': filtered_df['전력(W)'].mean(),
        '표준편차 (W)': filtered_df['전력(W)'].std(),
        '중앙값 (W)': filtered_df['전력(W)'].median(),
        '데이터 포인트 수': len(filtered_df),
        '시작 시간': filtered_df['시간'].iloc[0],
        '종료 시간': filtered_df['시간'].iloc[-1],
        '실제 시간 구간 (초)': (filtered_df['시간'].iloc[-1] - filtered_df['시간'].iloc[0]).total_seconds()
    }
    
    return stats

def print_stats(stats, start_seconds, end_seconds):
    """통계 결과를 출력합니다."""
    print(f"\n=== {start_seconds}~{end_seconds}초 구간 전력 통계 ===")
    print(f"시간 구간: {stats['시작 시간']} ~ {stats['종료 시간']}")
    print(f"실제 측정 시간: {stats['실제 시간 구간 (초)']:.1f}초")
    print(f"데이터 포인트 수: {stats['데이터 포인트 수']}개")
    print("-" * 50)
    print(f"최소 전력: {stats['최소 전력 (W)']:.3f} W")
    print(f"최대 전력: {stats['최대 전력 (W)']:.3f} W")
    print(f"평균 전력: {stats['평균 전력 (W)']:.3f} W")
    print(f"중앙값: {stats['중앙값 (W)']:.3f} W")
    print(f"표준편차: {stats['표준편차 (W)']:.3f} W")
    print(f"전력 범위: {stats['최대 전력 (W)'] - stats['최소 전력 (W)']:.3f} W")

def main():
    parser = argparse.ArgumentParser(description='로그 파일에서 전력 통계 계산')
    parser.add_argument('log_file', help='전력 로그 파일 경로')
    parser.add_argument('-s', '--start', type=int, default=0, help='시작 시간 (초, 기본값: 0)')
    parser.add_argument('-e', '--end', type=int, default=1700, help='종료 시간 (초, 기본값: 1700)')
    
    args = parser.parse_args()
    
    # 파일 존재 확인
    if not os.path.exists(args.log_file):
        print(f"오류: 파일 '{args.log_file}'이 존재하지 않습니다.")
        return
    
    # 데이터 로드
    print(f"로그 파일을 읽는 중: {args.log_file}")
    df = load_power_data(args.log_file)
    
    if df is None:
        print("데이터를 로드할 수 없습니다.")
        return
    
    print(f"총 {len(df)}개의 데이터 포인트를 로드했습니다.")
    print(f"전체 시간 범위: {df['시간'].min()} ~ {df['시간'].max()}")
    
    # 통계 계산
    stats = calculate_power_stats(df, args.start, args.end)
    
    if stats:
        print_stats(stats, args.start, args.end)
    else:
        print("통계를 계산할 수 없습니다.")

if __name__ == "__main__":
    main() 