import subprocess
import os
import sys
from datetime import datetime

def analyze_core_dump(executable_path, core_dump_path):
    """
    Core dump 파일을 분석하는 함수
    
    Args:
        executable_path (str): 실행 파일 경로
        core_dump_path (str): core dump 파일 경로
    """
    try:
        # GDB 명령어 구성
        gdb_commands = [
            "gdb",
            "-ex", "set pagination off",
            "-ex", "bt full",  # 전체 스택 트레이스
            "-ex", "info threads",  # 모든 스레드 정보
            "-ex", "info locals",  # 지역 변수 정보
            "-ex", "info registers",  # 레지스터 정보
            "-ex", "quit",
            executable_path,
            core_dump_path
        ]
        
        # GDB 실행 및 결과 캡처
        result = subprocess.run(gdb_commands, 
                              capture_output=True, 
                              text=True)
        
        # 결과 출력
        print("\n=== Core Dump Analysis Report ===")
        print(f"Analysis Time: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
        print(f"Executable: {executable_path}")
        print(f"Core Dump: {core_dump_path}")
        print("\n=== Stack Trace ===")
        print(result.stdout)
        
        if result.stderr:
            print("\n=== Error Messages ===")
            print(result.stderr)
            
    except Exception as e:
        print(f"Error analyzing core dump: {str(e)}")

def find_latest_core_dump(core_dir="/var/crash"):
    """
    가장 최근의 core dump 파일을 찾는 함수
    
    Args:
        core_dir (str): core dump 파일이 저장된 디렉토리
    """
    try:
        core_files = [f for f in os.listdir(core_dir) if f.startswith("core.")]
        if not core_files:
            print(f"No core dump files found in {core_dir}")
            return None
            
        latest_core = max(core_files, key=lambda x: os.pathgetctime(os.path.join(core_dir, x)))
        return os.path.join(core_dir, latest_core)
        
    except Exception as e:
        print(f"Error finding core dump: {str(e)}")
        return None

def main():
    if len(sys.argv) < 2:
        print("Usage: python analyze_core.py <executable_path> [core_dump_path]")
        sys.exit(1)
        
    executable_path = sys.argv[1]
    core_dump_path = sys.argv[2] if len(sys.argv) > 2 else find_latest_core_dump()
    
    if not core_dump_path:
        print("No core dump file found")
        sys.exit(1)
        
    if not os.path.exists(executable_path):
        print(f"Executable not found: {executable_path}")
        sys.exit(1)
        
    if not os.path.exists(core_dump_path):
        print(f"Core dump file not found: {core_dump_path}")
        sys.exit(1)
        
    analyze_core_dump(executable_path, core_dump_path)

if __name__ == "__main__":
    main()
