# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# timm: https://github.com/rwightman/pytorch-image-models/tree/master/timm
# DeiT: https://github.com/facebookresearch/deit
# --------------------------------------------------------
from functools import partial

import numpy as np
import timm.models.vision_transformer
import torch
import torch.nn as nn
import torch.nn.functional as F
from detectron2.modeling import BACKBONE_REGISTRY, Backbone, ShapeSpec


# Full list of arguments
# img_size=224, patch_size=16, in_chans=3, num_classes=1000, global_pool='token',
# embed_dim=768, depth=12, num_heads=12, mlp_ratio=4., qkv_bias=True, init_values=None,
# class_token=True, fc_norm=None, drop_rate=0., attn_drop_rate=0., drop_path_rate=0., weight_init='',
# embed_layer=PatchEmbed, norm_layer=None, act_layer=None, block_fn=Block)
class VisionTransformer(timm.models.vision_transformer.VisionTransformer):
    """ Vision Transformer with support for global average pooling
    """
    def __init__(self, global_pool=False, **kwargs):
        super(VisionTransformer, self).__init__(**kwargs)

        self.global_pool = global_pool
        if self.global_pool:
            norm_layer = kwargs['norm_layer']
            embed_dim = kwargs['embed_dim']
            self.fc_norm = norm_layer(embed_dim)

            del self.norm  # remove the original norm

    def forward(self, x):
        B = x.shape[0]
        x = self.patch_embed(x)

        cls_tokens = self.cls_token.expand(B, -1, -1)  # stole cls_tokens impl from Phil Wang, thanks
        x = torch.cat((cls_tokens, x), dim=1)
        x = x + self.pos_embed
        x = self.pos_drop(x)

        for blk in self.blocks:
            x = blk(x)

        if self.global_pool:
            x = x[:, 1:, :].mean(dim=1)  # global pool without cls token
            outcome = self.fc_norm(x)
        else:
            x = self.norm(x)
            # outcome = x[:, 0]
            outcome = x[:, 1:, :]

        return outcome


@BACKBONE_REGISTRY.register()
class D2VisionTransformer(VisionTransformer, Backbone):
    def __init__(self, cfg, input_shape):

        # pretrain_img_size = cfg.MODEL.VIT.PRETRAIN_IMG_SIZE
        img_size = cfg.MODEL.VIT.IMG_SIZE
        patch_size = cfg.MODEL.VIT.PATCH_SIZE
        embed_dim = cfg.MODEL.VIT.EMBED_DIM
        depth = cfg.MODEL.VIT.DEPTH
        num_heads = cfg.MODEL.VIT.NUM_HEADS
        mlp_ratio = cfg.MODEL.VIT.MLP_RATIO
        qkv_bias = cfg.MODEL.VIT.QKV_BIAS
        drop_rate = cfg.MODEL.VIT.DROP_RATE
        attn_drop_rate = cfg.MODEL.VIT.ATTN_DROP_RATE
        drop_path_rate = cfg.MODEL.VIT.DROP_PATH_RATE
        norm_layer = partial(nn.LayerNorm, eps=1e-6)

        super().__init__(
            img_size=img_size,
            patch_size=patch_size,
            embed_dim=embed_dim,
            depth=depth,
            num_heads=num_heads,
            mlp_ratio=mlp_ratio,
            qkv_bias=qkv_bias,
            drop_rate=drop_rate,
            attn_drop_rate=attn_drop_rate,
            drop_path_rate=drop_path_rate,
            norm_layer=norm_layer,
        )

        # Gradient checkpointing, we don't use for now
        # use_checkpoint = cfg.MODEL.SWIN.USE_CHECKPOINT

        # import ipdb; ipdb.set_trace()

        self._out_features = cfg.MODEL.VIT.OUT_FEATURES
        self.patch_size = patch_size

        self._out_feature_strides = {
            "res": patch_size,
        }
        self._out_feature_channels = {
            "res": embed_dim,
        }

    def forward(self, x):
        """
        Args:
            x: Tensor of shape (N,C,H,W). H, W must be a multiple of ``self.size_divisibility``.
        Returns:
            dict[str->Tensor]: names and the corresponding features
        """
        assert (
            x.dim() == 4
        ), f"VisionTransformer takes an input of shape (N, C, H, W). Got {x.shape} instead!"
        outputs = {}

        import ipdb; ipdb.set_trace()

        _, _, H, W = x.shape
        y = super().forward(x)

        # Reshape and save
        B, _, C = y.shape
        y = y.permute(0, 2, 1).reshape(B, C, H, W)
        outputs["res"] = y

        return outputs
    

    def output_shape(self):
        return {
            name: ShapeSpec(
                channels=self._out_feature_channels[name], stride=self._out_feature_strides[name]
            )
            for name in self._out_features
        }

    @property
    def size_divisibility(self):
        return 32



# def vit_base_patch16(**kwargs):
#     model = VisionTransformer(
#         patch_size=16, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, qkv_bias=True,
#         norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
#     return model


# def vit_large_patch16(**kwargs):
#     model = VisionTransformer(
#         patch_size=16, embed_dim=1024, depth=24, num_heads=16, mlp_ratio=4, qkv_bias=True,
#         norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
#     return model


# def vit_huge_patch14(**kwargs):
#     model = VisionTransformer(
#         patch_size=14, embed_dim=1280, depth=32, num_heads=16, mlp_ratio=4, qkv_bias=True,
#         norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
#     return model
