# Copyright (c) Facebook, Inc. and its affiliates.
import json
import os

from detectron2.data import DatasetCatalog, MetadataCatalog
from detectron2.data.datasets import load_sem_seg
from detectron2.utils.file_io import PathManager

COCO_CATEGORIES = [
    {"color": [220, 20, 60], "isthing": 1, "id": 0, "name": "person"},
    {"color": [119, 11, 32], "isthing": 1, "id": 1, "name": "bicycle"},
    {"color": [0, 0, 142], "isthing": 1, "id": 2, "name": "car"},
    {"color": [0, 0, 230], "isthing": 1, "id": 3, "name": "motorcycle"},
    {"color": [106, 0, 228], "isthing": 1, "id": 4, "name": "airplane"},
    {"color": [0, 60, 100], "isthing": 1, "id": 5, "name": "bus"},
    {"color": [0, 80, 100], "isthing": 1, "id": 6, "name": "train"},
    {"color": [0, 0, 70], "isthing": 1, "id": 7, "name": "truck"},
    {"color": [0, 0, 192], "isthing": 1, "id": 8, "name": "boat"},
    {"color": [250, 170, 30], "isthing": 1, "id": 9, "name": "traffic light"},
    {"color": [100, 170, 30], "isthing": 1, "id": 10, "name": "fire hydrant"},
    {"color": [220, 220, 0], "isthing": 1, "id": 11, "name": "stop sign"},
    {"color": [175, 116, 175], "isthing": 1, "id": 12, "name": "parking meter"},
    {"color": [250, 0, 30], "isthing": 1, "id": 13, "name": "bench"},
    {"color": [165, 42, 42], "isthing": 1, "id": 14, "name": "bird"},
    {"color": [255, 77, 255], "isthing": 1, "id": 15, "name": "cat"},
    {"color": [0, 226, 252], "isthing": 1, "id": 16, "name": "dog"},
    {"color": [182, 182, 255], "isthing": 1, "id": 17, "name": "horse"},
    {"color": [0, 82, 0], "isthing": 1, "id": 18, "name": "sheep"},
    {"color": [120, 166, 157], "isthing": 1, "id": 19, "name": "cow"},
    {"color": [110, 76, 0], "isthing": 1, "id": 20, "name": "elephant"},
    {"color": [174, 57, 255], "isthing": 1, "id": 21, "name": "bear"},
    {"color": [199, 100, 0], "isthing": 1, "id": 22, "name": "zebra"},
    {"color": [72, 0, 118], "isthing": 1, "id": 23, "name": "giraffe"},
    {"color": [255, 179, 240], "isthing": 1, "id": 24, "name": "backpack"},
    {"color": [0, 125, 92], "isthing": 1, "id": 25, "name": "umbrella"},
    {"color": [209, 0, 151], "isthing": 1, "id": 26, "name": "handbag"},
    {"color": [188, 208, 182], "isthing": 1, "id": 27, "name": "tie"},
    {"color": [0, 220, 176], "isthing": 1, "id": 28, "name": "suitcase"},
    {"color": [255, 99, 164], "isthing": 1, "id": 29, "name": "frisbee"},
    {"color": [92, 0, 73], "isthing": 1, "id": 30, "name": "skis"},
    {"color": [133, 129, 255], "isthing": 1, "id": 31, "name": "snowboard"},
    {"color": [78, 180, 255], "isthing": 1, "id": 32, "name": "sports ball"},
    {"color": [0, 228, 0], "isthing": 1, "id": 33, "name": "kite"},
    {"color": [174, 255, 243], "isthing": 1, "id": 34, "name": "baseball bat"},
    {"color": [45, 89, 255], "isthing": 1, "id": 35, "name": "baseball glove"},
    {"color": [134, 134, 103], "isthing": 1, "id": 36, "name": "skateboard"},
    {"color": [145, 148, 174], "isthing": 1, "id": 37, "name": "surfboard"},
    {"color": [255, 208, 186], "isthing": 1, "id": 38, "name": "tennis racket"},
    {"color": [197, 226, 255], "isthing": 1, "id": 39, "name": "bottle"},
    {"color": [171, 134, 1], "isthing": 1, "id": 40, "name": "wine glass"},
    {"color": [109, 63, 54], "isthing": 1, "id": 41, "name": "cup"},
    {"color": [207, 138, 255], "isthing": 1, "id": 42, "name": "fork"},
    {"color": [151, 0, 95], "isthing": 1, "id": 43, "name": "knife"},
    {"color": [9, 80, 61], "isthing": 1, "id": 44, "name": "spoon"},
    {"color": [84, 105, 51], "isthing": 1, "id": 45, "name": "bowl"},
    {"color": [74, 65, 105], "isthing": 1, "id": 46, "name": "banana"},
    {"color": [166, 196, 102], "isthing": 1, "id": 47, "name": "apple"},
    {"color": [208, 195, 210], "isthing": 1, "id": 48, "name": "sandwich"},
    {"color": [255, 109, 65], "isthing": 1, "id": 49, "name": "orange"},
    {"color": [0, 143, 149], "isthing": 1, "id": 50, "name": "broccoli"},
    {"color": [179, 0, 194], "isthing": 1, "id": 51, "name": "carrot"},
    {"color": [209, 99, 106], "isthing": 1, "id": 52, "name": "hot dog"},
    {"color": [5, 121, 0], "isthing": 1, "id": 53, "name": "pizza"},
    {"color": [227, 255, 205], "isthing": 1, "id": 54, "name": "donut"},
    {"color": [147, 186, 208], "isthing": 1, "id": 55, "name": "cake"},
    {"color": [153, 69, 1], "isthing": 1, "id": 56, "name": "chair"},
    {"color": [3, 95, 161], "isthing": 1, "id": 57, "name": "couch"},
    {"color": [163, 255, 0], "isthing": 1, "id": 58, "name": "potted plant"},
    {"color": [119, 0, 170], "isthing": 1, "id": 59, "name": "bed"},
    {"color": [0, 182, 199], "isthing": 1, "id": 60, "name": "dining table"},
    {"color": [0, 165, 120], "isthing": 1, "id": 61, "name": "toilet"},
    {"color": [183, 130, 88], "isthing": 1, "id": 62, "name": "tv"},
    {"color": [95, 32, 0], "isthing": 1, "id": 63, "name": "laptop"},
    {"color": [130, 114, 135], "isthing": 1, "id": 64, "name": "mouse"},
    {"color": [110, 129, 133], "isthing": 1, "id": 65, "name": "remote"},
    {"color": [166, 74, 118], "isthing": 1, "id": 66, "name": "keyboard"},
    {"color": [219, 142, 185], "isthing": 1, "id": 67, "name": "cell phone"},
    {"color": [79, 210, 114], "isthing": 1, "id": 68, "name": "microwave"},
    {"color": [178, 90, 62], "isthing": 1, "id": 69, "name": "oven"},
    {"color": [65, 70, 15], "isthing": 1, "id": 70, "name": "toaster"},
    {"color": [127, 167, 115], "isthing": 1, "id": 71, "name": "sink"},
    {"color": [59, 105, 106], "isthing": 1, "id": 72, "name": "refrigerator"},
    {"color": [142, 108, 45], "isthing": 1, "id": 73, "name": "book"},
    {"color": [196, 172, 0], "isthing": 1, "id": 74, "name": "clock"},
    {"color": [95, 54, 80], "isthing": 1, "id": 75, "name": "vase"},
    {"color": [128, 76, 255], "isthing": 1, "id": 76, "name": "scissors"},
    {"color": [201, 57, 1], "isthing": 1, "id": 77, "name": "teddy bear"},
    {"color": [246, 0, 122], "isthing": 1, "id": 78, "name": "hair drier"},
    {"color": [191, 162, 208], "isthing": 1, "id": 79, "name": "toothbrush"},
    {"color": [255, 255, 128], "isthing": 0, "id": 80, "name": "banner"},
    {"color": [147, 211, 203], "isthing": 0, "id": 81, "name": "blanket"},
    {"color": [150, 100, 100], "isthing": 0, "id": 82, "name": "bridge"},
    {"color": [168, 171, 172], "isthing": 0, "id": 83, "name": "cardboard"},
    {"color": [146, 112, 198], "isthing": 0, "id": 84, "name": "counter"},
    {"color": [210, 170, 100], "isthing": 0, "id": 85, "name": "curtain"},
    {"color": [92, 136, 89], "isthing": 0, "id": 86, "name": "door-stuff"},
    {"color": [218, 88, 184], "isthing": 0, "id": 87, "name": "floor-wood"},
    {"color": [241, 129, 0], "isthing": 0, "id": 88, "name": "flower"},
    {"color": [217, 17, 255], "isthing": 0, "id": 89, "name": "fruit"},
    {"color": [124, 74, 181], "isthing": 0, "id": 90, "name": "gravel"},
    {"color": [70, 70, 70], "isthing": 0, "id": 91, "name": "house"},
    {"color": [255, 228, 255], "isthing": 0, "id": 92, "name": "light"},
    {"color": [154, 208, 0], "isthing": 0, "id": 93, "name": "mirror-stuff"},
    {"color": [193, 0, 92], "isthing": 0, "id": 94, "name": "net"},
    {"color": [76, 91, 113], "isthing": 0, "id": 95, "name": "pillow"},
    {"color": [255, 180, 195], "isthing": 0, "id": 96, "name": "platform"},
    {"color": [106, 154, 176], "isthing": 0, "id": 97, "name": "playingfield"},
    {"color": [230, 150, 140], "isthing": 0, "id": 98, "name": "railroad"},
    {"color": [60, 143, 255], "isthing": 0, "id": 99, "name": "river"},
    {"color": [128, 64, 128], "isthing": 0, "id": 100, "name": "road"},
    {"color": [92, 82, 55], "isthing": 0, "id": 101, "name": "roof"},
    {"color": [254, 212, 124], "isthing": 0, "id": 102, "name": "sand"},
    {"color": [73, 77, 174], "isthing": 0, "id": 103, "name": "sea"},
    {"color": [255, 160, 98], "isthing": 0, "id": 104, "name": "shelf"},
    {"color": [255, 255, 255], "isthing": 0, "id": 105, "name": "snow"},
    {"color": [104, 84, 109], "isthing": 0, "id": 106, "name": "stairs"},
    {"color": [169, 164, 131], "isthing": 0, "id": 107, "name": "tent"},
    {"color": [225, 199, 255], "isthing": 0, "id": 108, "name": "towel"},
    {"color": [137, 54, 74], "isthing": 0, "id": 109, "name": "wall-brick"},
    {"color": [135, 158, 223], "isthing": 0, "id": 110, "name": "wall-stone"},
    {"color": [7, 246, 231], "isthing": 0, "id": 111, "name": "wall-tile"},
    {"color": [107, 255, 200], "isthing": 0, "id": 112, "name": "wall-wood"},
    {"color": [58, 41, 149], "isthing": 0, "id": 113, "name": "water-other"},
    {"color": [183, 121, 142], "isthing": 0, "id": 114, "name": "window-blind"},
    {"color": [255, 73, 97], "isthing": 0, "id": 115, "name": "window-other"},
    {"color": [107, 142, 35], "isthing": 0, "id": 116, "name": "tree-merged"},
    {"color": [190, 153, 153], "isthing": 0, "id": 117, "name": "fence-merged"},
    {"color": [146, 139, 141], "isthing": 0, "id": 118, "name": "ceiling-merged"},
    {"color": [70, 130, 180], "isthing": 0, "id": 119, "name": "sky-other-merged"},
    {"color": [134, 199, 156], "isthing": 0, "id": 120, "name": "cabinet-merged"},
    {"color": [209, 226, 140], "isthing": 0, "id": 121, "name": "table-merged"},
    {"color": [96, 36, 108], "isthing": 0, "id": 122, "name": "floor-other-merged"},
    {"color": [96, 96, 96], "isthing": 0, "id": 123, "name": "pavement-merged"},
    {"color": [64, 170, 64], "isthing": 0, "id": 124, "name": "mountain-merged"},
    {"color": [152, 251, 152], "isthing": 0, "id": 125, "name": "grass-merged"},
    {"color": [208, 229, 228], "isthing": 0, "id": 126, "name": "dirt-merged"},
    {"color": [206, 186, 171], "isthing": 0, "id": 127, "name": "paper-merged"},
    {"color": [152, 161, 64], "isthing": 0, "id": 128, "name": "food-other-merged"},
    {"color": [116, 112, 0], "isthing": 0, "id": 129, "name": "building-other-merged"},
    {"color": [0, 114, 143], "isthing": 0, "id": 130, "name": "rock-merged"},
    {"color": [102, 102, 156], "isthing": 0, "id": 131, "name": "wall-other-merged"},
    {"color": [250, 141, 255], "isthing": 0, "id": 132, "name": "rug-merged"},
]



def _get_custom_meta():
    stuff_ids = [k["id"] for k in COCO_CATEGORIES]

    # For semantic segmentation, this mapping maps from contiguous stuff id
    # (in [0, 91], used in models) to ids in the dataset (used for processing results)
    stuff_dataset_id_to_contiguous_id = {k: i for i, k in enumerate(stuff_ids)}
    stuff_classes = [k["name"] for k in COCO_CATEGORIES]

    ret = {
        "stuff_dataset_id_to_contiguous_id": stuff_dataset_id_to_contiguous_id,
        "stuff_classes": stuff_classes,
    }
    return ret



def _get_custom_files(image_dir, gt_dir, img_idx, it):
    img_root = format(img_idx, "06d")

    files = []
    # scan through the directory
        
    label_file = os.path.join(image_dir, img_root + "_" + str(it) + ".png")

    image_file = os.path.join(image_dir, img_root + '.png')
    json_file = os.path.join(image_dir, img_root + '.json')

    files.append((image_file, label_file, json_file))
    assert len(files), "No images found in {}".format(image_dir)
    for f in files[0]:
        assert PathManager.isfile(f), f
    return files


def load_custom_semantic(image_dir, gt_dir, img_idx, it):
    """
    Args:
        image_dir (str): path to the raw dataset. e.g., "~/cityscapes/leftImg8bit/train".
        gt_dir (str): path to the raw annotations. e.g., "~/cityscapes/gtFine/train".
    Returns:
        list[dict]: a list of dict, each has "file_name" and
            "sem_seg_file_name".
    """
    ret = []
    # gt_dir is small and contain many small files. make sense to fetch to local first
    gt_dir = PathManager.get_local_path(gt_dir)
    for image_file, label_file, json_file in _get_custom_files(image_dir, gt_dir, img_idx, it):
        with PathManager.open(json_file, "r") as f:
            jsonobj = json.load(f)
        ret.append(
            {
                "file_name": image_file,
                "sem_seg_file_name": label_file,
                "height": jsonobj["imgHeight"],
                "width": jsonobj["imgWidth"],
            }
        )
    assert len(ret), f"No images found in {image_dir}!"
    # assert PathManager.isfile(
    #     ret[0]["sem_seg_file_name"]
    # ), "Please generate labelTrainIds.png with cityscapesscripts/preparation/createTrainIdLabelImgs.py"  # noqa

    return ret


def _get_custom_val_files(image_dir, gt_dir, img_idx):
    # import ipdb; ipdb.set_trace()

    img_root = format(img_idx, "06d")

    files = []

    label_file = os.path.join(gt_dir, img_root + "_sem.png")
    image_file = os.path.join(image_dir, img_root + '.png')
    json_file = os.path.join(gt_dir, img_root + '.json')

    files.append((image_file, label_file, json_file))
    assert len(files), "No images found in {}".format(image_dir)
    for f in files[0]:
        assert PathManager.isfile(f), f
    return files


def load_custom_video_eval(image_dir, gt_dir, img_idx):
    """
    Args:
        image_dir (str): path to the raw dataset. e.g., "~/cityscapes/leftImg8bit/train".
        gt_dir (str): path to the raw annotations. e.g., "~/cityscapes/gtFine/train".
    Returns:
        list[dict]: a list of dict, each has "file_name" and
            "sem_seg_file_name".
    """
    ret = []
    # gt_dir is small and contain many small files. make sense to fetch to local first
    gt_dir = PathManager.get_local_path(gt_dir)
    for image_file, label_file, json_file in _get_custom_val_files(image_dir, gt_dir, img_idx):
        with PathManager.open(json_file, "r") as f:
            jsonobj = json.load(f)
        ret.append(
            {
                "file_name": image_file,
                "sem_seg_file_name": label_file,
                "height": jsonobj["imgHeight"],
                "width": jsonobj["imgWidth"],
            }
        )
    assert len(ret), f"No images found in {image_dir}!"
    # assert PathManager.isfile(
    #     ret[0]["sem_seg_file_name"]
    # ), "Please generate labelTrainIds.png with cityscapesscripts/preparation/createTrainIdLabelImgs.py"  # noqa

    return ret    



def register_custom_ttt_video_sem(root, img_idx, it):
    # import ipdb; ipdb.set_trace()

    root = os.path.join(root, "ttt_custom", "dropout_video", "0.1")
    meta = _get_custom_meta()

    # Register train set
    image_dir = os.path.join(root, "train", "berkeley")
    gt_dir = os.path.join(root, "val", "berkeley")
    name = f"custom_ttt_video_sem_train"
    DatasetCatalog.register(
        name, lambda x=image_dir, y=gt_dir, it=it: load_custom_semantic(x, y, img_idx, it)
    )
    MetadataCatalog.get(name).set(
        stuff_classes=meta["stuff_classes"][:],
        image_root=image_dir,
        sem_seg_root=gt_dir,
        evaluator_type="sem_seg",
        ignore_label=255,  # NOTE: gt is saved in 16-bit TIFF images
    )
    
    # Register val set
    image_dir = os.path.join(root, "train", "berkeley")
    gt_dir = os.path.join(root, "val", "berkeley")
    name = f"custom_ttt_video_sem_val"
    DatasetCatalog.register(
        name, lambda x=image_dir, y=gt_dir, it=it: load_custom_video_eval(x, y, img_idx)
    )
    MetadataCatalog.get(name).set(
        stuff_classes=meta["stuff_classes"][:],
        image_root=image_dir,
        sem_seg_root=gt_dir,
        evaluator_type="sem_seg",
        ignore_label=255,  # NOTE: gt is saved in 16-bit TIFF images
    )


# import ipdb; ipdb.set_trace()

# We register the Berkeley dataset by default
# Override at config or argument level

_root = os.getenv("DETECTRON2_DATASETS")
register_custom_ttt_video_sem(_root, 0, 0)

