import argparse
import os
from functools import partial as bind

import cv2
import gymnasium as gym
import numpy as np
import ruamel.yaml as yaml

import dreamerv3
import embodied
from embodied.envs.from_gymnasium import FromGymnasium
from robosuite_envs.wrapper import RobosuiteEnv


class ResizeWrapper(gym.ObservationWrapper):
    def __init__(self, env, image_size):
        super().__init__(env)
        self._image_size = image_size
        assert len(env.observation_space.shape) == 3
        assert env.observation_space.shape[2] == 3
        assert env.observation_space.dtype == np.uint8
        self.observation_space = gym.spaces.Box(
            low=0, high=255, shape=(self._image_size, self._image_size, 3), dtype=np.uint8
        )

    def observation(self, observation):
        return cv2.resize(observation, dsize=(self._image_size, self._image_size), interpolation=cv2.INTER_CUBIC)


def make_robosuite(config, task, horizon, image_size, seed, initialization_noise_magnitude=None,
                   use_random_object_position=False):
    env = RobosuiteEnv(task, horizon, seed, initialization_noise_magnitude, use_random_object_position)
    env = ResizeWrapper(env, image_size)
    env = FromGymnasium(env, obs_key='image')
    env = dreamerv3.wrap_env(env, config)

    return env


def make_robosuite_lambda(config, task, horizon, image_size, seed=None, initialization_noise_magnitude=None,
                          use_random_object_position=False):
    return lambda: make_robosuite(config, task, horizon, image_size, seed, initialization_noise_magnitude,
                                  use_random_object_position)


def make_batch_env(task, horizon, seed, image_size, config, n_envs, initialization_noise_magnitude=None,
                   use_random_object_position=False, parallel=None):
    envs = []
    if parallel is None:
        parallel = config.envs.parallel

    for i in range(n_envs):
        env_fn = make_robosuite_lambda(config, task, horizon, image_size, seed=seed + i,
                                       initialization_noise_magnitude=initialization_noise_magnitude,
                                       use_random_object_position=use_random_object_position)
        if parallel != 'none':
            env_fn = bind(embodied.Parallel, env_fn, parallel)

        envs.append(env_fn)

    envs = [env_fn() for env_fn in envs]
    return embodied.BatchEnv(envs, parallel=(parallel != 'none'))


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--config_path', type=str, required=True)
    parser.add_argument('--task', type=str, required=True)
    parser.add_argument('--horizon', type=int, required=True)
    parser.add_argument('--image_size', type=int, required=True)
    parser.add_argument('--n_envs', default=1, type=int)
    parser.add_argument('--initialization_noise_magnitude', type=float, required=False)
    parser.add_argument('--use_random_object_position', choices=['large', 'medium', 'small'], required=False)
    parser.add_argument('--checkpoint_path', type=str, required=False)
    parser.add_argument('--param_prefixes', nargs='*', type=str, required=False)
    return parser.parse_args()


def main():
    args = parse_args()
    print('Args:', vars(args))
    checkpoint_path = args.checkpoint_path
    param_prefixes = args.param_prefixes
    configs = yaml.YAML(typ='safe').load(embodied.Path(args.config_path).read())
    config = embodied.Config(configs['defaults'])
    config = config.update(dreamerv3.configs['medium'])

    logdir = embodied.Path(config.logdir)
    step = embodied.Counter()
    outputs = [
        embodied.logger.TerminalOutput(),
        embodied.logger.JSONLOutput(logdir, 'metrics.jsonl'),
    ]
    if config.wandb.project is not None:
        os.makedirs(config.wandb.dir, exist_ok=True)
        outputs.append(embodied.logger.WandBOutput(config))

    logger = embodied.Logger(step, outputs)
    env = make_batch_env(args.task, args.horizon, image_size=args.image_size, config=config, seed=config['seed'],
                         n_envs=args.n_envs, initialization_noise_magnitude=args.initialization_noise_magnitude,
                         use_random_object_position=args.use_random_object_position)
    eval_env = make_batch_env(args.task, args.horizon, image_size=args.image_size, config=config,
                              seed=config['seed'] + args.n_envs, n_envs=1, parallel='none',
                              initialization_noise_magnitude=args.initialization_noise_magnitude,
                              use_random_object_position=args.use_random_object_position)

    agent = dreamerv3.Agent(env.obs_space, env.act_space, step, config)
    replay = embodied.replay.Uniform(
        config.batch_length, config.replay_size, logdir / 'replay')
    args = embodied.Config(
        **config.run, logdir=config.logdir,
        batch_steps=config.batch_size * config.batch_length)
    if checkpoint_path is None:
        checkpoint_path = os.path.join(config.wandb.project, config.wandb.name)
    os.makedirs(checkpoint_path, exist_ok=True)
    embodied.run.train_eval_no_replay(agent, train_env=env, eval_env=eval_env, train_replay=replay, logger=logger,
                                      args=args, checkpoint_path=checkpoint_path, param_prefixes=param_prefixes)


if __name__ == "__main__":
    main()
