import argparse
import os
import random
from functools import partial as bind

import numpy as np
import ruamel.yaml as yaml
from omegaconf import OmegaConf

import dreamerv3
import embodied
import cw_envs.synthetic_envs
from cw_envs.synthetic_envs.utils import GymToGymnasium
from embodied.envs.from_gymnasium import FromGymnasium


def make_env(config, env_config_path, seed):
    env_config = OmegaConf.load(env_config_path)
    random.seed(seed)
    np.random.seed(seed)
    env = GymToGymnasium(getattr(cw_envs.synthetic_envs, env_config.env)(env_config, seed))
    env.action_space.seed(seed)
    env = FromGymnasium(env, obs_key='image')  # Or obs_key='vector'.
    env = dreamerv3.wrap_env(env, config)

    return env


def make_batch_env(env_config_path, config, seed, n_envs, parallel=None):
    envs = []
    if parallel is None:
        parallel = config.envs.parallel

    def create_env_fn(env_seed=None):
        return lambda: make_env(config, env_config_path, seed=env_seed)

    for i in range(n_envs):
        env_fn = create_env_fn(seed + i)
        if parallel != 'none':
            env_fn = bind(embodied.Parallel, env_fn, parallel)

        envs.append(env_fn)

    envs = [env_fn() for env_fn in envs]
    return embodied.BatchEnv(envs, parallel=(parallel != 'none'))


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--config_path', type=str, required=True)
    parser.add_argument('--env_config_path', type=str, required=True)
    parser.add_argument('--n_envs', default=1, type=int)
    parser.add_argument('--checkpoint_path', type=str, required=False)
    parser.add_argument('--param_prefixes', nargs='*', type=str, required=False)
    return parser.parse_args()


def main():
    args = parse_args()
    print('Args:', vars(args))
    checkpoint_path = args.checkpoint_path
    param_prefixes = args.param_prefixes
    configs = yaml.YAML(typ='safe').load(embodied.Path(args.config_path).read())
    config = embodied.Config(configs['defaults'])

    logdir = embodied.Path(config.logdir)
    step = embodied.Counter()
    outputs = [
        embodied.logger.TerminalOutput(),
        embodied.logger.JSONLOutput(logdir, 'metrics.jsonl'),
    ]
    if config.wandb.project is not None:
        os.makedirs(config.wandb.dir, exist_ok=True)
        outputs.append(embodied.logger.WandBOutput(config))

    logger = embodied.Logger(step, outputs)

    env = make_batch_env(env_config_path=args.env_config_path, config=config, seed=config['seed'],
                         n_envs=args.n_envs)
    eval_env = make_batch_env(env_config_path=args.env_config_path, config=config, seed=config['seed'] + args.n_envs,
                              n_envs=1, parallel='none')

    agent = dreamerv3.Agent(env.obs_space, env.act_space, step, config)
    replay = embodied.replay.Uniform(
        config.batch_length, config.replay_size, logdir / 'replay')
    args = embodied.Config(
        **config.run, logdir=config.logdir,
        batch_steps=config.batch_size * config.batch_length)
    if checkpoint_path is None:
        checkpoint_path = os.path.join(config.wandb.project, config.wandb.name)
    os.makedirs(checkpoint_path, exist_ok=True)
    embodied.run.train_eval_no_replay(agent, train_env=env, eval_env=eval_env, train_replay=replay, logger=logger,
                                      args=args, checkpoint_path=checkpoint_path, param_prefixes=param_prefixes)


if __name__ == "__main__":
    main()
