import numpy as np

def vectorize_tensor(T, B):
	"""
	Vectorizes the tensor T by selecting only the indices in B.

	Parameters:
	- T: numpy array of shape (num_samples, ...)
	- B: list of indices to select from each sample

	Returns:
	- V: vectorized representation of T of shape (num_samples, len(B))
	"""
	num_samples = T.shape[0]
	V = np.zeros((num_samples, len(B)))
	for i in range(num_samples):
		for j, idx in enumerate(B):
			V[i, j] = T[i, *idx]
	return V

def reconstruct_tensor(V, T_shape, B):
	"""
	Reconstructs the tensor T from its vectorized representation V.

	Parameters:
	- V: vectorized representation of T of shape (num_samples, len(B))
	- T_shape: original shape of the tensor T
	- B: list of indices that were used to create the vectorized representation

	Returns:
	- T: reconstructed tensor with the original shape, missing values filled with zero
	"""
	num_samples = V.shape[0]
	T = np.zeros(T_shape)
	for i in range(num_samples):
		for j, idx in enumerate(B):
			T[i, *idx] = V[i, j]
	return T

def renormalize_image(image, threshold=False, threshold_value=10):
	"""
	Renormalize an image such that pixel values are between 0 and 255.
	Optionally, set small pixel values to 0.

	Parameters:
	- image: numpy array with shape (28, 28)
	- threshold: bool, if True, set small pixel values to 0
	- threshold_value: int, the value below which pixels are set to 0 (if threshold is True)

	Returns:
	- renormalized_image: numpy array with shape (28, 28), pixel values between 0 and 255
	"""
	# Ensure the image is a numpy array
	image = np.array(image)

	# Renormalize pixel values to be between 0 and 255
	min_val = np.min(image)
	max_val = np.max(image)
	renormalized_image = 255 * (image - min_val) / (max_val - min_val)

	# Apply threshold if needed
	if threshold:
		renormalized_image[renormalized_image < threshold_value] = 0

	return renormalized_image.astype(np.uint8)