// Exponentially weighted forecasters for piecewise Lipschitz functions.
// For a function type F, computes a class to calculate the expert for
// each round. Forecaster is an abstract class for sampling an expert,
// exponential forecaster, fixed share etc are expected subclasses.

// Call to Init() resets the forecaster. Call to Forecast(t) returns
// prediction for time t. Forecast(u_t) increments time index and
// returns prediction for next time index given adversary chose u_t.

#include <algorithm>
#include <iostream>
#include <limits>
#include <math.h>
#include <string>
#include <time.h>  
#include <vector>

#include "interval.h"

// namespace plt = matplotlibcpp;

double rand01() {
  return ((double) rand() / (RAND_MAX));
}

// Abstract class for forecastable functions.
// NOTE: currently only instance is piecewise constant functions.
template<class R>
class Function {
  public:
    Function() : dom(R()) {}
    Function(const R domain) : dom(domain) {}
    // Integral of function over r \subseteq dom.
    const double integral(const R r) const;
    // Evaluate function at rho in domain.
    double Eval(const double rho);
    // Domain of function.
    const R dom;
};


class ConstantFunction : Function<Interval> {
  public:
    ConstantFunction(const double v, const double l, const double u) :
      value(v), dom(l, u) {}
    double value;
    const double integral(const Interval r) const {
      return value * r.length();
    }
    const Interval dom;
};

// If the utility functions u_t are piecewise constant, then so are the
// functions exp(lambda * u_t), so this class can be used to represent both.
class PiecewiseConstantFunction : Function<Interval> {
  public:
    PiecewiseConstantFunction() : dom(0,0), is_initialized_(true), pieces_({}) {}
    // Zero function over interval i.
    PiecewiseConstantFunction(Interval i) : dom(i.lower, i.upper), is_initialized_(true) {
      pieces_.push_back(ConstantFunction(0,i.lower,i.upper));
    }
    // TODO: The pointer is just to avoid copying, ideally we should use unique_ptr here.
    PiecewiseConstantFunction(std::vector<ConstantFunction> *pieces) :
      dom(pieces->front().dom.lower, pieces->back().dom.upper) , pieces_(*pieces) {
      is_initialized_ = true;
    }
    void ResetPieces(std::vector<ConstantFunction> pieces) {
      pieces_.clear();
      for (auto piece : pieces) {
        pieces_.push_back(piece);
      }
    }
    void Scale(double scale) {
      std::vector<ConstantFunction> scaled;
      for (auto piece : pieces_) {
        scaled.push_back(ConstantFunction(piece.value * scale,piece.dom.lower,piece.dom.upper));
      }
      ResetPieces(scaled);
    }
    void Print() {
      for (const auto piece : pieces_) {
        std::cout << "[" << piece.dom.lower << "," << piece.dom.upper << "] -> " << piece.value << "\n";
      }
      // std::cout << "\n";
    }

    double Eval(const double rho) {
      // Find the right piece and return value.
      for (const auto piece : pieces_) {
        if (piece.dom.upper > rho) return piece.value;
      }
      std::cout << "Bad rho query: " << rho << "\n";
      return -1;
    }

    const double integral(const Interval r) const {
        double sum  = 0;
        for (const auto piece : pieces_) {
            if (r.lower > piece.dom.upper) continue;
            if (r.upper < piece.dom.lower) break;
            const Interval i(std::max(r.lower, piece.dom.lower),
                             std::min(r.upper, piece.dom.upper));
            sum += piece.integral(i);
        }
        return sum;
    }
    // Samples a piece (returns zero-based piece index in the sorted vector)
    // in proportion to the integral weight of the piece.
    int SamplePiece() {
      std::vector<double> integrals;
      integrals.push_back(0);
      for (auto piece : pieces_) {
        integrals.push_back(integrals.back() + piece.integral(piece.dom));
      }
      double r  = rand01()*integrals.back();
      // pick index acc to r.
      for (int j = 1; j < integrals.size(); ++j) {
        if (integrals[j]>r) {
          return j-1;
        }
      }
      return integrals.size() - 1;
    }

    // Returns top quantile as a zero-one Piecewise constant function
    // based on given input quantile parameter; 0.1 implies top 10% pieces
    // (by measure) will be returned as one, others as zero.
    PiecewiseConstantFunction Quantile(double q) {
      std::vector<ConstantFunction> quantiled;
      double covered = 0.0;
      while (covered < q) {
        // Find the max piece.
        double mval = -99999.0, ll = dom.lower, ul = dom.upper;   // Assumes functions are largely non-negative.
        for (auto piece : pieces()) {
          if (piece.value > mval) {
            ll = piece.dom.lower;
            ul = piece.dom.upper;
            mval = piece.value;
          }
        }
        ConstantFunction mcf(1.0, ll, ul);
        quantiled.push_back(mcf);
        covered += mcf.dom.length();
      }
      // Fix up quantiled, sort and insert zero intervals.
      auto sortCF = [](std::vector<ConstantFunction> data) {
        std::vector<ConstantFunction> sorted;
        for (int i=0; i<data.size(); i++) {
          // Find min among remaining.
          double min = 9999999.0;
          int idx = -1;
          for (int j=0; j<data.size(); ++j) {
            if (data[j].value > 0 && data[j].dom.lower < min) {
              min = data[j].dom.lower;
              idx = j;
            }
          }
          ConstantFunction c(data[idx].value, data[idx].dom.lower, data[idx].dom.upper);
          sorted.push_back(c);
          data[idx].value = -1.0;
        }
      };
      sortCF(quantiled); 
      std::vector<ConstantFunction> fixed;
      double bar = 0.0;
      for (auto piece : quantiled) {
        if (piece.dom.lower > bar) {
          ConstantFunction cf(0.0, bar, piece.dom.lower);
          fixed.push_back(cf);
          fixed.push_back(piece);
          bar = piece.dom.upper;
        } else {
          fixed.push_back(piece);
          bar = piece.dom.upper;
        }
      }
      if (bar < dom.upper) {
        ConstantFunction cf(0.0, bar, dom.upper);
        fixed.push_back(cf);
      }
      PiecewiseConstantFunction p(dom);
      p.ResetPieces(fixed);
      return p;
    }

    bool is_initialized() {return is_initialized_;}
    std::vector<ConstantFunction> pieces() {return pieces_;}
    const Interval dom;
  private:
    // Pieces of function, in sorted order.
    std::vector<ConstantFunction> pieces_;
    bool is_initialized_;
};

template<class F>
class Forecaster {
  public:
    Forecaster() {}
    // u_t(.) is presented to the forecaster, the forecaster returns
    // prediction rho in the domain for t+1.
    double Forecast(F u_t);
    double payoff() {return payoff_;}
  private:
    // total payoff so far.
    double payoff_;
    // current prediction.
    double rho_;

};

template<class F>
class ExponentialForecaster : public Forecaster<F> {
  public:
    ExponentialForecaster(double R, double L, double H, int T, double w, int d, Interval i) : U_(i) {
      // R should be i.length().
      lambda_ = sqrt(d * log(R/w)/T)/H;
      // Uniformly random initial point.
      rho_ = rand01() * i.length() + i.lower;
      payoff_ = 0.0;
    }
    // u_t(.) is presented to the forecaster, the forecaster returns
    // prediction rho in the domain for t+1 and experiences payoff for
    // time t based on last prediction.
    double Forecast(F u_t) {
      // Add u_t to U_;
      Add(&U_, u_t);
      payoff_ += u_t.Eval(rho_);
      rho_ = Sample(U_, lambda_);
      return rho_;
    }
    double payoff() {return payoff_;}
  private:
    double lambda_;
    // total payoff so far.
    double payoff_;
    // current prediction.
    double rho_;
    // sum of payoff functions so far.
    F U_;
};

double Max(PiecewiseConstantFunction u) {
  double rho, max_val = -1;
  for (auto piece : u.pieces()) {
    if (piece.value > max_val) {
      max_val = piece.value;
      rho = (piece.dom.lower + piece.dom.upper)/2;
    }
  }
  return rho;
}

double MaxVal(PiecewiseConstantFunction u) {
  double rho, max_val = -1;
  for (auto piece : u.pieces()) {
    if (piece.value > max_val) {
      max_val = piece.value;
    }
  }
  return max_val;
}

// Normalize piecewise constant function to a probability distribution.

// Offline optimal forecaster, useful to compute offline optimal
// payoff.
template<class F>
class OfflineOptimalForecaster : public Forecaster<F> {
  public:
    OfflineOptimalForecaster(Interval i) : U_(i) {
      // Uniformly random initial point (not really used).
      rho_ = rand01() * i.length() + i.lower;
      payoff_ = 0.0;
    }
    // u_t(.) is presented to the forecaster, the forecaster computes
    // optimal rho for U_ with knowledge of u_t. payoff is reset to
    // offline optimal value at each turn.
    double Forecast(F u_t) {
      // Add u_t to U_;
      Add(&U_, u_t);
      rho_ = Max(U_);
      payoff_ = U_.Eval(rho_);
      return rho_;
    }
    double payoff() {return payoff_;}
  private:
    double payoff_;
    // current prediction.
    double rho_;
    // sum of payoff functions so far.
    F U_;
};

void PrintMatrix(std::vector<std::vector<double>> M) {
    std::cout <<"{";
  for (auto m : M) {
    std::cout <<"{";
    for (auto n : m) {
      std::cout << n << ",";
    }
    std::cout <<"}, ";
  }
    std::cout <<"}\n";
}

double ShiftedOptimal(std::vector<PiecewiseConstantFunction> U,
    int s, std::vector<std::vector<double>> *opt) {
      // No-shift OPTs for u_{i+1}+...+u_{t}; for i=0...t
      std::vector<double> opt0;
      for (auto u : U) {
        opt0.push_back(MaxVal(u));
      }
      if (s == 1) {
        return opt0.front();
      }
      (*opt)[0].push_back(opt0.front());
      // Cases: 1 to s-1 shifts.
      for (int j = 1; j < s; ++j) {
        // Compute OPT(j; U) by DP for each j.
        // OPT(j; U) = max_k OPT(j-1; u_1,...,u_k)+OPT(0; rest)
        double max = 0;  // NOTE: utility functions assumed non-negative
        for (int k = 1; k < U.size(); ++k) {
          // Assumes (*opt)[j-1] is already computed for u_1,...,u_k
          double v = (*opt)[j-1][k-1]+opt0[k];
          max = max >= v ? max : v;
        }
        if (j < s-1) {
          (*opt)[j].push_back(max);
        } else {
          return max;
        }
      }
      std::cout <<"shouldn't reach here!";
      return 0.0;
}

// Offline optimal forecaster, useful to compute offline optimal
// payoff for s-shifted adversary.
template<class F>
class OfflineShiftedOptimalForecaster : public Forecaster<F> {
  public:
    OfflineShiftedOptimalForecaster(int s, Interval i) : U_({F(i)}),
        dom_(i.lower, i.upper) {
      s_ = s;
      payoff_ = 0.0;
      for (int j=1; j < s_; ++j){
        opt_.push_back({});
      }
    }
    // u_t(.) is presented to the forecaster, the forecaster computes
    // optimal rho for U_ with knowledge of u_t. payoff is reset to
    // offline optimal value at each turn.
    // TODO: Forecast value here is currently unimplemented and unused.
    double Forecast(F u_t) {
      // Update the function sums.
      for (auto& f : U_) {
        Add(&f, u_t);
      }
      F zf(dom_);  // zero fn
      U_.push_back(zf);
      payoff_ = ShiftedOptimal(U_, s_, &opt_);
      return 0.0;
    }
    double payoff() {return payoff_;}
  private:
    Interval dom_;
    double payoff_;
    int s_;
    // sum of payoff functions from time i to t, in increasing order of i,
    // last element is the zero function, i.e. U_[i]=SUM_{i+1}^t{u_j}
    std::vector<F> U_;
    // optimal payoffs with up to 'i' shifted experts for {u_1, ..., u_{j+1}}
    // i.e., 's-1' vectors (i=1...s-1), each of size t
    std::vector<std::vector<double>> opt_;
};

// Adds u2 to u1 and update u1 with the sum.
// Domain is assumed to be the same.
void Add(PiecewiseConstantFunction *u1, PiecewiseConstantFunction u2, bool subtract = false) {
  int i = 0, j = 0, i1 = u1->pieces().size(), j1 = u2.pieces().size();
  if (u1->dom.upper != u2.dom.upper || u1->dom.lower != u2.dom.lower) {
    std::cout << "Error: unmatched domains [" << u1->dom.lower <<
        "," << u1->dom.upper << "], [" << u2.dom.lower <<
        "," << u2.dom.upper << "] for inputs to Add(.,.)";
    return;
  }
  // Special cases of adding empty functions. Note: since domains are
  // assumed equal, both or neither is empty.
  if (i1 == 0 || j1 == 0) {
    return;
  }
  // actually both lower bars should coincide.
  double bar = std::max(u1->pieces().front().dom.lower, u2.pieces().front().dom.lower);
  std::vector<ConstantFunction> sum;
  while(i<i1 && j<j1) {
    auto piece1 = u1->pieces()[i];
    auto piece2 = u2.pieces()[j];
    if (bar == piece1.dom.lower) {
      if (piece1.dom.upper < piece2.dom.upper) {
        if (subtract) {
          ConstantFunction c(piece1.value - piece2.value, bar, piece1.dom.upper);
          sum.push_back(c);
        } else {
          ConstantFunction c(piece1.value + piece2.value, bar, piece1.dom.upper);
          sum.push_back(c);
        }
        bar = piece1.dom.upper;
        i++;
        continue;
      } else if (piece1.dom.upper > piece2.dom.upper) {
        if (subtract) {
          ConstantFunction c(piece1.value - piece2.value, bar, piece2.dom.upper);
          sum.push_back(c);
        } else {
          ConstantFunction c(piece1.value + piece2.value, bar, piece2.dom.upper);
          sum.push_back(c);
        }
        bar = piece2.dom.upper;
        j++;
        continue;
      } else {
         // piece1.dom.upper == piece2.dom.upper
        if (subtract) {
          ConstantFunction c(piece1.value - piece2.value, bar, piece1.dom.upper);
          sum.push_back(c);
        } else {
          ConstantFunction c(piece1.value + piece2.value, bar, piece1.dom.upper);
          sum.push_back(c);
        }
        bar = piece1.dom.upper;
        j++;
        i++;
        continue;
      }
    } else { // bar == piece2.dom.lower
      if (piece2.dom.upper < piece1.dom.upper) {
        if (subtract) {
          ConstantFunction c(piece1.value - piece2.value, bar, piece2.dom.upper);
          sum.push_back(c);
        } else {
          ConstantFunction c(piece1.value + piece2.value, bar, piece2.dom.upper);
          sum.push_back(c);
        }
        bar = piece2.dom.upper;
        j++;
        continue;
      } else if (piece2.dom.upper > piece1.dom.upper) {
        if (subtract) {
          ConstantFunction c(piece1.value - piece2.value, bar, piece1.dom.upper);
          sum.push_back(c);
        } else {
          ConstantFunction c(piece1.value + piece2.value, bar, piece1.dom.upper);
          sum.push_back(c);
        }
        bar = piece1.dom.upper;
        i++;
        continue;
      } else {
         // piece1.dom.upper == piece2.dom.upper
        if (subtract) {
          ConstantFunction c(piece1.value - piece2.value, bar, piece1.dom.upper);
          sum.push_back(c);
        } else {
          ConstantFunction c(piece1.value + piece2.value, bar, piece1.dom.upper);
          sum.push_back(c);
        }
        bar = piece1.dom.upper;
        j++;
        i++;
        continue;
      }
    }
  }
  u1->ResetPieces(sum);
}



// Expectation, denoted as a constant function over the domain.
PiecewiseConstantFunction Expectation(PiecewiseConstantFunction u) {
  double integral = u.integral(u.dom);
  double expectation = integral/u.dom.length();
  ConstantFunction c(expectation, u.dom.lower, u.dom.upper);
  PiecewiseConstantFunction p(u.dom);
  p.ResetPieces({c});
  return p;
}

// Returns f - E(f), so that new mean is zero.
void Normalize(PiecewiseConstantFunction *u) {
  PiecewiseConstantFunction e = Expectation(*u);
  Add(u, e, /*subtract=*/true);
}


PiecewiseConstantFunction Prod(PiecewiseConstantFunction u1, PiecewiseConstantFunction u2) {
  int i = 0, j = 0, i1 = u1.pieces().size(), j1 = u2.pieces().size();
  if (u1.dom.upper != u2.dom.upper || u1.dom.lower != u2.dom.lower) {
    std::cout << "Error: unmatched domains [" << u1.dom.lower <<
        "," << u1.dom.upper << "], [" << u2.dom.lower <<
        "," << u2.dom.upper << "] for inputs to Add(.,.)";
    return u1;
  }
  // Another Special case of multiplying empty functions.
  if (i1 == 0 || j1 == 0) {
    return u1;
  }
  // actually both lower bars should coincide.
  double bar = std::max(u1.pieces().front().dom.lower, u2.pieces().front().dom.lower);
  std::vector<ConstantFunction> prod;
  while(i<i1 && j<j1) {
    auto piece1 = u1.pieces()[i];
    auto piece2 = u2.pieces()[j];
    if (bar == piece1.dom.lower) {
      if (piece1.dom.upper < piece2.dom.upper) {
        ConstantFunction c(piece1.value * piece2.value, bar, piece1.dom.upper);
        bar = piece1.dom.upper;
        prod.push_back(c);
        i++;
        continue;
      } else if (piece1.dom.upper > piece2.dom.upper) {
        ConstantFunction c(piece1.value * piece2.value, bar, piece2.dom.upper);
        bar = piece2.dom.upper;
        prod.push_back(c);
        j++;
        continue;
      } else {
         // piece1.dom.upper == piece2.dom.upper
        ConstantFunction c(piece1.value * piece2.value, bar, piece1.dom.upper);
        bar = piece1.dom.upper;
        prod.push_back(c);
        j++;
        i++;
        continue;
      }
    } else { // bar == piece2.dom.lower
      if (piece2.dom.upper < piece1.dom.upper) {
        ConstantFunction c(piece1.value * piece2.value, bar, piece2.dom.upper);
        bar = piece2.dom.upper;
        prod.push_back(c);
        j++;
        continue;
      } else if (piece2.dom.upper > piece1.dom.upper) {
        ConstantFunction c(piece1.value * piece2.value, bar, piece1.dom.upper);
        bar = piece1.dom.upper;
        prod.push_back(c);
        i++;
        continue;
      } else {
         // piece1.dom.upper == piece2.dom.upper
        ConstantFunction c(piece1.value * piece2.value, bar, piece1.dom.upper);
        bar = piece1.dom.upper;
        prod.push_back(c);
        j++;
        i++;
        continue;
      }
    }
  }
  PiecewiseConstantFunction p(u1.dom);
  p.ResetPieces(prod);
  return p;
}

PiecewiseConstantFunction Exp(PiecewiseConstantFunction u, double lambda) {
  PiecewiseConstantFunction v(u.dom);
  std::vector<ConstantFunction> pieces;
  for (auto& piece : u.pieces()) {
    ConstantFunction new_piece(
      exp(lambda*piece.value), piece.dom.lower, piece.dom.upper);
    pieces.push_back(new_piece);
  }
  v.ResetPieces(pieces);
  return v;
}

PiecewiseConstantFunction Scale(PiecewiseConstantFunction u, double c) {
  PiecewiseConstantFunction v(u.dom);
  std::vector<ConstantFunction> pieces;
  for (auto& piece : u.pieces()) {
    ConstantFunction new_piece(c*piece.value, piece.dom.lower, piece.dom.upper);
    pieces.push_back(new_piece);
  }
  v.ResetPieces(pieces);
  return v;
}

// Given piecewise constant u, and lambda, sample a point from
// u.dom according to u.
double Sample(PiecewiseConstantFunction u) {
  int i = u.SamplePiece();
  double r = rand01();
  return u.pieces()[i].dom.length() * r + u.pieces()[i].dom.lower;
}

// Given piecewise constant u, and lambda, sample a point from
// u.dom according to exp(lambda*u).
double Sample(PiecewiseConstantFunction u, double lambda) {
  PiecewiseConstantFunction explu = Exp(u, lambda);
  int i = explu.SamplePiece();
  double r = rand01();
  return explu.pieces()[i].dom.length() * r + explu.pieces()[i].dom.lower;
}

// FSEF

double Coeff(int i, int t, std::vector<double> weights, std::vector<double> prod_weights, double alpha) {
  if (i == t) {
    if (i == 1) {
      return 1;
    } else {
      return alpha;
    }
  } else {
    return (i == 1 ? 1 : alpha) * prod_weights[i-1]/weights[t-1];
  }
}

// given probabilities that sum to one, pick index proportional to p[idx]
int SampleIndex(std::vector<double> cs) {
  double z = rand01();
  for (int j = 1; j < cs.size(); ++j) {
    if (cs[j-1]>z) {
      return j;
    }
  }
  return cs.size();
}

template<class F>
class FixedShareExponentialForecaster : public Forecaster<F> {
  public:
    // R should be i.length().
    // s and T must be positive; for s=1, we obtain regular ExponentialForecaster
    FixedShareExponentialForecaster(double R, double L, double H, int T, int s,
        double w, int d, Interval i) : U_({F(i)}), dom_(i.lower, i.upper) {
      lambda_ = sqrt(s*(d * log(R/w*1.0) + log(T/s*1.0))/T)/H;
      // Uniformly random initial point.
      rho_ = rand01() * i.length() + i.lower;
      payoff_ = 0.0;
      alpha_ = T == 1 ? 0 : (s-1.0)/(T-1.0);
      t_ = 0;
      volc_ = i.length();
      idx_ = 0;
    }
    // u_t(.) is presented to the forecaster, the forecaster returns
    // prediction rho in the domain for t+1 and experiences payoff for
    // time t based on last prediction.
    double Forecast(F u_t) {
      t_++;
      payoff_ += u_t.Eval(rho_);
      rho_ = Sample(U_[idx_], lambda_);
      // Update the function sums.
      for (auto& f : U_) {
        Add(&f, u_t);
      }
      F zf(dom_);  // zero fn
      U_.push_back(zf);
      // Computes vector with ith element (1-alpha)^(t-i) * W_i * W~(i, t) for i=1...t-1
      std::vector<double> prod_weights;
      double volc_wt = 0; // we'll compute vol(C)*w_t here.
      for (int i = 1; i < t_; ++i) {
        auto expui = Exp(U_[i], lambda_);
        prod_weights.push_back(pow(1-alpha_,t_-i) * weights_[i-1] * expui.integral(dom_));
        volc_wt += (i == 1 ? 1 : alpha_) * prod_weights.back();
      }
      if (t_ == 1) {
        weights_.push_back(volc_);
      } else {
        weights_.push_back(volc_wt/volc_);
      }
      // Compute coefficients for mixed distribution.
      std::vector<double> Cs;
      // TODO: we don't need to compute all coeff, can just
      // sample a point in [0,1] and then compute cumulatively.
      for (int i = 1; i <= t_; ++i) {
        Cs.push_back(Coeff(i, t_, weights_, prod_weights, alpha_));
      }
      // Pick i proportional to coefficient_i,t = Cs[i-1]
      idx_ = SampleIndex(Cs);
      return rho_;
    }
    double payoff() {return payoff_;}
  private:
    Interval dom_;
    int t_;
    int idx_; // distribution w~(idx,t) for next point.
    double lambda_;
    // vol(C=domain)
    double volc_;
    // total payoff so far.
    double payoff_;
    // current prediction.
    double rho_;
    // mixture with uniform.
    double alpha_;
    // memoized weights W_i (i=1 ... t) of fsef.
    std::vector<double> weights_;
    // sum of payoff functions from time i to t, in increasing order of i,
    // last element is the zero function, i.e. U_[i]=SUM_{i+1}^t{u_j}
    std::vector<F> U_;
};

PiecewiseConstantFunction FormulaW(PiecewiseConstantFunction x,
    std::vector<PiecewiseConstantFunction> P, double alpha, double gamma) {
      auto x1 = Scale(x,1.0-alpha);
      double decay = exp(-1.0*gamma);
      double D = 1;
      for (int i = 0; i < P.size() - 1; ++i) {
        D = D*decay + 1;
      }
      double W = alpha*x.integral(x.dom)/D;
      for (int i = 0; i < P.size(); ++i) {
        Add(&x1, Scale(P[i], exp(-1.0*gamma*(P.size() - 1 - i))));
      }
      return x1;
    }

template<class F>
class GeneralizedShareExponentialForecaster : public Forecaster<F> {
  public:
    // R should be i.length().
    // s and T must be positive
    GeneralizedShareExponentialForecaster(double R, double L, double H, int T, int s, int m,
        double w, int d, Interval i) : w_(F(i)), dom_(i.lower, i.upper), P_({F(i)}) {
      lambda_ = sqrt((m*d * log(R/w*1.0) + s*log(m*T/s*1.0))/T)/H;
      // Uniformly random initial point.
      rho_ = rand01() * i.length() + i.lower;
      payoff_ = 0.0;
      alpha_ = s/T;
      gamma_ = alpha_/m;
      t_ = 0;
      volc_ = i.length();
      ConstantFunction uniw(1.0,i.lower,i.upper);
      w_.ResetPieces({uniw});
      ConstantFunction uni(1.0/volc_,i.lower,i.upper);
      P_[0].ResetPieces({uni});
    }
    // u_t(.) is presented to the forecaster, the forecaster returns
    // prediction rho in the domain for t+1 and experiences payoff for
    // time t based on last prediction.
    double Forecast(F u_t) {
      t_++;
      payoff_ += u_t.Eval(rho_);
      auto x = Prod(Exp(u_t, lambda_), w_);
      double W_t = x.integral(x.dom);
      w_.ResetPieces(FormulaW(x, P_, alpha_, gamma_).pieces());
      auto p_t = Scale(w_, 1.0/W_t);
      P_.push_back(p_t);
      rho_ = Sample(p_t);
      return rho_;
    }
    double payoff() {return payoff_;}
  private:
    Interval dom_;
    int t_;
    double lambda_;
    double gamma_;
    // vol(C=domain)
    double volc_;
    // total payoff so far.
    double payoff_;
    // current prediction.
    double rho_;
    // mixture with uniform.
    double alpha_;
    // last weight function, w_{t-1}
    F w_;
    // past probability distributions: p1 ... p{t-1}
    std::vector<F> P_;
};
