# pip install numpy scipy cvxpy
# get MOSEK license: https://www.mosek.com/products/academic-licenses/
# move MOSEK license to ~/mosek/mosek.lic


import cvxpy as cp
import numpy as np
from scipy import sparse as sp


def update(vols, rhos, eta, gamma, sparse=True, solver='MOSEK', verbose=False):
    '''
    Args:
        vols: numpy vector corresponding to volume in each discretization cell
        rhos: t * len(vols) numpy array corresponding to task optima
        eta: scalar step size
        gamma: scalar density bound
        sparse: convert rhos to sparse matrix before solving
        solver: 'MOSEK' or 'SCS' or 'ECOS'
        verbose: print solver output
    Returns:
        numpy vector with len(vols) entries
    '''

    w = cp.Variable(len(vols))
    if sparse:
        rhos = sp.csr_matrix(rhos)
    objective = cp.Minimize(sum(cp.kl_div(w, vols)) - eta * sum(cp.log(rhos @ w)))
    constraints = [w >= gamma * vols, sum(w) == 1]
    problem = cp.Problem(objective, constraints)
    problem.solve(solver=solver, verbose=verbose)
    return w.value


if __name__ == '__main__':

    T, dim = 100, 1000
    vols = np.random.rand(dim)
    vols /= vols.sum()
    rhos = np.random.rand(T, dim)
    update(vols, (rhos>0.9)*1.0, 0.01, 0.01, verbose=True)
