import pyrootutils

root = pyrootutils.setup_root(
    search_from=__file__,
    indicator=[".git", "pyproject.toml"],
    pythonpath=True,
    dotenv=True,
)


from typing import List, Optional
from pathlib import Path

import hydra
from omegaconf import OmegaConf, DictConfig
from pytorch_lightning import (
    Callback,
    LightningDataModule,
    LightningModule,
    Trainer,
    seed_everything,
)
from pytorch_lightning.loggers import LightningLoggerBase
import copy

from src.utils import utils

log = utils.get_logger(__name__)

def load_checkpoint(path, device='cpu'):
    path = Path(path).expanduser()
    if path.is_dir():
        path /= 'checkpoint_last.pt'
    # dst = f'cuda:{torch.cuda.current_device()}'
    log.info(f'Loading checkpoint from {str(path)}')
    state_dict = torch.load(path, map_location=device)
    # T2T-ViT checkpoint is nested in the key 'state_dict_ema'
    if state_dict.keys() == {'state_dict_ema'}:
        state_dict = state_dict['state_dict_ema']
    return state_dict



def finetune(config: DictConfig) -> Optional[float]:
    """Contains training pipeline.
    Instantiates all PyTorch Lightning objects from config.

    Args:
        config (DictConfig): Configuration composed by Hydra.

    Returns:
        Optional[float]: Metric score for hyperparameter optimization.
    """

    # Set seed for random number generators in pytorch, numpy and python.random
    if config.get("seed"):
        seed_everything(config.seed, workers=True)

    # We want to add fields to config so need to call OmegaConf.set_struct
    OmegaConf.set_struct(config, False)

    checkpoint = config.finetune.get("ckpt", None)
    if checkpoint is not None:
        checkpoint_type = config.finetune.get('checkpoint_type', 'lightning')
        if checkpoint_type not in ['lightning', 'pytorch', 'timm']:
            raise NotImplementedError(f'checkpoint_type ${checkpoint_type} not supported')

        if checkpoint_type == 'lightning':
            cls = hydra.utils.get_class(config.task._target_)
            trained_model = cls.load_from_checkpoint(checkpoint_path=config.finetune.ckpt)
        elif checkpoint_type == 'pytorch':
            trained_model: LightningModule = hydra.utils.instantiate(config.task, cfg=config,
                                                                     _recursive_=False)
            loaded = load_checkpoint(config.finetune.ckpt, device=trained_model.device)
            load_return = trained_model.model.load_state_dict(load_checkpoint(config.finetune.ckpt,
                                                                              device=trained_model.device),
                                                              strict=False)
            log.info(load_return)
        else:
            trained_model: LightningModule = hydra.utils.instantiate(config.task, cfg=config, _recursive_=False)

    else:
        #trained_model: LightningModule = hydra.utils.instantiate(config.task, cfg=config, _recursive_=False)
        model_cfg = {
                "model":{
                  "_target_":"src.models.vit.vit.vit_small_patch4_32",
                  "drop_rate":0.0,
                  "drop_path_rate":0.1,
                  "img_size":32
                }
            }
        base_model = hydra.utils.instantiate(model_cfg, _recursive_=False)

    # Init lightning model
    model: LightningModule = hydra.utils.instantiate(config.task, cfg=config, _recursive_=False)

    # Swap layers

    base_block_length = len(base_model.blocks)

    for b in config.finetune.block_indices:
        print(base_model.blocks[b])
        base_model.blocks[b] = fast_structure.blocks[b]
        base_model.blocks[b].train()
    exit()

    model.model = base_model

    datamodule: LightningDataModule = model._datamodule


    # Init lightning callbacks
    callbacks: List[Callback] = []
    if "callbacks" in config:
        for _, cb_conf in config.callbacks.items():
            if cb_conf is not None and "_target_" in cb_conf:
                log.info(f"Instantiating callback <{cb_conf._target_}>")
                callbacks.append(hydra.utils.instantiate(cb_conf))

    # Init lightning loggers
    logger: List[LightningLoggerBase] = []
    if "logger" in config:
        for _, lg_conf in config.logger.items():
            if "_target_" in lg_conf:
                log.info(f"Instantiating logger <{lg_conf._target_}>")
                logger.append(hydra.utils.instantiate(lg_conf))

    if config.get('resume'):
        try:
            checkpoint_path = Path(config.callbacks.model_checkpoint.dirpath)
            if checkpoint_path.is_dir():
                checkpoint_path /= 'last.ckpt'
            if checkpoint_path.is_file():
                config.trainer.resume_from_checkpoint = str(checkpoint_path)
            else:
                log.info(f'Checkpoint file {str(checkpoint_path)} not found. Will start training from scratch')
        except KeyError:
            pass
    # Init lightning trainer
    log.info(f"Instantiating trainer <{config.trainer._target_}>")
    trainer: Trainer = hydra.utils.instantiate(
        config.trainer, callbacks=callbacks, logger=logger, _convert_="partial"
    )

    # Train the model
    log.info("Starting fine-tuning!")
    trainer.fit(model=model, datamodule=datamodule)

    train_metrics = trainer.callback_metrics

    # Evaluate model on test set, using the best model achieved during training
    if config.get("test_after_training") and not config.trainer.get("fast_dev_run"):
        log.info("Starting testing!")
        trainer.test(model=model, datamodule=datamodule)

    # Make sure everything closed properly
    log.info("Finalizing!")
    utils.finish(
        config=config,
        model=model,
        datamodule=datamodule,
        trainer=trainer,
        callbacks=callbacks,
        logger=logger,
    )
    test_metrics = trainer.callback_metrics

    metric_dict = {**train_metrics, **test_metrics}

    # Print path to best checkpoint
    if not config.trainer.get("fast_dev_run"):
        log.info(f"Best model ckpt: {trainer.checkpoint_callback.best_model_path}")

    # Return metric score for hyperparameter optimization
    optimized_metric = config.get("optimized_metric")
    if optimized_metric:
        return metric_dict[optimized_metric]


#@hydra.main(config_path=root / "configs/", config_name="config.yaml", version_base="1.2")
def main(cfg: DictConfig) -> None:
    finetune(cfg)

if __name__ == "__main__":
    main()
