import pyrootutils

root = pyrootutils.setup_root(
    search_from=__file__,
    indicator=[".git", "pyproject.toml"],
    pythonpath=True,
    dotenv=True,
)


from copy import deepcopy
from typing import List, Optional
from pathlib import Path

import hydra
from omegaconf import OmegaConf, DictConfig
from pytorch_lightning import (
    Callback,
    LightningDataModule,
    LightningModule,
    Trainer,
    seed_everything,
)
from pytorch_lightning.loggers import LightningLoggerBase
import torch
import torch.nn as nn

from src.utils import utils
from src.models.layers.fouriermask import FourierMaskLR
from src.models.modules.olb import find_olb
from src.tasks.seq import SequenceModel

log = utils.get_logger(__name__)


def train(config: DictConfig) -> Optional[float]:
    """Contains training pipeline.
    Instantiates all PyTorch Lightning objects from config.

    Args:
        config (DictConfig): Configuration composed by Hydra.

    Returns:
        Optional[float]: Metric score for hyperparameter optimization.
    """

    # Set seed for random number generators in pytorch, numpy and python.random
    if config.get("seed"):
        seed_everything(config.seed, workers=True)

    # We want to add fields to config so need to call OmegaConf.set_struct
    OmegaConf.set_struct(config, False)
    # Init lightning model
    model: LightningModule = hydra.utils.instantiate(config.task, cfg=config, _recursive_=False)
    datamodule: LightningDataModule = model._datamodule

    # Init lightning callbacks
    callbacks: List[Callback] = []
    if "callbacks" in config:
        for _, cb_conf in config.callbacks.items():
            if cb_conf is not None and "_target_" in cb_conf:
                log.info(f"Instantiating callback <{cb_conf._target_}>")
                callbacks.append(hydra.utils.instantiate(cb_conf))

    # Init lightning loggers
    logger: List[LightningLoggerBase] = []
    if "logger" in config:
        for _, lg_conf in config.logger.items():
            if "_target_" in lg_conf:
                log.info(f"Instantiating logger <{lg_conf._target_}>")
                logger.append(hydra.utils.instantiate(lg_conf))

    if config.get('resume'):
        try:
            checkpoint_path = Path(config.callbacks.model_checkpoint.dirpath)
            if checkpoint_path.is_dir():
                checkpoint_path /= 'last.ckpt'
            if checkpoint_path.is_file():
                config.trainer.resume_from_checkpoint = str(checkpoint_path)
            else:
                log.info(f'Checkpoint file {str(checkpoint_path)} not found. Will start training from scratch')
        except KeyError:
            pass

    # Resume from the finetuning starting point
    if config.get('resume_ft'):
        #config.trainer.resume_from_checkpoint = str(path)
        model_resume_ft = model.load_from_checkpoint(checkpoint_path=config.resume_ft, strict=True)
        pft_dict = dict(model_resume_ft.model.named_parameters())
        for pn, p in model.model.named_parameters():
            #if not ('bert' in config.resume_ft and 'classifier' in pn):
            if p.size() ==  pft_dict[pn].size():
                p.data = pft_dict[pn].data
        log.info("Resuming from {}".format(config.resume_ft))


    # Init lightning trainer
    log.info(f"Instantiating trainer <{config.trainer._target_}>")
    trainer: Trainer = hydra.utils.instantiate(
        config.trainer, callbacks=callbacks, logger=logger, _convert_="partial"
    )


    #trainer.validate(model=model, datamodule=datamodule)
    # Train the model
    log.info("Starting training!")
    trainer.fit(model=model, datamodule=datamodule)

    train_metrics = trainer.callback_metrics

    # Evaluate model on test set, using the best model achieved during training
    if config.get("test_after_training") and not config.trainer.get("fast_dev_run"):
        log.info("Starting testing!")
        trainer.test(model=model, datamodule=datamodule)

    # Make sure everything closed properly
    log.info("Finalizing!")
    utils.finish(
        config=config,
        model=model,
        datamodule=datamodule,
        trainer=trainer,
        callbacks=callbacks,
        logger=logger,
    )
    test_metrics = trainer.callback_metrics

    metric_dict = {**train_metrics, **test_metrics}

    # Print path to best checkpoint
    if not config.trainer.get("fast_dev_run"):
        log.info(f"Best model ckpt: {trainer.checkpoint_callback.best_model_path}")

    # Return metric score for hyperparameter optimization
    optimized_metric = config.get("optimized_metric")
    if optimized_metric:
        return metric_dict[optimized_metric]


#@hydra.main(config_path=root / "configs/", config_name="config.yaml", version_base="1.2")
def main(cfg: DictConfig) -> None:
    train(cfg)

if __name__ == "__main__":
    main()
