##########################################################
# --------- EVOLVING DYNAMIC NETWORK ---------
# EvDyNET
##########################################################
# Extension File: 
# Dynamic Network: *.DYNE 
##########################################################
# Custom Class to Save and Load Evolving Dynamic Networks 
#
# *** Version 0.1
# *** Version 0.2
# Add: 
# -  Ricci curvature (done)
# -  Ricci curvature positive (done)

#%% Import libraries
from networkx.algorithms.shortest_paths import weighted
import numpy as np
import pandas as pd 
import networkx as nx
from IPython.display import display
import matplotlib.pyplot as plt
import pickle 
from GraphRicciCurvature.OllivierRicci import OllivierRicci 

#%% Custom Class
######################################################
# DYNAMIC NETWORK
######################################################
class Dynamic_Network:
    # -----------------------------------------------------
    # Class INIT
    def __init__(self, directed, multilayer, matDYNE, nameATTdyne, notes):
        self.directed = directed # True: Directed  |  False: UNDirected
        self.multilayer = multilayer # True: Multilayer  |  False: Unilayer
        self.df_property = pd.DataFrame(matDYNE, columns=nameATTdyne) # Properties of the Dynamic Network -> matDYNE: GxH Numpy Array, nameATTdyne: list 
        self.df_property.index.name = 'Dynamic_Net_Property'
        self.NOTES = notes # To save an string with notes about the network
        self.number_nets = 0 # Number of Nets in the Dynamic Network
        self.number_nodes = {} # Number of Nodes in each Net
        self.number_edges = {} # Number of Edges in each Net
        self.df_features = {} # Feature of the Network (e.g., class, property, etc)
        self.df_nodes = {} # Nodes and its attributes
        self.df_edges = {} # Edges and its attributes

    # -----------------------------------------------------
    # arrNodes: "Sorted" array of nodes in any format (integer, string, etc)
    # matNODE: Nodes' attributes, NxQ matrix -> where each column represent an attribute
    # nameATTnode: Name of Node's attributes
    # arrEdges: Mx2 matrix -> From, To -> where each row represent the edges
    # matEDGE: Edges' attributes, MxW matrix -> where each column represent an edge-attribute (i.e. weight, etc)
    # nameATTedge: Name of Edge's attributes 
    # matGRAPH: Network's attributes -> Numpy Matrix PxPc
    # nameATTgraph: Name of Network's attributes
    # extended_Extra: Add additional measures in DataFrames (degree, betweenness, eigenvector, clustering, etc)
    # --- Additional comments
    # Assuming: NumPy Arrays for arrNodes, matNODE, arrEdges, matEDGE, matGRAPH
    # Assuming: List Arrays for nameATTnode, nameATTedge, nameATTgraph
    # The number of elements in arrNodes and rows in matNODE must be the same
    # The number of rows in arrEdges and matEDGE must be the same
    # --- Return
    # False: It wasn't able to add the network
    # True: If the Network was successfully added
    def addNet(self, arrNodes, matNODE, nameATTnode, arrEdges, matEDGE, nameATTedge, matGRAPH, nameATTgraph, extended_Extra):  
        # --- Parameters Main
        N_nod = arrNodes.shape[0] # Number of Nodes
        #M_edg = arrEdges.shape[0] # Number of Edges
        M_edg = int(np.size(arrEdges)/2) # Number of Edges
        # --- Auxiliary Parameters 
        rowsNODE = 0 if np.size(matNODE)==0 else matNODE.shape[0] 
        rowsEDGE = 0 if np.size(matEDGE)==0 else matEDGE.shape[0] 
        # --- Input-Errors
        if((N_nod != rowsNODE) or (matNODE.shape[1] != len(nameATTnode)) or (M_edg != rowsEDGE) or (matEDGE.shape[1] != len(nameATTedge)) or (matGRAPH.shape[1] != len(nameATTgraph))): 
            print('*** EvDyNET - Error: discrepancy in the number of nodes / edges / net_features provided...')
            print(f' number of nodes: {N_nod}  vs  {rowsNODE}  --- {matNODE.shape[1]}  vs  {len(nameATTnode)}  \n number of edges: {M_edg}  vs  {rowsEDGE}  --- {matEDGE.shape[1]}  vs  {len(nameATTedge)}  \n number of net_features: {matGRAPH.shape[1]}  vs  {len(nameATTgraph)}')
            return False
        
        # --- Create the FEATURE DataFrame
        if(matGRAPH.shape[1] == 0):
            df_Feature = pd.DataFrame()
        else:
            df_Feature = pd.DataFrame(matGRAPH, columns=nameATTgraph) 
            df_Feature.index.name = 'Net_Features'
        
        # --- Create the NODE DataFrame
        if(N_nod == 0):
            df_Node = pd.DataFrame()
        else:
            # Node ID for computations, 0, 1, 2, 3, ... N-1
            df_NodeID = pd.DataFrame(np.arange(N_nod), columns=['ID'])
            # Labels from User 
            df_NodeLabel = pd.DataFrame(arrNodes, columns=['Label'])
            # Attributes from User 
            df_NodeAttNode = pd.DataFrame(matNODE, columns=nameATTnode)  
            # Final Dataframe
            df_Node = pd.concat([df_NodeID.copy(), df_NodeLabel.copy(), df_NodeAttNode.copy()], axis = 1) 
            df_Node.index.name = 'Net_Nodes'
        
        # --- Create the EDGE DataFrame
        if(M_edg == 0):
            df_Edge = pd.DataFrame()
        else:
            # Node ID for computations, 0, 1, 2, 3, ... N-1
            # + Searching for correspondence between ID and nodes in the list of Edges
            arrEdgeID = np.zeros((M_edg, 2), np.int64) - 1 # Starts with -1 
            for i, value in enumerate(arrNodes):
                indFound = np.where(arrEdges == value)
                if(len(indFound[0])>0): # If value was found  
                    arrEdgeID[indFound[0], indFound[1]] = i
            # + Verifying error
            indFound = np.where(arrEdgeID == -1) 
            if(len(indFound[0])>0):
                print('*** EvDyNET - Error: at least one node was not found among the matrix of edges...')
                print(indFound)
                return False
            # + Dataframe
            df_EdgeID = pd.DataFrame(arrEdgeID, columns=['From', 'To'])
            # Labels from User 
            df_EdgeLabel = pd.DataFrame(arrEdges, columns=['From_Label', 'To_Label'])
            # Attributes from User 
            df_EdgeAttEdge = pd.DataFrame(matEDGE, columns=nameATTedge)  
            # Final Dataframe 
            df_Edge = pd.concat([df_EdgeID.copy(), df_EdgeLabel.copy(), df_EdgeAttEdge.copy()], axis = 1) 
            df_Edge.index.name = 'Net_Edges'

        # --- Create the EXTENDED DataFrame
        # At least need to have nodes in  * df_Node * and edges in * df_Edges *
        if(N_nod>1 and M_edg>0 and extended_Extra==True):
            df_Node_Extra, df_Edge_Extra = self.extra_attributes(df_Node, df_Edge)
            df_Node = pd.concat([df_Node, pd.DataFrame(df_Node_Extra)], axis=1)  
            df_Edge = pd.concat([df_Edge, pd.DataFrame(df_Edge_Extra)], axis=1)  

        # --- Saving Dataframes and Updating object-variables
        self.df_features[self.number_nets] = df_Feature.copy()
        self.df_nodes[self.number_nets] = df_Node.copy()  
        self.df_edges[self.number_nets] = df_Edge.copy() 
        self.number_nodes[self.number_nets] = N_nod # Number of Nodes
        self.number_edges[self.number_nets] = M_edg # Number of Edges
        self.number_nets = self.number_nets + 1
        return True
    
    # -----------------------------------------------------
    # The input-data has been previously validated by the calling function
    # 
    def extra_attributes(self, df_Node, df_Edge):
        #display(df_Node)
        #display(df_Edge)
        # Parameters
        NVertices = df_Node.shape[0]
        MEdges = df_Edge.shape[0] 
        # --- Add Information, only nodes and edges 
        G = nx.Graph()
        # Add vertices/nodes
        G.add_nodes_from(df_Node.ID.to_list()) 
        # Add node's attributes from table
        colATTnodes = list( df_Node.columns[range(2, len(df_Node.columns))] )
        for k in range(2, len(df_Node.columns)):
            nx.set_node_attributes(G, df_Node[df_Node.columns[k]], df_Node.columns[k])  
        # Add edges
        colATTedges = list( df_Edge.columns[range(4, len(df_Edge.columns))] )
        for i in range(MEdges): # Over all Edges
            dicATT = dict(zip(colATTedges, df_Edge[colATTedges].iloc[i].to_list()))  
            #print(dicATT)
            #G.add_edge(df_Edge.iloc[i].From, df_Edge.iloc[i].To, dicATT)
            G.add_edges_from([(df_Edge.iloc[i].From, df_Edge.iloc[i].To, dicATT)])

        # --- Computing Node Measures 
        # https://networkx.org/documentation/stable/index.html
        # Section Reference -> Top Menu
        # The New Extra Matrix
        col_Nodes_extra = [] 
        #matNode_Extra = np.empty((NVertices, 8 + 3*len(colATTedges)))
        matNode_Extra = np.empty((NVertices, 7 + 3*len(colATTedges)))  
        matNode_Extra[:] = np.NaN  
        # The Degree  
        idExt = 0
        col_Nodes_extra.append('Degree')  
        vec_degree = np.array( [x[1] for x in list(G.degree())] )  
        matNode_Extra[:,idExt] = vec_degree  
        # The Weighted Degree  
        if(len(colATTedges)>0):
            for txtATT in colATTedges:
                idExt = idExt + 1
                col_Nodes_extra.append('Degree_'+txtATT)  
                vec_degree_w = np.array( [x[1] for x in list(G.degree(weight = txtATT))] )  
                matNode_Extra[:, idExt] = vec_degree_w  
        # Betweenness
        idExt = idExt + 1
        col_Nodes_extra.append('Betweenness')  
        vec_betw = np.array( list(nx.betweenness_centrality(G, normalized=False).values()) )  # G, normalized=False, weight='oh_1'  
        matNode_Extra[:,idExt] = vec_betw  
        # Closeness 
        idExt = idExt + 1
        col_Nodes_extra.append('Closeness')  
        vec_close = np.array( list(nx.closeness_centrality(G, distance=None, wf_improved=True).values()) )  
        matNode_Extra[:,idExt] = vec_close  
        # The Weighted Closeness  
        if(len(colATTedges)>0):
            for txtATT in colATTedges:
                idExt = idExt + 1
                col_Nodes_extra.append('Closeness_'+txtATT)  
                vec_close_w = np.array( list(nx.closeness_centrality(G, distance=txtATT, wf_improved=True).values()) )  
                matNode_Extra[:,idExt] = vec_close_w  
        # Square Clustering 
        idExt = idExt + 1
        col_Nodes_extra.append('Square_Clustering')  
        vec_sq_clus = np.array( list(nx.square_clustering(G).values()) )  
        matNode_Extra[:,idExt] = vec_sq_clus  
        # Clustering 
        idExt = idExt + 1
        col_Nodes_extra.append('Clustering')  
        vec_clus = np.array( list(nx.clustering(G).values()) )  
        matNode_Extra[:,idExt] = vec_clus  
        # The Weighted Clustering  
        if(len(colATTedges)>0):
            for txtATT in colATTedges:  
                idExt = idExt + 1
                col_Nodes_extra.append('Clustering_'+txtATT)  
                vec_clus_w = np.array( list(nx.clustering(G, weight=txtATT).values()) )  
                matNode_Extra[:,idExt] = vec_clus_w  
        # Eigenvector 
        idExt = idExt + 1
        col_Nodes_extra.append('Eigenvector')  
        #print(G.number_of_nodes())
        #print(G.number_of_edges())
        #print(G.edges())
        if(G.number_of_nodes()>2):
            vec_eigen = np.array( list(nx.eigenvector_centrality_numpy(G).values()) )  
        elif(G.number_of_nodes()==2): # Error when there are only two nodes
            vec_eigen = np.array([0.5, 0.5])
        elif(G.number_of_nodes()==1): # Error when there is only one node
            vec_eigen = np.array([0.5])
        else:
            vec_eigen = np.array([])
        matNode_Extra[:,idExt] = vec_eigen  
        # Katz 
        idExt = idExt + 1
        col_Nodes_extra.append('Katz')  
        vec_katz = np.array( list(nx.katz_centrality_numpy(G).values()) )  
        matNode_Extra[:,idExt] = vec_katz  
        # # Second_Order 
        # idExt = idExt + 1
        # col_Nodes_extra.append('Second_Order')  
        # vec_second = np.array( list(nx.second_order_centrality(G).values()) )  
        # matNode_Extra[:,idExt] = vec_second  

        # Additional Columns for Node's DataFramse
        df_Node_Extra = pd.DataFrame(matNode_Extra, columns=col_Nodes_extra)
        
        # -----------------------------------------------
        # Computing Edge Measures
        # https://networkx.org/documentation/stable/index.html
        # nx.edge_betweenness_centrality # Based on Short paths
        # nx.edge_current_flow_betweenness_centrality(auxG, normalized = False)  # Based on Electrical current Model
        # The New Extra Matrix
        col_Edges_extra = []   
        #matEdge_Extra = np.empty((MEdges, 2 + 2*len(colATTedges)))  
        matEdge_Extra = np.empty((MEdges, 3 + 1*len(colATTedges)))
        matEdge_Extra[:] = np.NaN  
        IDFrom = df_Edge.From.to_numpy()
        IDTo = df_Edge.To.to_numpy()
        # Betweenness
        idExt = 0
        col_Edges_extra.append('Betweenness')
        rels_betw = nx.edge_betweenness_centrality(G, normalized=False)  
        for n1, n2 in rels_betw:
            attr = rels_betw[(n1, n2)]    
            # Search From-To 
            from_to = np.where((IDFrom==n1)*(IDTo==n2))[0]
            to_from = np.where((IDFrom==n2)*(IDTo==n1))[0] 
            id_found = np.concatenate((from_to, to_from))
            if(id_found.shape[0]>0): 
                #print(id_found)
                for idx in id_found:
                    matEdge_Extra[idx, idExt] = attr
            else:
                print('*** EvDyNET: extra_attributes(): Betweenness: Edge not found -> Weird... ({n1}, {n2}) -> {attr}')     
        # The Weighted Betweenness
        if(len(colATTedges)>0):
            for txtATT in colATTedges:
                idExt = idExt + 1 
                col_Edges_extra.append('Betweenness_'+txtATT)
                rels_betw_w = nx.edge_betweenness_centrality(G, normalized=False, weight=txtATT)  
                for n1, n2 in rels_betw_w:
                    attr = rels_betw_w[(n1, n2)]    
                    # Search From-To 
                    from_to = np.where((IDFrom==n1)*(IDTo==n2))[0]
                    to_from = np.where((IDFrom==n2)*(IDTo==n1))[0] 
                    id_found = np.concatenate((from_to, to_from))
                    if(id_found.shape[0]>0): 
                        #print(id_found)
                        for idx in id_found:
                            matEdge_Extra[idx, idExt] = attr
                    else:
                        print('*** EvDyNET: extra_attributes(): Betweenness: Edge not found -> Weird... ({n1}, {n2}) -> {attr}')     
        # # Flow-Betweenness
        # idExt = idExt + 1
        # col_Edges_extra.append('Flow_Betweenness')
        # rels_flow = nx.edge_current_flow_betweenness_centrality(G, normalized=False)  
        # for n1, n2 in rels_flow:
        #     attr = rels_flow[(n1, n2)]    
        #     # Search From-To 
        #     from_to = np.where((IDFrom==n1)*(IDTo==n2))[0]
        #     to_from = np.where((IDFrom==n2)*(IDTo==n1))[0] 
        #     id_found = np.concatenate((from_to, to_from))
        #     if(id_found.shape[0]>0): 
        #         #print(id_found)
        #         for idx in id_found:
        #             matEdge_Extra[idx, idExt] = attr
        #     else:
        #         print('*** EvDyNET: extra_attributes(): Betweenness: Edge not found -> Weird... ({n1}, {n2}) -> {attr}')     
        # # The Weighted Flow-Betweenness
        # if(len(colATTedges)>0):
        #     for txtATT in colATTedges:
        #         idExt = idExt + 1 
        #         col_Edges_extra.append('Flow_Betweenness_'+txtATT)
        #         rels_flow_w = nx.edge_current_flow_betweenness_centrality(G, normalized=False, weight=txtATT)  
        #         for n1, n2 in rels_flow_w:
        #             attr = rels_flow_w[(n1, n2)]    
        #             # Search From-To 
        #             from_to = np.where((IDFrom==n1)*(IDTo==n2))[0]
        #             to_from = np.where((IDFrom==n2)*(IDTo==n1))[0] 
        #             id_found = np.concatenate((from_to, to_from))
        #             if(id_found.shape[0]>0): 
        #                 #print(id_found)
        #                 for idx in id_found:
        #                     matEdge_Extra[idx, idExt] = attr
        #             else:
        #                 print('*** EvDyNET: extra_attributes(): Betweenness: Edge not found -> Weird... ({n1}, {n2}) -> {attr}')     
        
        # Ollivier-Ricci curvature (Possible Translation to Positive numbers)
        idExt = idExt + 1
        col_Edges_extra.append('Ricci')
        orc = OllivierRicci(G, alpha=0.5, verbose="INFO")
        orc.compute_ricci_curvature()
        rels_ricci = nx.get_edge_attributes(orc.G, "ricciCurvature")
        for n1, n2 in rels_ricci:
            attr = rels_ricci[(n1, n2)]    
            # Search From-To 
            from_to = np.where((IDFrom==n1)*(IDTo==n2))[0]
            to_from = np.where((IDFrom==n2)*(IDTo==n1))[0] 
            id_found = np.concatenate((from_to, to_from))
            if(id_found.shape[0]>0): 
                #print(id_found)
                for idx in id_found:
                    matEdge_Extra[idx, idExt] = attr
            else:  
                print('*** EvDyNET: extra_attributes(): Ricci: Edge not found -> Weird... ({n1}, {n2}) -> {attr}')     
        
        # Ollivier-Ricci curvature (Possible Translation to Positive numbers)
        idExt = idExt + 1
        col_Edges_extra.append('Ricci_Positive')
        orc = OllivierRicci(G, alpha=0.5, verbose="INFO")
        orc.compute_ricci_curvature()
        rels_ricci = nx.get_edge_attributes(orc.G, "ricciCurvature")
        # To verify there are negative numbers
        minRicci = min(rels_ricci.values())
        flagNeg = False
        if(minRicci<0): # There are negative numbers
            flagNeg = True

        for n1, n2 in rels_ricci:
            attr = rels_ricci[(n1, n2)]    
            # Search From-To 
            from_to = np.where((IDFrom==n1)*(IDTo==n2))[0]
            to_from = np.where((IDFrom==n2)*(IDTo==n1))[0] 
            id_found = np.concatenate((from_to, to_from))
            if(id_found.shape[0]>0): 
                #print(id_found)
                for idx in id_found:
                    if(flagNeg): # There are negative numbers (translate all to positive + epsilon)
                        matEdge_Extra[idx, idExt] = -minRicci + 0.0001 + attr
                    else:
                        matEdge_Extra[idx, idExt] = attr
            else:  
                print('*** EvDyNET: extra_attributes(): Ricci-Positive: Edge not found -> Weird... ({n1}, {n2}) -> {attr}')     

        # Additional Columns for Node's DataFramse
        df_Edge_Extra = pd.DataFrame(matEdge_Extra, columns=col_Edges_extra)  

        return df_Node_Extra, df_Edge_Extra

    # -----------------------------------------------------
    # To show the content of the Dynamic Network
    # 
    def show(self, typeVal, value=0):
        # Select type of visualization according with typeVal
        if(typeVal=='step'):
            for t in range(self.number_nets):
                if(t%value == 0):  
                    print(f'\n *****   NETWORK -> time-index: {t}   ***** ')
                    print(f'number_nodes: {self.number_nodes[t]}')
                    print(f'number_edges: {self.number_edges[t]}')
                    display(self.df_nodes[t])
                    display(self.df_edges[t])
                    display(self.df_features[t])  
        elif(typeVal=='index'):
            print(f'\n *****   NETWORK -> time-index: {value}   ***** ')
            print(f'number_nodes: {self.number_nodes[value]}')
            print(f'number_edges: {self.number_edges[value]}')
            display(self.df_nodes[value])
            display(self.df_edges[value])
            display(self.df_features[value])
        elif(typeVal=='first'):
            print(f'\n *****   NETWORK -> time-index: {0}   ***** ')
            print(f'number_nodes: {self.number_nodes[0]}')
            print(f'number_edges: {self.number_edges[0]}')
            display(self.df_nodes[0])
            display(self.df_edges[0])
            display(self.df_features[0])
        elif(typeVal=='last'):
            print(f'\n *****   NETWORK -> time-index: {self.number_nets-1}   ***** ')
            print(f'number_nodes: {self.number_nodes[self.number_nets-1]}')
            print(f'number_edges: {self.number_edges[self.number_nets-1]}')
            display(self.df_nodes[self.number_nets-1])
            display(self.df_edges[self.number_nets-1])
            display(self.df_features[self.number_nets-1])
        # General Information        
        display(self.df_property)
        print(f'number_nets: {self.number_nets}') 
        print(f'is_directed: {self.directed}') 
        print(f'is_multilayer: {self.multilayer}') 
        print('NOTES: ' + self.NOTES)


#%% Functions to execute actions on DYNAMIC NETWORKs
#######################################################
# ACTIONS on DYNAMIC NETWORKs
#######################################################

# -----------------------------------------------------
# To Save a Dynamic Network
# It uses pickle with protocol = 5
# Minimum Python version = 3.8
def Save_Dynamic_Network(DNet, fname):  
    fhandler = open(fname, 'wb')  
    pickle.dump(DNet, fhandler, protocol = 5)
    fhandler.close()

# -----------------------------------------------------
# To Load a Dynamic Network
# It uses pickle with protocol = 5
# Minimum Python version = 3.8
def Load_Dynamic_Network(fname):  
    infile = open(fname, 'rb')   
    DNet = pickle.load(infile)   
    infile.close()
    return DNet

# -----------------------------------------------------
# To Visualize a Dynamic Network
# DNet: Dynamic_Network object 
# nodeATT: 1 node attribute, same as name-column in 'df_nodes' dataframe
# edgeATT: 1 edge attribute, same as name-column in 'df_nodes' dataframe
# indexPeriod: Numpy Array of indices to plot < number_nets
# factorPlot: A factor to control de height of the plot, assumption: each net is plot in square-like size 
# --------- Return --------- 
# False: We couldn't create the visualization
# True: Visualization was shown/saved
def Visualize_Dynamic_Network(DNet, nodeATT, edgeATT, indexPeriod, out_file, factorPlot=1.0):
    # --- Validations
    if(np.amin(indexPeriod)<0 or np.amax(indexPeriod)>DNet.number_nets): # Indices outside number of nets
        print(f'***** ERROR: Visualize_Dynamic_Network -> indices: {np.amin(indexPeriod)} & {np.amax(indexPeriod)}   vs  Nets: {DNet.number_nets}')
        return False
    if((not isinstance(nodeATT, str)) or (not isinstance(edgeATT, str))): # Tyoe of Node/Edge Column-name
        print(f'***** ERROR: Visualize_Dynamic_Network -> Column-name must be *str*:  Node_Column: {type(nodeATT)}  &  Edge_Column: {type(edgeATT)}')  
        return False
    if(factorPlot<0.0): # Factor value
        print(f'***** ERROR: Visualize_Dynamic_Network -> factor must be positive: {factorPlot}')  
        return False
    # --- Parameters
    valHeight = factorPlot*7 
    sizeWindow = indexPeriod.shape[0]  
    # Create Figure
    plt.figure(num=None, figsize=(sizeWindow*valHeight, valHeight), dpi=80, facecolor='w', edgecolor='k')  

    # --- Add Information in NetworkX and Visualize
    #GraphsNetX = {}  
    for key, idx in enumerate(indexPeriod): 
        # --- Add Information in NetworkX
        G = nx.Graph()
        # Add vertices/nodes
        G.add_nodes_from(DNet.df_nodes[idx].ID.to_list()) 
        # Add node's attributes from table
        nx.set_node_attributes(G, DNet.df_nodes[idx][nodeATT], nodeATT)
        # Add edges
        for i in range(DNet.number_edges[idx]):
            dicATT = dict(zip([edgeATT], DNet.df_edges[idx][[edgeATT]].iloc[i].to_list()))
            G.add_edges_from([(DNet.df_edges[idx].iloc[i].From, DNet.df_edges[idx].iloc[i].To, dicATT)])
        # To Save NetworkX Graph
        #GraphsNetX[key] = G
        # --- Visualize Network    
        plt.subplot(1, sizeWindow, key+1)
        plt.title(str(idx))
        pos = nx.circular_layout(G)
        nx.draw(G, pos, node_size=20, node_color='limegreen', edge_color='r', with_labels=False)
        labels = nx.get_edge_attributes(G, edgeATT)
        for lab in labels: 
            labels[lab] = format(labels[lab], '.3E')
            #labels[lab] = round(labels[lab],5)
        nx.draw_networkx_edge_labels(G, pos, edge_labels=labels, font_size=4)
        allLabels = dict(enumerate(DNet.df_nodes[idx][nodeATT].to_numpy().astype(str), 0)) 
        nx.draw_networkx_labels(G, pos, allLabels, font_size=5, horizontalalignment='center', verticalalignment='center')  

    plt.savefig(out_file, bbox_inches='tight')
    plt.show()
    return True

