# Install the required packages if not already installed
if (!requireNamespace("clue", quietly = TRUE)) {
  install.packages("clue")
}
if (!requireNamespace("lpSolve", quietly = TRUE)) {
  install.packages("lpSolve")
}
if (!requireNamespace("microbenchmark", quietly = TRUE)) {
  install.packages("microbenchmark")
}

# Load the required packages
library(clue)
library(lpSolve)
library(microbenchmark)


# Solve argmax tr(PQ)
# note: solve_LSAP(C) optimizes c_{i,j} X_{i,j} which is equal to tr(C^T X) = tr(X C^T)
solve_lap_LSAP <- function(C) {
  min_value <- min(C)
  if (min_value < 0) {
    C <- C - min_value
  }
  optimal_matching <- solve_LSAP(t(C), maximum = TRUE)
  solution <- matrix(0, nrow = nrow(C), ncol = ncol(C))
  row_indices <- 1:nrow(C)
  solution[cbind(row_indices, optimal_matching)] <- 1
  return(solution)
}


#Solve argmax tr(PQ)
#note: lp.assign(C) optimizes c_{i,j} X_{i,j} which is equal to tr(C^T X) = tr(X C^T)
solve_lap<-function(C){
  lp = lp.assign(t(C), direction = "max")
  return(lp$solution)
}

# Test function to compare outputs of solve_lap and solve_lap_LSAP
test_solve_lap_functions <- function() {
  set.seed(42)
  for (i in 1:10) {
    problem_size <- sample(2:10, 1)
    cost_matrix <- matrix(runif(problem_size^2, min = -10, max = 10), nrow = problem_size)
    
    original_solution <- solve_lap(cost_matrix)
    new_solution <- solve_lap_LSAP(cost_matrix)
    
    if (!isTRUE(all.equal(original_solution, new_solution, tolerance = 1e-9))) {
      cat("Mismatch in test", i, "\n")
      cat("Original solution:\n")
      print(original_solution)
      cat("New solution:\n")
      print(new_solution)
    } else {
      cat("Test", i, "passed\n")
    }
  }
}

# Run the test suite
test_solve_lap_functions()



# Set the size of the problem (e.g., 100 x 100 cost matrix)
problem_size <- 100

# Create a random cost matrix
set.seed(42)
cost_matrix <- matrix(runif(problem_size^2, min = 0, max = 100), nrow = problem_size)

# Benchmark the performance of solve_LSAP and lp.assign
benchmark_results <- microbenchmark(
  solve_LSAP = {
    solve_LSAP(cost_matrix)
  },
  lp_assign = {
    lp.assign(cost_matrix)
  },
  times = 200
)

print(benchmark_results)
