library(Matrix)

# Function to compute block sums of a sparse adjacency matrix A
block_sums_gpt4 <- function(A, z) {
  if (!inherits(A, "sparseMatrix")) {
    stop("A must be a sparse matrix.")
  }
  
  if (nrow(A) != length(z)) {
    stop("The length of z must be equal to the number of rows in A.")
  }

  
  # Create an empty sparse matrix with dimensions equal to the number of unique groups in z
  unique_groups <- sort(unique(z))
  num_groups <- length(unique_groups)
  B <- Matrix(0, num_groups, num_groups, sparse = TRUE)
  
  # Compute block sums
  for (i in 1:num_groups) {
    for (j in 1:num_groups) {
      # Indices of rows/columns corresponding to the current pair of groups
      group_i_indices <- which(z == unique_groups[i])
      group_j_indices <- which(z == unique_groups[j])
      
      # Subset the original matrix and compute the sum of its elements
      sub_matrix <- A[group_i_indices, group_j_indices]
      
      if (isSymmetric(A) && i == j) {
        # If the matrix is symmetric and we are on a diagonal block, sum over the upper diagonal elements, including the diagonal
        # upper_tri_indices <- lower.tri(sub_matrix, diag = TRUE). # This give the warning
        dense_sub_matrix <- as.matrix(sub_matrix)
        upper_tri_indices <- which(row(dense_sub_matrix) <= col(dense_sub_matrix), arr.ind = TRUE)
        B[i, j] <- sum(sub_matrix[upper_tri_indices])
        
        B[i, j] <- sum(sub_matrix[upper_tri_indices])
      } else {
        B[i, j] <- sum(sub_matrix)
      }
    }
  }
  
  return(B)
}

# # Example usage
# # A <- Matrix(c(0, 1, 0, 1, 0, 1, 0, 1, 0), nrow = 3, sparse = TRUE)
# A <- sparseMatrix(i = c(2, 1, 3, 2),
#                   j = c(1, 2, 2, 3),
#                   x = c(1, 1, 1, 1),
#                   dims = c(3, 3))
# 
# z <- c(1, 2, 1)
#  
# print(block_sums_gpt4(A, z))
# block_sums(A, z)
