import torch
import numpy as np
import random
from transformers import set_seed


def fix_seeds(seed):
    # random
    random.seed(seed)
    # Numpy
    np.random.seed(seed)
    # Pytorch
    torch.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False
    # Huggingface
    set_seed(seed)


def print_trainable_parameters(model):
    """
    Prints the number of trainable parameters in the model.
    """
    trainable_params = 0
    all_param = 0
    for _, param in model.named_parameters():
        all_param += param.numel()
        if param.requires_grad:
            trainable_params += param.numel()
    print(
        f"trainable params: {trainable_params} || all params: {all_param} || trainable%: {100 * trainable_params / all_param}"
    )
    

def gpu_memory():
    """
    Return the GPU memory usage and total memory available in GB for all available GPUs.
    """
    gpu_memory_used = 0
    gpu_memory_total = 0

    for i in range(torch.cuda.device_count()):
        gpu_memory_used += torch.cuda.memory_allocated(i)
        gpu_memory_total += torch.cuda.get_device_properties(i).total_memory

    return gpu_memory_used / 1024**3, gpu_memory_total / 1024**3