import os, requests
import json
from pathlib import Path
from typing import Optional, Callable

from torch import Tensor
from torch.utils.data import Dataset
from torch.utils.data import DataLoader, Subset
import lightning as L

from torchvision.datasets import CIFAR10, Food101, OxfordIIITPet, DTD, EuroSAT, Caltech101
import torch
from datasets import load_dataset

import os
from PIL import Image
import sys
import os
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

imagenet_classes = [
    "tench", "goldfish", "great white shark", "tiger shark", "hammerhead shark", "electric ray", "stingray", "rooster", "hen", "ostrich", "brambling", "goldfinch", "house finch", "junco", "indigo bunting", "American robin", "bulbul", "jay", "magpie", "chickadee", "American dipper", "kite (bird of prey)", "bald eagle", "vulture", "great grey owl", "fire salamander", "smooth newt", "newt", "spotted salamander", "axolotl", "American bullfrog", "tree frog", "tailed frog", "loggerhead sea turtle", "leatherback sea turtle", "mud turtle", "terrapin", "box turtle", "banded gecko", "green iguana", "Carolina anole", "desert grassland whiptail lizard", "agama", "frilled-necked lizard", "alligator lizard", "Gila monster", "European green lizard", "chameleon", "Komodo dragon", "Nile crocodile", "American alligator", "triceratops", "worm snake", "ring-necked snake", "eastern hog-nosed snake", "smooth green snake", "kingsnake", "garter snake", "water snake", "vine snake", "night snake", "boa constrictor", "African rock python", "Indian cobra", "green mamba", "sea snake", "Saharan horned viper", "eastern diamondback rattlesnake", "sidewinder rattlesnake", "trilobite", "harvestman", "scorpion", "yellow garden spider", "barn spider", "European garden spider", "southern black widow", "tarantula", "wolf spider", "tick", "centipede", "black grouse", "ptarmigan", "ruffed grouse", "prairie grouse", "peafowl", "quail", "partridge", "african grey parrot", "macaw", "sulphur-crested cockatoo", "lorikeet", "coucal", "bee eater", "hornbill", "hummingbird", "jacamar", "toucan", "duck", "red-breasted merganser", "goose", "black swan", "tusker", "echidna", "platypus", "wallaby", "koala", "wombat", "jellyfish", "sea anemone", "brain coral", "flatworm", "nematode", "conch", "snail", "slug", "sea slug", "chiton", "chambered nautilus", "Dungeness crab", "rock crab", "fiddler crab", "red king crab", "American lobster", "spiny lobster", "crayfish", "hermit crab", "isopod", "white stork", "black stork", "spoonbill", "flamingo", "little blue heron", "great egret", "bittern bird", "crane bird", "limpkin", "common gallinule", "American coot", "bustard", "ruddy turnstone", "dunlin", "common redshank", "dowitcher", "oystercatcher", "pelican", "king penguin", "albatross", "grey whale", "killer whale", "dugong", "sea lion", "Chihuahua", "Japanese Chin", "Maltese", "Pekingese", "Shih Tzu", "King Charles Spaniel", "Papillon", "toy terrier", "Rhodesian Ridgeback", "Afghan Hound", "Basset Hound", "Beagle", "Bloodhound", "Bluetick Coonhound", "Black and Tan Coonhound", "Treeing Walker Coonhound", "English foxhound", "Redbone Coonhound", "borzoi", "Irish Wolfhound", "Italian Greyhound", "Whippet", "Ibizan Hound", "Norwegian Elkhound", "Otterhound", "Saluki", "Scottish Deerhound", "Weimaraner", "Staffordshire Bull Terrier", "American Staffordshire Terrier", "Bedlington Terrier", "Border Terrier", "Kerry Blue Terrier", "Irish Terrier", "Norfolk Terrier", "Norwich Terrier", "Yorkshire Terrier", "Wire Fox Terrier", "Lakeland Terrier", "Sealyham Terrier", "Airedale Terrier", "Cairn Terrier", "Australian Terrier", "Dandie Dinmont Terrier", "Boston Terrier", "Miniature Schnauzer", "Giant Schnauzer", "Standard Schnauzer", "Scottish Terrier", "Tibetan Terrier", "Australian Silky Terrier", "Soft-coated Wheaten Terrier", "West Highland White Terrier", "Lhasa Apso", "Flat-Coated Retriever", "Curly-coated Retriever", "Golden Retriever", "Labrador Retriever", "Chesapeake Bay Retriever", "German Shorthaired Pointer", "Vizsla", "English Setter", "Irish Setter", "Gordon Setter", "Brittany dog", "Clumber Spaniel", "English Springer Spaniel", "Welsh Springer Spaniel", "Cocker Spaniel", "Sussex Spaniel", "Irish Water Spaniel", "Kuvasz", "Schipperke", "Groenendael dog", "Malinois", "Briard", "Australian Kelpie", "Komondor", "Old English Sheepdog", "Shetland Sheepdog", "collie", "Border Collie", "Bouvier des Flandres dog", "Rottweiler", "German Shepherd Dog", "Dobermann", "Miniature Pinscher", "Greater Swiss Mountain Dog", "Bernese Mountain Dog", "Appenzeller Sennenhund", "Entlebucher Sennenhund", "Boxer", "Bullmastiff", "Tibetan Mastiff", "French Bulldog", "Great Dane", "St. Bernard", "husky", "Alaskan Malamute", "Siberian Husky", "Dalmatian", "Affenpinscher", "Basenji", "pug", "Leonberger", "Newfoundland dog", "Great Pyrenees dog", "Samoyed", "Pomeranian", "Chow Chow", "Keeshond", "brussels griffon", "Pembroke Welsh Corgi", "Cardigan Welsh Corgi", "Toy Poodle", "Miniature Poodle", "Standard Poodle", "Mexican hairless dog (xoloitzcuintli)", "grey wolf", "Alaskan tundra wolf", "red wolf or maned wolf", "coyote", "dingo", "dhole", "African wild dog", "hyena", "red fox", "kit fox", "Arctic fox", "grey fox", "tabby cat", "tiger cat", "Persian cat", "Siamese cat", "Egyptian Mau", "cougar", "lynx", "leopard", "snow leopard", "jaguar", "lion", "tiger", "cheetah", "brown bear", "American black bear", "polar bear", "sloth bear", "mongoose", "meerkat", "tiger beetle", "ladybug", "ground beetle", "longhorn beetle", "leaf beetle", "dung beetle", "rhinoceros beetle", "weevil", "fly", "bee", "ant", "grasshopper", "cricket insect", "stick insect", "cockroach", "praying mantis", "cicada", "leafhopper", "lacewing", "dragonfly", "damselfly", "red admiral butterfly", "ringlet butterfly", "monarch butterfly", "small white butterfly", "sulphur butterfly", "gossamer-winged butterfly", "starfish", "sea urchin", "sea cucumber", "cottontail rabbit", "hare", "Angora rabbit", "hamster", "porcupine", "fox squirrel", "marmot", "beaver", "guinea pig", "common sorrel horse", "zebra", "pig", "wild boar", "warthog", "hippopotamus", "ox", "water buffalo", "bison", "ram (adult male sheep)", "bighorn sheep", "Alpine ibex", "hartebeest", "impala (antelope)", "gazelle", "arabian camel", "llama", "weasel", "mink", "European polecat", "black-footed ferret", "otter", "skunk", "badger", "armadillo", "three-toed sloth", "orangutan", "gorilla", "chimpanzee", "gibbon", "siamang", "guenon", "patas monkey", "baboon", "macaque", "langur", "black-and-white colobus", "proboscis monkey", "marmoset", "white-headed capuchin", "howler monkey", "titi monkey", "Geoffroy's spider monkey", "common squirrel monkey", "ring-tailed lemur", "indri", "Asian elephant", "African bush elephant", "red panda", "giant panda", "snoek fish", "eel", "silver salmon", "rock beauty fish", "clownfish", "sturgeon", "gar fish", "lionfish", "pufferfish", "abacus", "abaya", "academic gown", "accordion", "acoustic guitar", "aircraft carrier", "airliner", "airship", "altar", "ambulance", "amphibious vehicle", "analog clock", "apiary", "apron", "trash can", "assault rifle", "backpack", "bakery", "balance beam", "balloon", "ballpoint pen", "Band-Aid", "banjo", "baluster / handrail", "barbell", "barber chair", "barbershop", "barn", "barometer", "barrel", "wheelbarrow", "baseball", "basketball", "bassinet", "bassoon", "swimming cap", "bath towel", "bathtub", "station wagon", "lighthouse", "beaker", "military hat (bearskin or shako)", "beer bottle", "beer glass", "bell tower", "baby bib", "tandem bicycle", "bikini", "ring binder", "binoculars", "birdhouse", "boathouse", "bobsleigh", "bolo tie", "poke bonnet", "bookcase", "bookstore", "bottle cap", "hunting bow", "bow tie", "brass memorial plaque", "bra", "breakwater", "breastplate", "broom", "bucket", "buckle", "bulletproof vest", "high-speed train", "butcher shop", "taxicab", "cauldron", "candle", "cannon", "canoe", "can opener", "cardigan", "car mirror", "carousel", "tool kit", "cardboard box / carton", "car wheel", "automated teller machine", "cassette", "cassette player", "castle", "catamaran", "CD player", "cello", "mobile phone", "chain", "chain-link fence", "chain mail", "chainsaw", "storage chest", "chiffonier", "bell or wind chime", "china cabinet", "Christmas stocking", "church", "movie theater", "cleaver", "cliff dwelling", "cloak", "clogs", "cocktail shaker", "coffee mug", "coffeemaker", "spiral or coil", "combination lock", "computer keyboard", "candy store", "container ship", "convertible", "corkscrew", "cornet", "cowboy boot", "cowboy hat", "cradle", "construction crane", "crash helmet", "crate", "infant bed", "Crock Pot", "croquet ball", "crutch", "cuirass", "dam", "desk", "desktop computer", "rotary dial telephone", "diaper", "digital clock", "digital watch", "dining table", "dishcloth", "dishwasher", "disc brake", "dock", "dog sled", "dome", "doormat", "drilling rig", "drum", "drumstick", "dumbbell", "Dutch oven", "electric fan", "electric guitar", "electric locomotive", "entertainment center", "envelope", "espresso machine", "face powder", "feather boa", "filing cabinet", "fireboat", "fire truck", "fire screen", "flagpole", "flute", "folding chair", "football helmet", "forklift", "fountain", "fountain pen", "four-poster bed", "freight car", "French horn", "frying pan", "fur coat", "garbage truck", "gas mask or respirator", "gas pump", "goblet", "go-kart", "golf ball", "golf cart", "gondola", "gong", "gown", "grand piano", "greenhouse", "radiator grille", "grocery store", "guillotine", "hair clip", "hair spray", "half-track", "hammer", "hamper", "hair dryer", "hand-held computer", "handkerchief", "hard disk drive", "harmonica", "harp", "combine harvester", "hatchet", "holster", "home theater", "honeycomb", "hook", "hoop skirt", "gymnastic horizontal bar", "horse-drawn vehicle", "hourglass", "iPod", "clothes iron", "carved pumpkin", "jeans", "jeep", "T-shirt", "jigsaw puzzle", "rickshaw", "joystick", "kimono", "knee pad", "knot", "lab coat", "ladle", "lampshade", "laptop computer", "lawn mower", "lens cap", "letter opener", "library", "lifeboat", "lighter", "limousine", "ocean liner", "lipstick", "slip-on shoe", "lotion", "music speaker", "loupe magnifying glass", "sawmill", "magnetic compass", "messenger bag", "mailbox", "tights", "one-piece bathing suit", "manhole cover", "maraca", "marimba", "mask", "matchstick", "maypole", "maze", "measuring cup", "medicine cabinet", "megalith", "microphone", "microwave oven", "military uniform", "milk can", "minibus", "miniskirt", "minivan", "missile", "mitten", "mixing bowl", "mobile home", "ford model t", "modem", "monastery", "monitor", "moped", "mortar and pestle", "graduation cap", "mosque", "mosquito net", "vespa", "mountain bike", "tent", "computer mouse", "mousetrap", "moving van", "muzzle", "metal nail", "neck brace", "necklace", "baby pacifier", "notebook computer", "obelisk", "oboe", "ocarina", "odometer", "oil filter", "pipe organ", "oscilloscope", "overskirt", "bullock cart", "oxygen mask", "product packet / packaging", "paddle", "paddle wheel", "padlock", "paintbrush", "pajamas", "palace", "pan flute", "paper towel", "parachute", "parallel bars", "park bench", "parking meter", "railroad car", "patio", "payphone", "pedestal", "pencil case", "pencil sharpener", "perfume", "Petri dish", "photocopier", "plectrum", "Pickelhaube", "picket fence", "pickup truck", "pier", "piggy bank", "pill bottle", "pillow", "ping-pong ball", "pinwheel", "pirate ship", "drink pitcher", "block plane", "planetarium", "plastic bag", "plate rack", "farm plow", "plunger", "Polaroid camera", "pole", "police van", "poncho", "pool table", "soda bottle", "plant pot", "potter's wheel", "power drill", "prayer rug", "printer", "prison", "missile", "projector", "hockey puck", "punching bag", "purse", "quill", "quilt", "race car", "racket", "radiator", "radio", "radio telescope", "rain barrel", "recreational vehicle", "fishing casting reel", "reflex camera", "refrigerator", "remote control", "restaurant", "revolver", "rifle", "rocking chair", "rotisserie", "eraser", "rugby ball", "ruler measuring stick", "sneaker", "safe", "safety pin", "salt shaker", "sandal", "sarong", "saxophone", "scabbard", "weighing scale", "school bus", "schooner", "scoreboard", "CRT monitor", "screw", "screwdriver", "seat belt", "sewing machine", "shield", "shoe store", "shoji screen / room divider", "shopping basket", "shopping cart", "shovel", "shower cap", "shower curtain", "ski", "balaclava ski mask", "sleeping bag", "slide rule", "sliding door", "slot machine", "snorkel", "snowmobile", "snowplow", "soap dispenser", "soccer ball", "sock", "solar thermal collector", "sombrero", "soup bowl", "keyboard space bar", "space heater", "space shuttle", "spatula", "motorboat", "spider web", "spindle", "sports car", "spotlight", "stage", "steam locomotive", "through arch bridge", "steel drum", "stethoscope", "scarf", "stone wall", "stopwatch", "stove", "strainer", "tram", "stretcher", "couch", "stupa", "submarine", "suit", "sundial", "sunglasses", "sunglasses", "sunscreen", "suspension bridge", "mop", "sweatshirt", "swim trunks / shorts", "swing", "electrical switch", "syringe", "table lamp", "tank", "tape player", "teapot", "teddy bear", "television", "tennis ball", "thatched roof", "front curtain", "thimble", "threshing machine", "throne", "tile roof", "toaster", "tobacco shop", "toilet seat", "torch", "totem pole", "tow truck", "toy store", "tractor", "semi-trailer truck", "tray", "trench coat", "tricycle", "trimaran", "tripod", "triumphal arch", "trolleybus", "trombone", "hot tub", "turnstile", "typewriter keyboard", "umbrella", "unicycle", "upright piano", "vacuum cleaner", "vase", "vaulted or arched ceiling", "velvet fabric", "vending machine", "vestment", "viaduct", "violin", "volleyball", "waffle iron", "wall clock", "wallet", "wardrobe", "military aircraft", "sink", "washing machine", "water bottle", "water jug", "water tower", "whiskey jug", "whistle", "hair wig", "window screen", "window shade", "Windsor tie", "wine bottle", "airplane wing", "wok", "wooden spoon", "wool", "split-rail fence", "shipwreck", "sailboat", "yurt", "website", "comic book", "crossword", "traffic or street sign", "traffic light", "dust jacket", "menu", "plate", "guacamole", "consomme", "hot pot", "trifle", "ice cream", "popsicle", "baguette", "bagel", "pretzel", "cheeseburger", "hot dog", "mashed potatoes", "cabbage", "broccoli", "cauliflower", "zucchini", "spaghetti squash", "acorn squash", "butternut squash", "cucumber", "artichoke", "bell pepper", "cardoon", "mushroom", "Granny Smith apple", "strawberry", "orange", "lemon", "fig", "pineapple", "banana", "jackfruit", "cherimoya (custard apple)", "pomegranate", "hay", "carbonara", "chocolate syrup", "dough", "meatloaf", "pizza", "pot pie", "burrito", "red wine", "espresso", "tea cup", "eggnog", "mountain", "bubble", "cliff", "coral reef", "geyser", "lakeshore", "promontory", "sandbar", "beach", "valley", "volcano", "baseball player", "bridegroom", "scuba diver", "rapeseed", "daisy", "yellow lady's slipper", "corn", "acorn", "rose hip", "horse chestnut seed", "coral fungus", "agaric", "gyromitra", "stinkhorn mushroom", "earth star fungus", "hen of the woods mushroom", "bolete", "corn cob", "toilet paper"]

cifar100_classes = [
    'apple', 'aquarium_fish', 'baby', 'bear', 'beaver', 'bed', 'bee', 'beetle', 'bicycle', 
    'bottle', 'bowl', 'boy', 'bridge', 'bus', 'butterfly', 'camel', 'can', 'castle', 'caterpillar', 
    'cattle', 'chair', 'chimpanzee', 'clock', 'cloud', 'cockroach', 'couch', 'crab', 'crocodile', 'cup', 
    'dinosaur', 'dolphin', 'elephant', 'flatfish', 'forest', 'fox', 'girl', 'hamster', 'house', 'kangaroo', 
    'keyboard', 'lamp', 'lawn_mower', 'leopard', 'lion', 'lizard', 'lobster', 'man', 'maple_tree', 'motorcycle', 
    'mountain', 'mouse', 'mushroom', 'oak_tree', 'orange', 'orchid', 'otter', 'palm_tree', 'pear', 'pickup_truck', 
    'pine_tree', 'plain', 'plate', 'poppy', 'porcupine', 'possum', 'rabbit', 'raccoon', 'ray', 'road', 'rocket', 
    'rose', 'sea', 'seal', 'shark', 'shrew', 'skunk', 'skyscraper', 'snail', 'snake', 'spider', 'squirrel', 
    'streetcar', 'sunflower', 'sweet_pepper', 'table', 'tank', 'telephone', 'television', 'tiger', 'tractor', 
    'train', 'trout', 'tulip', 'turtle', 'wardrobe', 'whale', 'willow_tree', 'wolf', 'woman', 'worm'
    ]

cifar10_classes = [
    'airplane', 'automobile', 'bird', 'cat',
    'deer', 'dog', 'frog', 'horse',
    'ship', 'truck'
    ]

food101_classes = [
    'apple pie', 'baby back ribs', 'baklava', 'beef carpaccio', 'beef tartare', 'beet salad',
    'beignets', 'bibimbap', 'bread pudding', 'breakfast burrito', 'bruschetta', 'caesar salad',
    'cannoli', 'caprese salad', 'carrot cake', 'ceviche', 'cheese plate', 'cheesecake',
    'chicken curry', 'chicken quesadilla', 'chicken wings', 'chocolate cake', 'chocolate mousse', 'churros',
    'clam chowder', 'club sandwich', 'crab cakes', 'creme brulee', 'croque madame', 'cup cakes',
    'deviled eggs', 'donuts', 'dumplings', 'edamame', 'eggs benedict', 'escargots',
    'falafel', 'filet mignon', 'fish and chips', 'foie gras', 'french fries', 'french onion soup',
    'french toast', 'fried calamari', 'fried rice', 'frozen yogurt', 'garlic bread', 'gnocchi',
    'greek salad', 'grilled cheese sandwich', 'grilled salmon', 'guacamole', 'gyoza', 'hamburger',
    'hot and sour soup', 'hot dog', 'huevos rancheros', 'hummus', 'ice cream', 'lasagna',
    'lobster bisque', 'lobster roll sandwich', 'macaroni and cheese', 'macarons', 'miso soup', 'mussels',
    'nachos', 'omelette', 'onion rings', 'oysters', 'pad thai', 'paella',
    'pancakes', 'panna cotta', 'peking duck', 'pho', 'pizza', 'pork chop',
    'poutine', 'prime rib', 'pulled pork sandwich', 'ramen', 'ravioli', 'red velvet cake',
    'risotto', 'samosa', 'sashimi', 'scallops', 'seaweed salad', 'shrimp and grits',
    'spaghetti bolognese', 'spaghetti carbonara', 'spring rolls', 'steak', 'strawberry shortcake', 'sushi',
    'tacos', 'takoyaki', 'tiramisu', 'tuna tartare', 'waffles'
] 

sun397_classes = [
    'abbey', 'airplane cabin', 'airport terminal', 'alley', 'amphitheater', 'amusement arcade',
    'amusement park', 'anechoic chamber', 'apartment building outdoor', 'apse indoor', 'aquarium', 'aqueduct',
    'arch', 'archive', 'arrival gate outdoor', 'art gallery', 'art school', 'art studio',
    'assembly line', 'athletic field outdoor', 'atrium public', 'attic', 'auditorium', 'auto factory',
    'badlands', 'badminton court indoor', 'baggage claim', 'bakery shop', 'balcony exterior', 'balcony interior',
    'ball pit', 'ballroom', 'bamboo forest', 'banquet hall', 'bar', 'barn',
    'barndoor', 'baseball field', 'basement', 'basilica', 'basketball court outdoor', 'bathroom',
    'batters box', 'bayou', 'bazaar indoor', 'bazaar outdoor', 'beach', 'beauty salon',
    'bedroom', 'berth', 'biology laboratory', 'bistro indoor', 'boardwalk', 'boat deck',
    'boathouse', 'bookstore', 'booth indoor', 'botanical garden', 'bow window indoor', 'bow window outdoor',
    'bowling alley', 'boxing ring', 'brewery indoor', 'bridge', 'building facade', 'bullring',
    'burial chamber', 'bus interior', 'butchers shop', 'butte', 'cabin outdoor', 'cafeteria',
    'campsite', 'campus', 'canal natural', 'canal urban', 'candy store', 'canyon',
    'car interior backseat', 'car interior frontseat', 'carrousel', 'casino indoor', 'castle', 'catacomb',
    'cathedral indoor', 'cathedral outdoor', 'cavern indoor', 'cemetery', 'chalet', 'cheese factory',
    'chemistry lab', 'chicken coop indoor', 'chicken coop outdoor', 'childs room', 'church indoor', 'church outdoor',
    'classroom', 'clean room', 'cliff', 'cloister indoor', 'closet', 'clothing store',
    'coast', 'cockpit', 'coffee shop', 'computer room', 'conference center', 'conference room',
    'construction site', 'control room', 'control tower outdoor', 'corn field', 'corral', 'corridor',
    'cottage garden', 'courthouse', 'courtroom', 'courtyard', 'covered bridge exterior', 'creek',
    'crevasse', 'crosswalk', 'cubicle office', 'dam', 'delicatessen', 'dentists office',
    'desert sand', 'desert vegetation', 'diner indoor', 'diner outdoor', 'dinette home', 'dinette vehicle',
    'dining car', 'dining room', 'discotheque', 'dock', 'doorway outdoor', 'dorm room',
    'driveway', 'driving range outdoor', 'drugstore', 'electrical substation', 'elevator door', 'elevator interior',
    'elevator shaft', 'engine room', 'escalator indoor', 'excavation', 'factory indoor', 'fairway',
    'fastfood restaurant', 'field cultivated', 'field wild', 'fire escape', 'fire station', 'firing range indoor',
    'fishpond', 'florist shop indoor', 'food court', 'forest broadleaf', 'forest needleleaf', 'forest path',
    'forest road', 'formal garden', 'fountain', 'galley', 'game room', 'garage indoor',
    'garbage dump', 'gas station', 'gazebo exterior', 'general store indoor', 'general store outdoor', 'gift shop',
    'golf course', 'greenhouse indoor', 'greenhouse outdoor', 'gymnasium indoor', 'hangar indoor', 'hangar outdoor',
    'harbor', 'hayfield', 'heliport', 'herb garden', 'highway', 'hill',
    'home office', 'hospital', 'hospital room', 'hot spring', 'hot tub outdoor', 'hotel outdoor',
    'hotel room', 'house', 'hunting lodge outdoor', 'ice cream parlor', 'ice floe', 'ice shelf',
    'ice skating rink indoor', 'ice skating rink outdoor', 'iceberg', 'igloo', 'industrial area', 'inn outdoor',
    'islet', 'jacuzzi indoor', 'jail cell', 'jail indoor', 'jewelry shop', 'kasbah',
    'kennel indoor', 'kennel outdoor', 'kindergarden classroom', 'kitchen', 'kitchenette', 'labyrinth outdoor',
    'lake natural', 'landfill', 'landing deck', 'laundromat', 'lecture room', 'library indoor',
    'library outdoor', 'lido deck outdoor', 'lift bridge', 'lighthouse', 'limousine interior', 'living room',
    'lobby', 'lock chamber', 'locker room', 'mansion', 'manufactured home', 'market indoor',
    'market outdoor', 'marsh', 'martial arts gym', 'mausoleum', 'medina', 'moat water',
    'monastery outdoor', 'mosque indoor', 'mosque outdoor', 'motel', 'mountain', 'mountain snowy',
    'movie theater indoor', 'museum indoor', 'music store', 'music studio', 'nuclear power plant outdoor', 'nursery',
    'oast house', 'observatory outdoor', 'ocean', 'office', 'office building', 'oil refinery outdoor',
    'oilrig', 'operating room', 'orchard', 'outhouse outdoor', 'pagoda', 'palace',
    'pantry', 'park', 'parking garage indoor', 'parking garage outdoor', 'parking lot', 'parlor',
    'pasture', 'patio', 'pavilion', 'pharmacy', 'phone booth', 'physics laboratory',
    'picnic area', 'pilothouse indoor', 'planetarium outdoor', 'playground', 'playroom', 'plaza',
    'podium indoor', 'podium outdoor', 'pond', 'poolroom establishment', 'poolroom home', 'power plant outdoor',
    'promenade deck', 'pub indoor', 'pulpit', 'putting green', 'racecourse', 'raceway',
    'raft', 'railroad track', 'rainforest', 'reception', 'recreation room', 'residential neighborhood',
    'restaurant', 'restaurant kitchen', 'restaurant patio', 'rice paddy', 'riding arena', 'river',
    'rock arch', 'rope bridge', 'ruin', 'runway', 'sandbar', 'sandbox',
    'sauna', 'schoolhouse', 'sea cliff', 'server room', 'shed', 'shoe shop',
    'shopfront', 'shopping mall indoor', 'shower', 'skatepark', 'ski lodge', 'ski resort',
    'ski slope', 'sky', 'skyscraper', 'slum', 'snowfield', 'squash court',
    'stable', 'stadium baseball', 'stadium football', 'stage indoor', 'staircase', 'street',
    'subway interior', 'subway station platform', 'supermarket', 'sushi bar', 'swamp', 'swimming pool indoor',
    'swimming pool outdoor', 'synagogue indoor', 'synagogue outdoor', 'television studio', 'temple east asia', 'temple south asia',
    'tennis court indoor', 'tennis court outdoor', 'tent outdoor', 'theater indoor procenium', 'theater indoor seats', 'thriftshop',
    'throne room', 'ticket booth', 'toll plaza', 'topiary garden', 'tower', 'toyshop',
    'track outdoor', 'train railway', 'train station platform', 'tree farm', 'tree house', 'trench',
    'underwater coral reef', 'utility room', 'valley', 'van interior', 'vegetable garden', 'veranda',
    'veterinarians office', 'viaduct', 'videostore', 'village', 'vineyard', 'volcano',
    'volleyball court indoor', 'volleyball court outdoor', 'waiting room', 'warehouse indoor', 'water tower', 'waterfall block',
    'waterfall fan', 'waterfall plunge', 'watering hole', 'wave', 'wet bar', 'wheat field',
    'wind farm', 'windmill', 'wine cellar barrel storage', 'wine cellar bottle storage', 'wrestling ring indoor', 'yard',
    'youth hostel'
]

oxford_pet_classes = [
    'Abyssinian', 'American Bulldog', 'American Pit Bull Terrier',
    'Basset Hound', 'Beagle', 'Bengal', 'Birman', 'Bombay', 'Boxer',
    'British Shorthair', 'Chihuahua', 'Egyptian Mau', 'English Cocker Spaniel',
    'English Setter', 'German Shorthaired', 'Great Pyrenees',
    'Havanese', 'Japanese Chin', 'Keeshond', 'Leonberger',
    'Maine Coon', 'Miniature Pinscher', 'Newfoundland',
    'Persian', 'Pomeranian', 'Pug', 'Ragdoll', 'Russian Blue',
    'Saint Bernard', 'Samoyed', 'Scottish Terrier',
    'Shiba Inu', 'Siamese', 'Sphynx', 'Staffordshire Bull Terrier', 'Wheaten Terrier',
    'Yorkshire Terrier'
]

dtd_classes = [
    'banded','blotchy', 'braided', 'bubbly', 'bumpy',
    'chequered', 'cobwebbed', 'cracked', 'crosshatched',
    'crystalline', 'dotted', 'fibrous', 'flecked',
    'freckled', 'frilly', 'gauzy', 'grid', 'grooved',
    'honeycombed', 'interlaced', 'knitted', 'lacelike',
    'lined', 'marbled', 'matted', 'meshed',
    'paisley', 'perforated', 'pitted',
    'pleated', 'polka-dotted', 'porous', 'potholed',
    'scaly', 'smeared', 'spiralled', 'sprinkled', 'stained',
    'stratified', 'striped', 'studded', 'swirly', 'veined',
    'waffled', 'woven', 'wrinkled', 'zigzagged'
]

eurosat_classes = [
    'AnnualCrop', 'Forest', 'HerbaceousVegetation', 'Highway', 'Industrial',
    'Pasture', 'PermanentCrop', 'Residential', 'River', 'SeaLake'
]

caltech101_classes = [
    'Faces','Faces_easy','Leopards','Motorbikes','accordion','airplanes',
    'anchor','ant','barrel','bass','beaver','binocular','bonsai','brain',
    'brontosaurus','buddha','butterfly','camera','cannon','car_side',
    'ceiling_fan','cellphone','chair','chandelier','cougar_body','cougar_face',
    'crab','crayfish','crocodile','crocodile_head','cup','dalmatian','dollar_bill',
    'dolphin','dragonfly','electric_guitar','elephant','emu','euphonium',
    'ewer','ferry','flamingo','flamingo_head','garfield','gerenuk',
    'gramophone','grand_piano','hawksbill','headphone','hedgehog','helicopter',
    'ibis','inline_skate','joshua_tree','kangaroo','ketch','lamp','laptop','llama',
    'lobster','lotus','mandolin','mayfly','menorah','metronome','minaret','nautilus','octopus',
    'okapi','pagoda','panda','pigeon','pizza','platypus','pyramid','revolver','rhino',
    'rooster','saxophone','schooner','scissors','scorpion','sea_horse','snoopy',
    'soccer_ball','stapler','starfish','stegosaurus','stop_sign','strawberry',
    'sunflower','tick','trilobite','umbrella','watch','water_lilly','wheelchair',
    'wild_cat','windsor_chair','wrench','yin_yang'
    ]

class CIFAR10DataModule(L.LightningDataModule):

    def __init__(self, transform, data_dir="data/cifar10", batch_size=64, num_workers=4, seed=42, subset_size=1000):
        super().__init__()
        self.data_dir = data_dir
        self.batch_size = batch_size
        self.num_workers = num_workers
        self.seed = seed
        self.subset_size = subset_size

        self.transform = transform

    def prepare_data(self):
        # download only
        CIFAR10(root=self.data_dir, train=True, download=True)
        CIFAR10(root=self.data_dir, train=False, download=True)

    def setup(self, stage=None):
        # Set deterministic seed for subset selection
        torch.manual_seed(self.seed)
        if stage in (None, "fit"):
            train = CIFAR10(root=self.data_dir, train=True, transform=self.transform)
            if self.subset_size:
                all_indices = torch.randperm(len(train))[:self.subset_size]
                train = Subset(train, all_indices.tolist())
            self.train = train
        if stage in (None, "test"):
            self.test = CIFAR10(root=self.data_dir, train=False, transform=self.transform)

    def train_dataloader(self):
        return DataLoader(
            self.train,
            batch_size=self.batch_size,
            shuffle=True,  # shuffle within subset
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )
    
    def test_dataloader(self):
        return DataLoader(
            self.test,
            batch_size=self.batch_size,
            shuffle=False,
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )

    def val_dataloader(self):
        pass
    
    def get_indices(self):
        return {"test": 0}
    
class Food101DataModule(L.LightningDataModule):
    def __init__(
        self,
        transform,
        data_dir="data/food101",
        batch_size=64,
        num_workers=4,
        seed=42,
        train_subset_size=None,  # None = use full train set
    ):
        super().__init__()
        self.data_dir = data_dir
        self.batch_size = batch_size
        self.num_workers = num_workers
        self.seed = seed
        self.train_subset_size = train_subset_size

        self.transform = transform

    def prepare_data(self):
        # Downloads if not already present
        Food101(root=self.data_dir, split="train", download=True)
        Food101(root=self.data_dir, split="test", download=True)

    def setup(self, stage=None):
        # Load full splits
        if stage in (None, "fit"):
            train = Food101(root=self.data_dir, split="train", transform=self.transform)
            if self.train_subset_size is not None:
                torch.manual_seed(self.seed)
                indices = torch.randperm(len(train))[:self.train_subset_size]
                train = Subset(train, indices.tolist())
            self.train = train
        
        if stage in (None, "test"):
            self.test = Food101(root=self.data_dir, split="test", transform=self.transform)

    def train_dataloader(self):
        return DataLoader(
            self.train,
            batch_size=self.batch_size,
            shuffle=True,
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )

    def val_dataloader(self):
        pass

    def test_dataloader(self):
        return DataLoader(
            self.test,
            batch_size=self.batch_size,
            shuffle=False,
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )

    def get_indices(self):
        return {"test": 0}

class OxfordIIITPetDataModule(L.LightningDataModule):
    def __init__(
        self,
        transform,
        data_dir="data",
        batch_size=64,
        num_workers=4,
        seed=42,
        train_subset_size=None,  # None = use full train set
    ):
        super().__init__()
        self.data_dir = data_dir
        self.batch_size = batch_size
        self.num_workers = num_workers
        self.seed = seed
        self.train_subset_size = train_subset_size

        self.transform = transform

    def prepare_data(self):
        # Downloads if not already present
        OxfordIIITPet(root=self.data_dir, split="trainval", download=True)
        OxfordIIITPet(root=self.data_dir, split="test", download=True)

    def setup(self, stage=None):
        # Load full splits
        if stage in (None, "fit"):
            train = OxfordIIITPet(root=self.data_dir, split="trainval", transform=self.transform)
            if self.train_subset_size is not None:
                torch.manual_seed(self.seed)
                indices = torch.randperm(len(train))[:self.train_subset_size]
                train = Subset(train, indices.tolist())
            self.train = train
        
        if stage in (None, "test"):
            self.test = OxfordIIITPet(root=self.data_dir, split="test", transform=self.transform)

    def train_dataloader(self):
        return DataLoader(
            self.train,
            batch_size=self.batch_size,
            shuffle=True,
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )

    def val_dataloader(self):
        pass

    def test_dataloader(self):
        return DataLoader(
            self.test,
            batch_size=self.batch_size,
            shuffle=False,
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )

    def get_indices(self):
        return {"test": 0}

class DTDDataModule(L.LightningDataModule):
    def __init__(
        self,
        transform,
        data_dir="data",
        batch_size=64,
        num_workers=4,
        seed=42,
    ):
        super().__init__()
        self.data_dir = data_dir
        self.batch_size = batch_size
        self.num_workers = num_workers
        self.seed = seed

        self.transform = transform

    def prepare_data(self):
        # Downloads if not already present
        DTD(root=self.data_dir, split="train", download=True)
        DTD(root=self.data_dir, split="val", download=True)
        DTD(root=self.data_dir, split="test", download=True)

    def setup(self, stage=None):
        # Load full splits
        if stage in (None, "fit"):
            self.train = DTD(root=self.data_dir, split="train", transform=self.transform)
            self.val = DTD(root=self.data_dir, split="val", transform=self.transform)
        
        if stage in (None, "test"):
            self.test = DTD(root=self.data_dir, split="test", transform=self.transform)

    def train_dataloader(self):
        return DataLoader(
            self.train,
            batch_size=self.batch_size,
            shuffle=True,
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )

    def val_dataloader(self):
        return DataLoader(
            self.val,
            batch_size=self.batch_size,
            shuffle=False,
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )

    def test_dataloader(self):
        return DataLoader(
            self.test,
            batch_size=self.batch_size,
            shuffle=False,
            num_workers=self.num_workers,
            pin_memory=True,
            persistent_workers=False
        )

    def get_indices(self):
        return {"test": 0}

class SplitDataset(Dataset):
    def __init__(
        self,
        full_dataset,
        split: str,
        split_json: str,
        transform: Optional[Callable[[Image.Image], Tensor]] = None,
    ):
        assert split in ["train", "val", "test"]
        self.transform = transform

        # load json
        with open(split_json, "r") as f:
            db = json.load(f)
        # use requested split
        self.db = db[split]

        # Map full_dataset indices from JSON filenames
        self.full_dataset = full_dataset
        self.indices = []
        name_to_idx = {Path(path).name: i for i, (path, _) in enumerate(full_dataset.samples)}

        for x in self.db:
            file_name = Path(x[0]).name  # relative path in JSON
            if file_name in name_to_idx:
                self.indices.append(name_to_idx[file_name])
            else:
                raise ValueError(f"{file_name} not found.")

    def __len__(self):
        return len(self.indices)

    def __getitem__(self, idx):
        real_idx = self.indices[idx]
        img, target = self.full_dataset[real_idx]
        if self.transform:
            img = self.transform(img)
        return img, target
    
class EuroSATDataModule(L.LightningDataModule):
    def __init__(
        self,
        transform,
        data_dir: str = "data",
        batch_size: int = 64,
        num_workers: int = 4,
        seed: int = 42,
    ):
        super().__init__()
        self.data_dir = data_dir
        self.batch_size = batch_size
        self.num_workers = num_workers
        self.seed = seed
        self.transform = transform
        self.split_json = os.path.join(self.data_dir, "eurosat/split_zhou_EuroSAT.json")

    def prepare_data(self):
        # Download the full EuroSAT dataset
        EuroSAT(root=self.data_dir, download=True)

    def setup(self, stage=None):
        full_dataset = EuroSAT(root=self.data_dir, download=False)

        if stage in (None, "fit"):
            self.train = SplitDataset(full_dataset, split="train", split_json=self.split_json, transform=self.transform)
            self.val = SplitDataset(full_dataset, split="val", split_json=self.split_json, transform=self.transform)
        if stage in (None, "test"):
            self.test = SplitDataset(full_dataset, split="test", split_json=self.split_json, transform=self.transform)

    def train_dataloader(self):
        return DataLoader(
            self.train,
            batch_size=self.batch_size,
            shuffle=True,
            num_workers=self.num_workers,
            pin_memory=True,
        )

    def val_dataloader(self):
        return DataLoader(
            self.val,
            batch_size=self.batch_size,
            shuffle=False,
            num_workers=self.num_workers,
            pin_memory=True,
        )

    def test_dataloader(self):
        return DataLoader(
            self.test,
            batch_size=self.batch_size,
            shuffle=False,
            num_workers=self.num_workers,
            pin_memory=True,
        )
    
    def get_indices(self):
        return {"test": 0}

class Caltech101_(Caltech101):
    def __init__(self, root, **kwargs):
        super().__init__(root, **kwargs)

        self.samples = []
        base_dir = os.path.join(self.root, "101_ObjectCategories")

        # Map category -> index
        cat_to_idx = {cat: i for i, cat in enumerate(self.categories)}

        for cat in self.categories:
            class_dir = os.path.join(base_dir, cat) 
            if not os.path.isdir(class_dir):
                continue
            for fname in os.listdir(class_dir):
                if fname.lower().endswith((".jpg", ".jpeg", ".png")):
                    path = os.path.join(class_dir, fname)
                    target = cat_to_idx[cat]
                    self.samples.append((str(path), target))

class Caltech101DataModule(L.LightningDataModule):
    def __init__(
        self,
        transform,
        data_dir: str = "data",
        batch_size: int = 64,
        num_workers: int = 4,
        seed: int = 42,
    ):
        super().__init__()
        self.data_dir = data_dir
        self.batch_size = batch_size
        self.num_workers = num_workers
        self.seed = seed
        self.transform = transform
        self.split_json = os.path.join(self.data_dir, "caltech101/split_zhou_Caltech101.json")

    def prepare_data(self):
        Caltech101_(root=self.data_dir, download=True)

    def setup(self, stage=None):
        full_dataset = Caltech101_(root=self.data_dir, download=False)

        if stage in (None, "fit"):
            self.train = SplitDataset(full_dataset, "train", self.split_json, transform=self.transform)
            self.val = SplitDataset(full_dataset, "val", self.split_json, transform=self.transform)
        if stage in (None, "test"):
            self.test = SplitDataset(full_dataset, "test", self.split_json, transform=self.transform)

    def train_dataloader(self):
        return DataLoader(
            self.train, 
            batch_size=self.batch_size, 
            shuffle=True, 
            num_workers=self.num_workers, 
            pin_memory=True
            )

    def val_dataloader(self):
        return DataLoader(
            self.val, 
            batch_size=self.batch_size, 
            shuffle=False, 
            num_workers=self.num_workers, 
            pin_memory=True
            )

    def test_dataloader(self):
        return DataLoader(
            self.test, 
            batch_size=self.batch_size, 
            shuffle=False, 
            num_workers=self.num_workers, 
            pin_memory=True
            )
    
    def get_indices(self):
        return {"test": 0}
    
class HuggingFaceDictWrapper(Dataset):
    def __init__(self, hf_dataset, transform=None):
        self.dataset = hf_dataset
        self.transform = transform

    def __len__(self):
        return len(self.dataset)

    def __getitem__(self, idx):
        sample = self.dataset[idx]
        image = sample["image"]
        if "label" in sample:
            label = sample["label"]
        if "wnid" in sample:
            wnid = sample["wnid"]
            label = imagenet_r_classes[wnid][1]
        if self.transform:
            image = self.transform(image)
        return image, label

class Sun397DataModule(L.LightningDataModule):
    def __init__(self, 
                 transform,
                 batch_size=64, 
                 num_workers=4, 
                 train_subset_size=None, 
                 seed=42):
        super().__init__()
        self.batch_size = batch_size
        self.num_workers = num_workers
        self.train_subset_size = train_subset_size
        self.seed = seed

        self.transform = transform 

    def prepare_data(self):
        load_dataset('tanganke/sun397', split="train")
        load_dataset('tanganke/sun397', split="test")

    def setup(self, stage=None):
        train_ds = load_dataset('tanganke/sun397', split="train")
        test_ds = load_dataset('tanganke/sun397', split="test")

        train_wrapped = HuggingFaceDictWrapper(train_ds, transform=self.transform)
        test_wrapped = HuggingFaceDictWrapper(test_ds, transform=self.transform)

        if self.train_subset_size is not None:
            torch.manual_seed(self.seed)
            indices = torch.randperm(len(train_wrapped))[:self.train_subset_size]
            train_wrapped = Subset(train_wrapped, indices.tolist())

        self.train = train_wrapped
        self.test = test_wrapped

    def train_dataloader(self):
        return DataLoader(self.train,
                          batch_size=self.batch_size,
                          shuffle=True,
                          num_workers=self.num_workers,
                          pin_memory=True,
                          persistent_workers=False)

    def test_dataloader(self):
        return DataLoader(self.test,
                          batch_size=self.batch_size,
                          shuffle=False,
                          num_workers=self.num_workers,
                          pin_memory=True,
                          persistent_workers=False)
    
    def val_dataloader(self):
        pass
    
    def get_indices(self):
        return {"test": 0}