import pandas as pd
from pydantic import BaseModel
from typing import List, Dict, Union


class OverwriteCell(BaseModel):
    row: int
    col: str
    new_value: Union[str, float, int, bool, None]


class InsertRow(BaseModel):
    index: int
    row: Dict[str, Union[str, float, int, bool, None]]


class TableDeltaSpec(BaseModel):
    """Data model for describing table perturbations."""

    insert_rows: List[InsertRow] = []
    overwrite_cells: List[OverwriteCell] = []

    def pretty_print_diff(self) -> str:
        lines = []
        if self.insert_rows:
            lines.append("📌 Inserted Rows:")
            for insert in self.insert_rows:
                row_items = ", ".join(f"{k}={repr(v)}" for k, v in insert.row.items())
                lines.append(f"  [index {insert.index + 1}] → {{ {row_items} }}")

        if self.overwrite_cells:
            lines.append("✏️  Overwritten Cells:")
            for cell in self.overwrite_cells:
                val_str = "NaN" if cell.new_value is None else repr(cell.new_value)
                lines.append(f"  [row {cell.row + 1}, column '{cell.col}'] → {val_str}")

        if not lines:
            lines.append("✅ No changes detected.")

        return "\n".join(lines)


def apply_transform_spec(
    df_clean: pd.DataFrame, transform_spec: TableDeltaSpec
) -> pd.DataFrame:
    """Applies a TableDeltaSpec to a DataFrame."""
    df_perturbed = df_clean.copy(deep=True)
    df_perturbed = df_perturbed.astype(str)
    for cell in transform_spec.overwrite_cells:
        value = "" if cell.new_value is None else cell.new_value
        df_perturbed.at[cell.row, cell.col] = value
    for insert in sorted(transform_spec.insert_rows, key=lambda x: x.index):
        row_df = pd.DataFrame(
            [{col: "" if val is None else val for col, val in insert.row.items()}]
        )
        # Split and concat around the insert point
        top = df_perturbed.iloc[: insert.index]
        bottom = df_perturbed.iloc[insert.index :]
        df_perturbed = pd.concat([top, row_df, bottom], ignore_index=True)
    return df_perturbed
