import torch

def create_approx_random_standard_orthogonal_matrix(dim):
    """
    生成一个n×n的随机矩阵，元素来自标准正态分布，并按列归一化
    利用随机生成的矩阵作为近似正交矩阵，这个是来源于
    参数:
        dim: 矩阵的维度

    返回:
        归一化后的n×n矩阵
    """
    # 1. 生成n×n的随机矩阵，元素来自标准正态分布N(0,1)
    matrix = torch.randn(dim, dim)
    # 2. 按列归一化(使每列的L2范数为1)
    # 计算每列的L2范数
    col_norms = torch.norm(matrix, p=2, dim=0, keepdim=True)
    # 归一化(防止除以0)
    normalized_matrix = matrix / (col_norms + 1e-7)

    return normalized_matrix

def create_precise_random_standard_orthogonal_matrix(dim):
    '''
    生成一个n×n的随机矩阵，元素来自标准正态分布
    并使用QR分解得到一个精准的正交矩阵


    :param dim:矩阵的维度，可以是一个int，也可以是一个tuple(二维)
    :return:归一化后的n×n矩阵
    '''
    if type(dim) is int:
        matrix = torch.randn(dim, dim)  # 示例：5×3 矩阵
        # 计算 QR 分解
        Q,_ = torch.linalg.qr(matrix, mode='complete')  # 'reduced' 是默认模式
        return Q
    elif type(dim) is tuple:
        assert len(dim)==2, f"error dim len, expect 2 dim but receive {len(dim)}"
        matrix = torch.randn(dim[0], dim[1])  # 示例：5×3 矩阵
        Q,_ = torch.linalg.qr(matrix, mode='reduced')  # 'reduced' 是默认模式
        return Q

    return None

def verify_orthogonal(Q, atol=1e-5):
    '''
    判断给定矩阵是否满足正交的特性，即 Q * Q^T = I

    :param Q:给定的矩阵Q
    :param atol:所允许的误差范围，通常比较小，默认1e-5
    :return:True or False，True的话代表Q满足正交特性
    '''
    QtQ = Q.T @ Q
    #生成的那位矩阵
    identity = torch.eye(Q.shape[0])

    return torch.allclose(QtQ, identity, atol=atol)

