import datetime
import io
import random
import traceback
from collections import defaultdict

import numpy as np
import torch
import torch.nn as nn
from torch.utils.data import IterableDataset
import pickle
import copy


def episode_len(episode):
    # subtract -1 because the dummy first transition
    return next(iter(episode.values())).shape[0] - 1


def save_episode(episode, fn):
    with io.BytesIO() as bs:
        np.savez_compressed(bs, **episode)
        bs.seek(0)
        with fn.open('wb') as f:
            f.write(bs.read())


def load_episode(fn):
    with fn.open('rb') as f:
        episode = np.load(f)
        episode = {k: episode[k] for k in episode.keys()}
        return episode


class ReplayBufferStorage:
    def __init__(self, data_specs, replay_dir, adaptive_truncate, adaptive_truncate_alpha, adaptive_truncate_offset, buffer_truncate_by_progress, adaptive_discount):
        self._data_specs = data_specs
        self._replay_dir = replay_dir
        self.adaptive_truncate = adaptive_truncate
        self.adaptive_truncate_offset = adaptive_truncate_offset
        self.adaptive_truncate_alpha = adaptive_truncate_alpha
        self.buffer_truncate_by_progress = buffer_truncate_by_progress
        self.adaptive_discount = adaptive_discount
        replay_dir.mkdir(exist_ok=True)
        (replay_dir / 'backup').mkdir(exist_ok=True)
        self._current_episode = defaultdict(list)
        self._preload()

    def __len__(self):
        return self._num_transitions

    def add(self, time_step):
        for spec in self._data_specs:
            value = time_step[spec.name]
            if np.isscalar(value):
                value = np.full(spec.shape, value, spec.dtype)
            assert spec.shape == value.shape and spec.dtype == value.dtype
            self._current_episode[spec.name].append(value)
        if time_step.last():
            episode = dict()
            for spec in self._data_specs:
                value = self._current_episode[spec.name]
                episode[spec.name] = np.array(value, spec.dtype)
            self._current_episode = defaultdict(list)

            truncate = None
            if self.adaptive_truncate:
                reward = episode['reward']
                for i in range(1, reward.shape[0]):
                    if np.min(reward[:i]) * self.adaptive_truncate_alpha > np.mean(reward[i:]):
                        truncate = min(i + self.adaptive_truncate_offset, reward.shape[0])
            elif self.buffer_truncate_by_progress:
                truncate = int(self.progress * (episode['reward'].shape[0] - 1))
            if truncate is not None:
                for spec in self._data_specs:
                    episode[spec.name] = episode[spec.name][:truncate + 1]
            self._store_episode(episode)

    def _preload(self):
        self._num_episodes = 0
        self._num_transitions = 0
        for fn in self._replay_dir.glob('*.npz'):
            _, _, eps_len = fn.stem.split('_')
            self._num_episodes += 1
            self._num_transitions += int(eps_len)

    def _store_episode(self, episode):
        eps_idx = self._num_episodes
        eps_len = episode_len(episode)
        self._num_episodes += 1
        self._num_transitions += eps_len
        ts = datetime.datetime.now().strftime('%Y%m%dT%H%M%S')
        eps_fn = f'{ts}_{eps_idx}_{eps_len}.npz'
        save_episode(episode, self._replay_dir / eps_fn)

    def update_parameters(self, parameters):
        for k, v in parameters.items():
            self.__dict__[k] = v
        
        data = {'progress': self.progress}
        if self.adaptive_discount:
            data['_discount'] = self._discount
        fn = self._replay_dir / 'parameters.pkl'
        with open(str(fn), 'wb') as f:
            pickle.dump(data, f)


class ReplayBuffer(IterableDataset):
    def __init__(self, replay_dir, max_size, num_workers, nstep, discount,
                 fetch_every, save_snapshot, progress_guide, progress_truncate, buffer_truncate_by_progress, biased_sampling, mask_reward_by_progress, oversample_timeout, use_timeout, mask_reward_by_progress_offset, sample_by_length, idxs_count):
        self._replay_dir = replay_dir
        self._size = 0
        self._max_size = max_size
        self._num_workers = max(1, num_workers)
        self.sample_by_length = sample_by_length
        if self.sample_by_length:
            self._max_len = 1
            self._episode_fns = [[], []]
        else:
            self._episode_fns = []
        self._episodes = dict()
        self._nstep = nstep
        self._discount = discount
        self._fetch_every = fetch_every
        self._samples_since_last_fetch = fetch_every
        self._save_snapshot = save_snapshot
        if idxs_count is None:
            self.idxs_count = np.ones(200)
        else:
            self.idxs_count = idxs_count
        self.reward_sum = 0
        self.reward_count = 1e-5

        self.progress_guide = progress_guide
        self.progress_truncate = progress_truncate
        self.buffer_truncate_by_progress = buffer_truncate_by_progress
        self.biased_sampling = biased_sampling
        self.mask_reward_by_progress = mask_reward_by_progress
        self.mask_reward_by_progress_offset = mask_reward_by_progress_offset
        self.oversample_timeout = oversample_timeout
        self.use_timeout = use_timeout

    def _sample_episode(self, length=None):
        if self.sample_by_length:
            eps_fn = random.choice(self._episode_fns[length])
        else:
            eps_fn = random.choice(self._episode_fns)
        
        return self._episodes[eps_fn]

    def _store_episode(self, eps_fn):
        try:
            episode = load_episode(eps_fn)
        except:
            return False
        eps_len = episode_len(episode)
        while eps_len + self._size > self._max_size:
            if self.sample_by_length:
                earliest_eps_pos = self._max_len
                for i in range(1, self._max_len):
                    if len(self._episode_fns[i]) > 0 and self._episode_fns[i][0] < self._episode_fns[earliest_eps_pos][0]:
                        earliest_eps_pos = i
                early_eps_fn = self._episode_fns[earliest_eps_pos].pop(0)
            else:
                early_eps_fn = self._episode_fns.pop(0)
            early_eps = self._episodes.pop(early_eps_fn)
            self._size -= episode_len(early_eps)
            early_eps_fn.unlink(missing_ok=True)
        if self.sample_by_length:
            while self._max_len < eps_len:
                self._max_len += 1
                self._episode_fns.append([])
            self._episode_fns[eps_len].append(eps_fn)
            self._episode_fns[eps_len].sort()
        else:
            self._episode_fns.append(eps_fn)
            self._episode_fns.sort()
        self._episodes[eps_fn] = episode
        self._size += eps_len

        if self._save_snapshot:
            eps_fn.replace(eps_fn.parent / 'backup' / eps_fn.name)
        else:
            eps_fn.unlink(missing_ok=True)
        return True

    def _try_fetch(self):
        if self._samples_since_last_fetch < self._fetch_every:
            return

        fn = self._replay_dir / 'parameters.pkl'
        with fn.open('rb') as f:
            parameters = pickle.load(f)
            for k, v in parameters.items():
                self.__dict__[k] = v
            
        self._samples_since_last_fetch = 0
        try:
            worker_id = torch.utils.data.get_worker_info().id
        except:
            worker_id = 0
        eps_fns = sorted(self._replay_dir.glob('*.npz'), reverse=True)
        fetched_size = 0
        for eps_fn in eps_fns:
            eps_idx, eps_len = [int(x) for x in eps_fn.stem.split('_')[1:]]
            if eps_idx % self._num_workers != worker_id:
                continue
            if eps_fn in self._episodes.keys():
                break
            if fetched_size + eps_len > self._max_size:
                break
            fetched_size += eps_len
            if not self._store_episode(eps_fn):
                break

    def _sample(self):
        try:
            self._try_fetch()
        except:
            traceback.print_exc()
        self._samples_since_last_fetch += 1

        if self.use_timeout:
            offset = 0
        else:
            offset = 1 - self._nstep
        if self.sample_by_length:
            length = self._max_len
            while True:
                idx = np.random.randint(0, length + offset) + 1 #warning!!
                sample_weights = [len(self._episode_fns[i]) for i in range(idx, self._max_len + 1)]
                if np.sum(sample_weights) > 0:
                    break
            self.idxs_count[idx] += 1
            episode = self._sample_episode(random.choices(range(idx, self._max_len + 1), sample_weights)[0])
        else:
            episode = self._sample_episode()
            while True:
                if self.progress_guide or self.progress_truncate is not None:
                    assert not (self.progress_guide and self.progress_truncate)
                    if self.progress_truncate is not None:
                        length = int(self.progress_truncate * episode_len(episode))
                    elif self.buffer_truncate_by_progress:
                        length = episode_len(episode)
                    else:
                        length = int(self.progress * episode_len(episode))
                    if self.mask_reward_by_progress is None:
                        if self.biased_sampling:
                            sample_weights = [1/x for x in self.idxs_count]
                        else:
                            sample_weights = [1] * length
                        idx = random.choices(range(length + offset), weights=sample_weights[:length + offset], k=1)[0]
                        self.idxs_count[idx] += 1
                        idx += 1
                    else:
                        idx = np.random.randint(0, min(episode_len(episode), length + self.mask_reward_by_progress_offset) + offset) + 1
                        episode = copy.deepcopy(episode)
                        self.reward_sum += episode['reward'][1:length + 1].sum()
                        self.reward_count += length
                        episode['reward'][length + 1:] = self.mask_reward_by_progress * abs(self.reward_sum / self.reward_count)
                else:
                    idx = np.random.randint(0, episode_len(episode) + offset) + 1
                    length = episode_len(episode)
                if self.oversample_timeout is None or idx < length - self._nstep + 1:
                    break

        # if self.oversample_timeout is not None and np.random.uniform() < self.oversample_timeout:
        #     raise NotImplementedError
        #     idx = length - self._nstep + 1

        obs = episode['observation'][idx - 1]
        action = episode['action'][idx]
        next_one_obs = episode['observation'][idx]
        reward = np.zeros_like(episode['reward'][idx])
        discount = np.ones_like(episode['discount'][idx])
        timeout = np.zeros_like(episode['discount'][idx])
        for i in range(self._nstep):
            step_reward = episode['reward'][idx + i]
            reward += discount * step_reward
            discount *= episode['discount'][idx + i] * self._discount
            if self.use_timeout:
                if idx + i == length:
                    timeout = np.ones_like(episode['discount'][idx])
                    break
                if idx + i == episode['reward'].shape[0] - 1:
                    for j in range(i + 1, self._nstep):
                        reward += discount * episode['reward'][idx + i]
                        discount *= episode['discount'][idx + i] * self._discount
                    timeout = np.ones_like(episode['discount'][idx])
                    break
        if timeout:
            next_obs = np.zeros_like(obs)
        else:
            next_obs = episode['observation'][idx + self._nstep - 1]
        ts = np.ones_like(episode['discount'][idx]) * (idx - 1)
        mean_reward = np.ones_like(episode['discount'][idx]) * self.reward_sum / self.reward_count

        return (obs, action, reward, discount, next_obs, next_one_obs, timeout, ts, mean_reward, self.idxs_count)

    def __iter__(self):
        while True:
            yield self._sample()


class ExpertReplayBuffer(IterableDataset):
    def __init__(self, dataset_path, num_demos, obs_type):
        with open(dataset_path, 'rb') as f:
            data = pickle.load(f)
            if len(data) == 5:
                data = data[:4]
            if obs_type == 'pixels':
                obses, _, actions, _ = data
            elif obs_type == 'features':
                _, obses, actions, _ = data

        self._episodes = []
        for i in range(num_demos):
            episode = dict(observation=obses[i], action=actions[i])
            self._episodes.append(episode)

    def _sample_episode(self):
        episode = random.choice(self._episodes)
        return episode

    def _sample(self):
        episode = self._sample_episode()
        idx = np.random.randint(0, episode_len(episode) - 1) + 1
        obs = episode['observation'][idx]
        action = episode['action'][idx]
        next_obs = episode['observation'][idx + 1]

        return (obs, action, next_obs)

    def __iter__(self):
        while True:
            yield self._sample()


def _worker_init_fn(worker_id):
    seed = np.random.get_state()[1][0] + worker_id
    np.random.seed(seed)
    random.seed(seed)


def make_replay_loader(replay_dir, max_size, batch_size, num_workers,
                       save_snapshot, nstep, discount, progress_guide, progress_truncate, buffer_truncate_by_progress, biased_sampling, mask_reward_by_progress, oversample_timeout, use_timeout, mask_reward_by_progress_offset, sample_by_length, idxs_count=None):
    max_size_per_worker = max_size // max(1, num_workers)

    iterable = ReplayBuffer(replay_dir,
                            max_size_per_worker,
                            num_workers,
                            nstep,
                            discount,
                            fetch_every=1000,
                            save_snapshot=save_snapshot,
                            progress_guide=progress_guide,
                            progress_truncate=progress_truncate,
                            buffer_truncate_by_progress=buffer_truncate_by_progress,
                            biased_sampling=biased_sampling,
                            mask_reward_by_progress=mask_reward_by_progress,
                            oversample_timeout=oversample_timeout,
                            use_timeout=use_timeout,
                            mask_reward_by_progress_offset=mask_reward_by_progress_offset,
                            sample_by_length=sample_by_length,
                            idxs_count=idxs_count)

    loader = torch.utils.data.DataLoader(iterable,
                                         batch_size=batch_size,
                                         num_workers=num_workers,
                                         pin_memory=True,
                                         worker_init_fn=_worker_init_fn)
    return loader


def make_expert_replay_loader(replay_dir, batch_size, num_demos, obs_type):
    iterable = ExpertReplayBuffer(replay_dir, num_demos, obs_type)

    loader = torch.utils.data.DataLoader(iterable,
                                         batch_size=batch_size,
                                         num_workers=2,
                                         pin_memory=True,
                                         worker_init_fn=_worker_init_fn)
    return loader
