#!/bin/bash

# Minimal, cluster-agnostic example for ImageNet-100 pretraining.
# Customize resources, env activation, and W&B settings to your needs.

set -euo pipefail

# Optional: load per-repo secrets
# if [[ -f "$(dirname "$0")/.env" ]]; then
#   # Contains e.g., export WANDB_API_KEY=... and WANDB_ENTITY=..., WANDB_PROJECT=...
#   source "$(dirname "$0")/.env"
# fi

# Optional W&B variables (leave empty to disable entity/project override)
: "${WANDB_ENTITY:=}"
: "${WANDB_PROJECT:=solo-learn}"
: "${USE_WANDB:=true}"

JOB_NAME="imagenet100-pretrain"
RUN_DATE=$(date +%b%d)
SLURM_LOG_DIR="${SCRATCH:-./}/solo-learn-gaussianization/slurm/${RUN_DATE}/${JOB_NAME}"
mkdir -p "$SLURM_LOG_DIR"

sbatch \
  --job-name="$JOB_NAME" \
  --nodes=1 \
  --gpus-per-node=1 \
  --cpus-per-task=16 \
  --time=40:00:00 \
  --mem=64G \
  --error="${SLURM_LOG_DIR}/%j_%N.err" \
  --output="${SLURM_LOG_DIR}/%j_%N.out" \
  --wrap="bash -lc '
    set -euo pipefail
    # Activate your environment
    # source ~/.bashrc
    # conda activate <your_env>

    cd "$(git rev-parse --show-toplevel)"

    # Optional: ensure WANDB_API_KEY is available in env if you use W&B
    # export WANDB_API_KEY=\"<your-wandb-api-key>\"

    python3 main_pretrain.py \
      --config-path scripts/pretrain/imagenet-100/ \
      --config-name radialvicreg.yaml \
      ++wandb.enabled=${USE_WANDB} \
      ++wandb.entity=${WANDB_ENTITY} \
      ++wandb.project=${WANDB_PROJECT} \
      ++max_epochs=400
  '"

echo "Submitted: $JOB_NAME (logs: $SLURM_LOG_DIR)"

