# Copyright 2023 solo-learn development team.

# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all copies
# or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
# INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
# FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

import torch
import torch.nn.functional as F
from solo.utils.misc import gather
import math


def invariance_loss(z1: torch.Tensor, z2: torch.Tensor) -> torch.Tensor:
    """Computes mse loss given batch of projected features z1 from view 1 and
    projected features z2 from view 2.

    Args:
        z1 (torch.Tensor): NxD Tensor containing projected features from view 1.
        z2 (torch.Tensor): NxD Tensor containing projected features from view 2.

    Returns:
        torch.Tensor: invariance loss (mean squared error).
    """

    return F.mse_loss(z1, z2)


def variance_loss(z1: torch.Tensor, z2: torch.Tensor) -> torch.Tensor:
    """Computes variance loss given batch of projected features z1 from view 1 and
    projected features z2 from view 2.

    Args:
        z1 (torch.Tensor): NxD Tensor containing projected features from view 1.
        z2 (torch.Tensor): NxD Tensor containing projected features from view 2.

    Returns:
        torch.Tensor: variance regularization loss.
    """

    eps = 1e-4
    std_z1 = torch.sqrt(z1.var(dim=0) + eps)
    std_z2 = torch.sqrt(z2.var(dim=0) + eps)
    std_loss = torch.mean(F.relu(1 - std_z1)) + torch.mean(F.relu(1 - std_z2))
    return std_loss


def covariance_loss(z1: torch.Tensor, z2: torch.Tensor) -> torch.Tensor:
    """Computes covariance loss given batch of projected features z1 from view 1 and
    projected features z2 from view 2.

    Args:
        z1 (torch.Tensor): NxD Tensor containing projected features from view 1.
        z2 (torch.Tensor): NxD Tensor containing projected features from view 2.

    Returns:
        torch.Tensor: covariance regularization loss.
    """

    N, D = z1.size()

    z1 = z1 - z1.mean(dim=0)
    z2 = z2 - z2.mean(dim=0)
    cov_z1 = (z1.T @ z1) / (N - 1)
    cov_z2 = (z2.T @ z2) / (N - 1)

    diag = torch.eye(D, device=z1.device)
    cov_loss = cov_z1[~diag.bool()].pow_(2).sum() / D + cov_z2[~diag.bool()].pow_(2).sum() / D
    return cov_loss

def empirical_minimal_chi_nll_without_constant(d):
    """
    Computes the minimal achievable Negative Log-Likelihood (NLL) loss
    when ||z||_2 follows a Chi distribution with `d` degrees of freedom.
    
    Args:
        d (int): Dimension of the feature vector (degrees of freedom).
        
    Returns:
        float: Minimal NLL loss value.
    """
    d_minus_1 = torch.tensor(d-1, dtype=torch.float32)
    return -(d_minus_1) * torch.log(torch.sqrt(d_minus_1)) + 0.5 * d_minus_1

def chi2_radial_nll_loss(z1: torch.Tensor, z2: torch.Tensor, eps: float = 1e-6) -> torch.Tensor:
    """Computes the cross-entropy term E[-log p_chi(r)] for two views, including log-normalization constants."""
    _, D = z1.size()
    D_float = torch.tensor(D, dtype=z1.dtype, device=z1.device)

    # Log-normalization constant for the Chi distribution PDF
    # log(2^(1-D/2) / Gamma(D/2)) = (1-D/2)*log(2) - lgamma(D/2)
    # The negative of this is added to the NLL.
    log_norm_constant = (D_float / 2 - 1) * math.log(2) + torch.lgamma(D_float / 2)


    # z1 radii
    r1 = torch.norm(z1, dim=1)
    r1_safe = torch.clamp(r1, min=eps)
    # z2 radii
    r2 = torch.norm(z2, dim=1)
    r2_safe = torch.clamp(r2, min=eps)

    # NLL for each view, including the constant
    nll1 = torch.mean(0.5 * (r1_safe**2) - (D - 1) * torch.log(r1_safe)) + log_norm_constant
    nll2 = torch.mean(0.5 * (r2_safe**2) - (D - 1) * torch.log(r2_safe)) + log_norm_constant

    return nll1 + nll2

def m_spacing_entropy_loss(r: torch.Tensor, eps: float = 1e-6) -> torch.Tensor:
    """Computes the m-spacing entropy estimation loss for a batch of radii.

    Args:
        r (torch.Tensor): 1D Tensor of radii.
        eps (float): Small epsilon for numerical stability.

    Returns:
        torch.Tensor: The entropy loss.
    """
    r = r.unsqueeze(1)
    # r = torch.sigmoid(r)

    N, _ = r.size() # corresponds to batch size
    if N < 2:
        return torch.tensor(0.0, device=r.device)

    m = round(math.sqrt(N))

    if m == 0 or (N - m) <= 0:
        return torch.tensor(0.0, device=r.device)

    r_sorted, _ = torch.sort(r, dim=0)
    spacings = r_sorted[m:] - r_sorted[: N - m]
    spacings = spacings * (N + 1) / m

    marginal_ents = torch.log(spacings + eps).sum(dim=0) / (N - m)
    return marginal_ents.mean()

def chi2_radial_nll_loss_for_lightning_logging(z: torch.Tensor, eps: float = 1e-6) -> torch.Tensor:
    """Cross-entropy term for a single set of features (one view), including log-normalization constants."""
    _, D = z.size()
    D_float = torch.tensor(D, dtype=z.dtype, device=z.device)

    # Log-normalization constant for the Chi distribution PDF
    # log(2^(1-D/2) / Gamma(D/2)) = (1-D/2)*log(2) - lgamma(D/2)
    # The negative of this is added to the NLL.
    log_norm_constant = (D_float / 2 - 1) * math.log(2) + torch.lgamma(D_float / 2)


    r = torch.norm(z, dim=1)
    r_safe = torch.clamp(r, min=eps)
    
    nll = torch.mean(0.5 * (r_safe**2) - (D - 1) * torch.log(r_safe)) + log_norm_constant
    return nll

def uniform_loss(x, t=2):
    x = F.normalize(x, dim=1)  # normalize to unit sphere
    return torch.pdist(x, p=2).pow(2).mul(-t).exp().mean().log()

def batch_sparsity_metric(tensor_data, epsilon=1e-12):
    """
    Calculates the sparsity metric S = (||z||_1)^2 / (||z||_2)^2 for each column of a PyTorch tensor.
    
    Args:
        tensor_data (torch.Tensor): The input tensor of shape (B, D), where B is batch size
                                    and D is the number of columns/features.
        epsilon (float): A small value added to the denominator to prevent division by zero.
                         Default is 1e-12.
                         
    Returns:
        torch.Tensor: A 1D tensor of shape (D,) containing the sparsity metric for each column.
    """
    if not isinstance(tensor_data, torch.Tensor):
        raise TypeError("Input must be a torch.Tensor.")
    if tensor_data.ndim != 2:
        raise ValueError("Input tensor must be 2-dimensional (B x D).")
    
    B, D = tensor_data.shape

    # Calculate L1 norm for each column
    l1_norm_per_column = torch.linalg.norm(tensor_data, ord=1, dim=0)
    
    # Calculate squared L2 norm for each column
    l2_norm_sq_per_column = torch.linalg.norm(tensor_data, ord=2, dim=0)**2
    
    # Add epsilon to the denominator for numerical stability
    # The result will be a 1D tensor (D,)
    sparsity_metric_per_column = (l1_norm_per_column**2) / (l2_norm_sq_per_column + epsilon)
    
    # normalize by batch size
    sparsity_metric_per_column = sparsity_metric_per_column / B

    # get max, mean, min
    sparsity_metric_max = sparsity_metric_per_column.max().item()
    sparsity_metric_mean = sparsity_metric_per_column.mean().item()
    sparsity_metric_min = sparsity_metric_per_column.min().item()

    return sparsity_metric_max, sparsity_metric_mean, sparsity_metric_min


def embedding_sparsity_metric(embeddings: torch.Tensor, epsilon: float = 1e-12):
    """
    Calculates the embedding sparsity metric S = (1/D) * (||z_row||_1 / ||z_row||_2)^2 for each
    row (embedding) of a PyTorch tensor. This measures how sparsely features are
    activated within each embedding.

    Args:
        embeddings (torch.Tensor): The input tensor of shape (B, D), where B is batch size
                                   (number of embeddings) and D is the number of
                                   dimensions/features per embedding.
        epsilon (float): A small value added to the L2 norm before division to prevent
                         division by zero. Default is 1e-12.

    Returns:
        tuple[float, float, float]: (max_sparsity, mean_sparsity, min_sparsity)
                                     calculated over the B embedding sparsity values.
    """
    if not isinstance(embeddings, torch.Tensor):
        raise TypeError("Input must be a torch.Tensor.")
    if embeddings.ndim != 2:
        raise ValueError("Input tensor must be 2-dimensional (B x D).")

    B, D = embeddings.shape

    # Calculate L1 norm for each row (embedding)
    l1_norm_per_row = torch.linalg.norm(embeddings, ord=1, dim=1)

    # Calculate L2 norm for each row (embedding)
    l2_norm_per_row = torch.linalg.norm(embeddings, ord=2, dim=1)

    l2_norm_per_row_stable = l2_norm_per_row + epsilon

    ratio = l1_norm_per_row / l2_norm_per_row_stable

    metric_per_embedding = (1.0 / D) * (ratio**2)

    # Get max, mean, min
    max_sparsity = metric_per_embedding.max().item()
    mean_sparsity = metric_per_embedding.mean().item()
    min_sparsity = metric_per_embedding.min().item()

    # similarly return a tensor
    return max_sparsity, mean_sparsity, min_sparsity


def anisotropy_loss(x: torch.Tensor) -> torch.Tensor:
    """Computes the anisotropy of a batch of features.
    0 means isotropic (average cosine similarity ~ 0).
    1 means collapsed (all vectors point the same way, average cosine similarity ~ 1).

    Args:
        x (torch.Tensor): NxD Tensor of features.

    Returns:
        torch.Tensor: Anisotropy value.
    """
    N, _ = x.shape # N is the number of features - 512
    if N <= 1:
        return torch.tensor(0.0, device=x.device, dtype=x.dtype)

    x_norm = F.normalize(x, dim=1) # this is kind of redundant, since cosine similarity is invariant to scaling
    cosine_sim_matrix = x_norm @ x_norm.T # this is the Grahm matrix - multiply by 1/D even after normalization?

    mask = ~torch.eye(N, dtype=torch.bool, device=x.device)
    
    off_diagonal_cosine_sim = cosine_sim_matrix[mask] # grahm matrix without the diagonal
    
    # Should not happen if N > 1, but as a safeguard if somehow mask results in empty
    if off_diagonal_cosine_sim.numel() == 0:
        return torch.tensor(0.0, device=x.device, dtype=x.dtype)
        
    mean_cosine_sim = off_diagonal_cosine_sim.mean()
    return mean_cosine_sim # this is basically telling you the average angle between two random vectors from your features

def _sample_chi_radii(num_samples: int, dim: int, device, dtype) -> torch.Tensor:
    """Samples radii from Chi(df=dim) by drawing standard normal vectors and taking norms."""
    z = torch.randn((num_samples, dim), device=device, dtype=dtype)
    return torch.norm(z, dim=1)

def w1_distance_to_chi(r: torch.Tensor, dim: int, target_num_samples: int = 0) -> torch.Tensor:
    """Computes 1D Wasserstein-1 distance between empirical radii r and Chi(df=dim).

    Args:
        r: Tensor of shape [N] containing radii.
        dim: Feature dimension D (degrees of freedom for Chi).
        target_num_samples: If > 0, number of Chi samples to draw; otherwise use N.

    Returns:
        Scalar tensor with the W1 distance.
    """
    r = r.reshape(-1)
    N = r.numel()
    if N == 0:
        return torch.tensor(0.0, device=r.device, dtype=r.dtype)

    device, dtype = r.device, r.dtype
    K = min(N, target_num_samples) if target_num_samples and target_num_samples > 0 else N

    # If K < N, subsample r to match target sample count
    if K < N:
        idx = torch.randperm(N, device=device)[:K]
        r_used = r[idx]
    else:
        r_used = r

    # Draw Chi targets and compute sorted L1 distance
    u = _sample_chi_radii(K, dim, device, dtype)
    r_sorted, _ = torch.sort(r_used)
    u_sorted, _ = torch.sort(u)
    return torch.mean(torch.abs(r_sorted - u_sorted))

def w1_radial_loss_single(z: torch.Tensor, target_num_samples: int = 512) -> torch.Tensor:
    """W1 distance between radii of features z (one view) and Chi(df=D)."""
    _, D = z.size()
    r = torch.norm(z, dim=1)
    return w1_distance_to_chi(r, D, target_num_samples)

def w1_radial_loss_two_views(z1: torch.Tensor, z2: torch.Tensor, target_num_samples: int = 512) -> torch.Tensor:
    """Sum of W1 distances for two views against Chi(df=D)."""
    _, D = z1.size()
    r1 = torch.norm(z1, dim=1)
    r2 = torch.norm(z2, dim=1)
    w1_1 = w1_distance_to_chi(r1, D, target_num_samples)
    w1_2 = w1_distance_to_chi(r2, D, target_num_samples)
    return w1_1 + w1_2

def radial_vicreg_loss_func(
    z1: torch.Tensor,
    z2: torch.Tensor,
    sim_loss_weight: float = 25.0,
    var_loss_weight: float = 25.0,
    cov_loss_weight: float = 1.0,
    radial_loss_weight: float = 1.0,
    radial_ent_loss_weight: float = 0.0,
) -> torch.Tensor:
    """Computes VICReg's loss given batch of projected features z1 from view 1 and
    projected features z2 from view 2.

    Args:
        z1 (torch.Tensor): NxD Tensor containing projected features from view 1.
        z2 (torch.Tensor): NxD Tensor containing projected features from view 2.
        sim_loss_weight (float): invariance loss weight.
        var_loss_weight (float): variance loss weight.
        cov_loss_weight (float): covariance loss weight.
        radial_loss_weight (float): radial nll loss weight.
        radial_ent_loss_weight (float): radial entropy loss weight.

    Returns:
        torch.Tensor: VICReg loss.
    """
    sim_loss = invariance_loss(z1, z2)

    # vicreg's official code gathers the tensors here
    # https://github.com/facebookresearch/vicreg/blob/main/main_vicreg.py
    z1, z2 = gather(z1), gather(z2)

    var_loss = variance_loss(z1, z2)
    cov_loss = covariance_loss(z1, z2)
    radial_loss = chi2_radial_nll_loss(z1, z2)  # unscaled CE (sum over two views)
    
    r1 = torch.norm(z1, dim=1)
    r2 = torch.norm(z2, dim=1)
    radial_ent_loss = m_spacing_entropy_loss(r1) + m_spacing_entropy_loss(r2)

    # KL estimate (sum over views): CE - H
    kl_loss = radial_loss - radial_ent_loss


    # During optimization, scale only the radial cross-entropy by 1/D for dimension-invariance.
    # Entropy remains unscaled.
    D = z1.size(1)
    loss = (
        sim_loss_weight * sim_loss
        + var_loss_weight * var_loss
        + cov_loss_weight * cov_loss
        + radial_loss_weight * (radial_loss / D)
        - radial_ent_loss_weight * radial_ent_loss
    )
    return loss, sim_loss, var_loss, cov_loss, radial_loss, radial_ent_loss, kl_loss
