# Copyright 2023 solo-learn development team.

# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all copies
# or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
# INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
# FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

from typing import Any, Dict, List, Sequence

import omegaconf
import torch
import torch.nn as nn
from solo.losses.radialsimclr import radial_simclr_loss_func
from solo.losses.radialvicreg import (
    anisotropy_loss,
    batch_sparsity_metric,
    chi2_radial_nll_loss,
    embedding_sparsity_metric,
    uniform_loss,
)
from solo.losses.vicreg_e2mc import entropy_and_hypercovariance_loss
from solo.methods.base import BaseMethod
from solo.utils.misc import gather, omegaconf_select


class RadialSimCLR(BaseMethod):
    def __init__(self, cfg: omegaconf.DictConfig):
        """Implements SimCLR with radial loss.

        Extra cfg settings:
            method_kwargs:
                proj_output_dim (int): number of dimensions of the projected features.
                proj_hidden_dim (int): number of neurons in the hidden layers of the projector.
                temperature (float): temperature for the softmax in the contrastive loss.
                radial_loss_weight (float): weight of the radial loss.
                lambda_strategy (str): 'standard' or 'self_tune'.
        """

        super().__init__(cfg)

        self.temperature: float = cfg.method_kwargs.temperature
        self.radial_loss_weight: float = cfg.method_kwargs.radial_loss_weight
        self.lambda_strategy: str = cfg.method_kwargs.lambda_strategy

        proj_hidden_dim: int = cfg.method_kwargs.proj_hidden_dim
        proj_output_dim: int = cfg.method_kwargs.proj_output_dim

        # projector
        self.projector = nn.Sequential(
            nn.Linear(self.features_dim, proj_hidden_dim),
            nn.ReLU(),
            nn.Linear(proj_hidden_dim, proj_output_dim),
        )

    @staticmethod
    def add_and_assert_specific_cfg(cfg: omegaconf.DictConfig) -> omegaconf.DictConfig:
        """Adds method specific default values/checks for config."""

        cfg = super(RadialSimCLR, RadialSimCLR).add_and_assert_specific_cfg(cfg)

        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_output_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_hidden_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.temperature")
        cfg.method_kwargs.radial_loss_weight = omegaconf_select(
            cfg, "method_kwargs.radial_loss_weight", 0.0
        )
        cfg.method_kwargs.lambda_strategy = omegaconf_select(
            cfg, "method_kwargs.lambda_strategy", "standard"
        )
        assert cfg.method_kwargs.lambda_strategy in ["standard", "self_tune"]

        return cfg

    @property
    def learnable_params(self) -> List[dict]:
        """Adds projector parameters to the parent's learnable parameters.

        Returns:
            List[dict]: list of learnable parameters.
        """

        extra_learnable_params = [{"name": "projector", "params": self.projector.parameters()}]
        return super().learnable_params + extra_learnable_params

    def forward(self, X: torch.tensor) -> Dict[str, Any]:
        """Performs the forward pass of the backbone and the projector.

        Args:
            X (torch.Tensor): a batch of images in the tensor format.

        Returns:
            Dict[str, Any]:
                a dict containing the outputs of the parent
                and the projected features.
        """

        out = super().forward(X)
        z = self.projector(out["feats"])
        out.update({"z": z})
        return out

    def multicrop_forward(self, X: torch.tensor) -> Dict[str, Any]:
        """Performs the forward pass for the multicrop views.

        Args:
            X (torch.Tensor): batch of images in tensor format.

        Returns:
            Dict[]: a dict containing the outputs of the parent
                and the projected features.
        """

        out = super().multicrop_forward(X)
        z = self.projector(out["feats"])
        out.update({"z": z})
        return out

    def training_step(self, batch: Sequence[Any], batch_idx: int) -> torch.Tensor:
        """Training step for SimCLR reusing BaseMethod training step.

        Args:
            batch (Sequence[Any]): a batch of data in the format of [img_indexes, [X], Y], where
                [X] is a list of size num_crops containing batches of images.
            batch_idx (int): index of the batch.

        Returns:
            torch.Tensor: total loss composed of SimCLR loss and classification loss.
        """

        indexes = batch[0]
        out = super().training_step(batch, batch_idx)
        class_loss = out["loss"]

        # --- ENCODER-LEVEL LOGGING ---
        feats1_b, feats2_b = out["feats"]
        gathered_feats1_b = gather(feats1_b)
        gathered_feats2_b = gather(feats2_b)

        with torch.no_grad():
            # radial loss
            radial_loss_encoder = chi2_radial_nll_loss(gathered_feats1_b, gathered_feats2_b)
            self.log(
                "encoder/train_backbone_radial_loss",
                radial_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )

            # uniformity
            self.log(
                "encoder_uniformity/feats1",
                uniform_loss(gathered_feats1_b),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "encoder_uniformity/feats2",
                uniform_loss(gathered_feats2_b),
                on_epoch=True,
                sync_dist=True,
            )

            # anisotropy
            self.log(
                "encoder_anisotropy/feats1",
                anisotropy_loss(gathered_feats1_b),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "encoder_anisotropy/feats2",
                anisotropy_loss(gathered_feats2_b),
                on_epoch=True,
                sync_dist=True,
            )

            # entropy and hypercovariance
            ent_loss_encoder, hypercov_loss_encoder = entropy_and_hypercovariance_loss(
                gathered_feats1_b, gathered_feats2_b, epsilon=1e-7
            )
            self.log("encoder/ent_loss", -ent_loss_encoder, on_epoch=True, sync_dist=True)
            self.log("encoder/hypercov_loss", hypercov_loss_encoder, on_epoch=True, sync_dist=True)

            # sparsity metrics
            for i, feats in enumerate([gathered_feats1_b, gathered_feats2_b]):
                b_max, b_mean, b_min = batch_sparsity_metric(feats)
                e_max, e_mean, e_min = embedding_sparsity_metric(feats)
                self.log(f"batch_sparsity/encoder_feats{i+1}_max", b_max, on_epoch=True)
                self.log(f"batch_sparsity/encoder_feats{i+1}_mean", b_mean, on_epoch=True)
                self.log(f"batch_sparsity/encoder_feats{i+1}_min", b_min, on_epoch=True)
                self.log(f"embedding_sparsity/encoder_feats{i+1}_max", e_max, on_epoch=True)
                self.log(f"embedding_sparsity/encoder_feats{i+1}_mean", e_mean, on_epoch=True)
                self.log(f"embedding_sparsity/encoder_feats{i+1}_min", e_min, on_epoch=True)

        # --- PROJECTOR-LEVEL ---
        z = torch.cat(out["z"])

        # --- LOSS CALCULATION ---
        n_augs = self.num_large_crops + self.num_small_crops
        indexes = indexes.repeat(n_augs)

        nce_loss, radial_loss = radial_simclr_loss_func(
            z, indexes=indexes, temperature=self.temperature
        )

        # --- TOTAL LOSS ---
        if self.lambda_strategy == "self_tune":
            sim_loss_detached = nce_loss.detach()
            w_rad = (self.radial_loss_weight * sim_loss_detached) / (radial_loss.detach() + 1e-6)
            total_loss = nce_loss + w_rad * radial_loss + class_loss
        else:  # standard
            total_loss = nce_loss + self.radial_loss_weight * radial_loss + class_loss

        self.log("train_nce_loss", nce_loss, on_epoch=True, sync_dist=True)
        self.log("train_radial_loss", radial_loss, on_epoch=True, sync_dist=True)

        # --- PROJECTOR-LEVEL LOGGING ---
        with torch.no_grad():
            N, D = z.shape
            z1 = z[: N // 2]
            z2 = z[N // 2 :]
            gathered_z1 = gather(z1)
            gathered_z2 = gather(z2)

            self.log(
                "projector_norm/train_z1_norm_mean",
                torch.norm(z1, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "projector_norm/train_z2_norm_mean",
                torch.norm(z2, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )

            # uniformity
            self.log(
                "projector_uniformity/z1",
                uniform_loss(gathered_z1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "projector_uniformity/z2",
                uniform_loss(gathered_z2),
                on_epoch=True,
                sync_dist=True,
            )

            # anisotropy
            self.log(
                "projector_anisotropy/z1",
                anisotropy_loss(gathered_z1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "projector_anisotropy/z2",
                anisotropy_loss(gathered_z2),
                on_epoch=True,
                sync_dist=True,
            )

            # entropy and hypercovariance
            ent_loss_proj, hypercov_loss_proj = entropy_and_hypercovariance_loss(
                gathered_z1, gathered_z2, epsilon=1e-7
            )
            self.log("projector/ent_loss", -ent_loss_proj, on_epoch=True, sync_dist=True)
            self.log("projector/hypercov_loss", hypercov_loss_proj, on_epoch=True, sync_dist=True)

            # sparsity metrics
            for i, p_feats in enumerate([gathered_z1, gathered_z2]):
                b_max, b_mean, b_min = batch_sparsity_metric(p_feats)
                e_max, e_mean, e_min = embedding_sparsity_metric(p_feats)
                self.log(f"batch_sparsity/proj_z{i+1}_max", b_max, on_epoch=True)
                self.log(f"batch_sparsity/proj_z{i+1}_mean", b_mean, on_epoch=True)
                self.log(f"batch_sparsity/proj_z{i+1}_min", b_min, on_epoch=True)
                self.log(f"embedding_sparsity/proj_z{i+1}_max", e_max, on_epoch=True)
                self.log(f"embedding_sparsity/proj_z{i+1}_mean", e_mean, on_epoch=True)
                self.log(f"embedding_sparsity/proj_z{i+1}_min", e_min, on_epoch=True)

        return total_loss
