# Copyright 2023 solo-learn development team.

# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all copies
# or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
# INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
# FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

from typing import Any, Dict, List, Sequence, Tuple, OrderedDict

import omegaconf
import torch
import torch.distributed as dist
import torch.nn as nn
import torch.nn.functional as F
from torch import Tensor
from solo.losses.swav import swav_loss_func
from solo.losses.radialvicreg import (
    anisotropy_loss,
    batch_sparsity_metric,
    chi2_radial_nll_loss,
    chi2_radial_nll_loss_for_lightning_logging,
    covariance_loss,
    embedding_sparsity_metric,
    invariance_loss,
    uniform_loss,
    variance_loss,
)
from solo.methods.base import BaseMethod
from solo.utils.misc import gather, omegaconf_select
from solo.utils.sinkhorn_knopp import SinkhornKnopp
from solo.utils.metrics import compute_balanced_accuracy


class RadialSwAV(BaseMethod):
    def __init__(self, cfg: omegaconf.DictConfig):
        """Implements SwAV (https://arxiv.org/abs/2006.09882) with radial loss.

        Extra cfg settings:
            method_kwargs:
                proj_output_dim (int): number of dimensions of the projected features.
                proj_hidden_dim (int): number of neurons of the hidden layers of the projector.
                num_prototypes (int): number of prototypes.
                sk_iters (int): number of sinkhorn-knopp iterations.
                sk_epsilon (float): epsilon for sinkhorn-knopp algorithm.
                temperature (float): temperature for the softmax normalization.
                queue_size (int): number of samples to store in the queue.
                epoch_queue_starts (int): epochs the queue starts.
                freeze_prototypes_epochs (int): number of epochs to freeze the prototypes.
                radial_loss_weight (float): weight of the radial loss.
        """

        super().__init__(cfg)

        self.proj_hidden_dim: int = cfg.method_kwargs.proj_hidden_dim
        self.proj_output_dim: int = cfg.method_kwargs.proj_output_dim
        self.sk_iters: int = cfg.method_kwargs.sk_iters
        self.sk_epsilon: float = cfg.method_kwargs.sk_epsilon
        self.temperature: float = cfg.method_kwargs.temperature
        self.queue_size: int = cfg.method_kwargs.queue_size
        self.epoch_queue_starts: int = cfg.method_kwargs.epoch_queue_starts
        self.freeze_prototypes_epochs: int = cfg.method_kwargs.freeze_prototypes_epochs
        self.num_prototypes: int = cfg.method_kwargs.num_prototypes
        self.radial_loss_weight: float = cfg.method_kwargs.radial_loss_weight

        # projector
        self.projector = nn.Sequential(
            nn.Linear(self.features_dim, self.proj_hidden_dim),
            nn.BatchNorm1d(self.proj_hidden_dim),
            nn.ReLU(),
            nn.Linear(self.proj_hidden_dim, self.proj_output_dim),
        )

        # prototypes
        self.prototypes = nn.utils.weight_norm(
            nn.Linear(self.proj_output_dim, self.num_prototypes, bias=False)
        )
        self.prototypes.weight_g.data.fill_(1)  # type: ignore
        self.prototypes.weight_g.requires_grad = False

    @staticmethod
    def add_and_assert_specific_cfg(cfg: omegaconf.DictConfig) -> omegaconf.DictConfig:
        """Adds method specific default values/checks for config.

        Args:
            cfg (omegaconf.DictConfig): DictConfig object.

        Returns:
            omegaconf.DictConfig: same as the argument, used to avoid errors.
        """

        cfg = super(RadialSwAV, RadialSwAV).add_and_assert_specific_cfg(cfg)

        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_output_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_hidden_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.temperature")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.queue_size")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.epoch_queue_starts")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.freeze_prototypes_epochs")

        cfg.method_kwargs.num_prototypes = omegaconf_select(
            cfg,
            "method_kwargs.num_prototypes",
            3000,
        )
        cfg.method_kwargs.sk_epsilon = omegaconf_select(cfg, "method_kwargs.sk_epsilon", 0.05)
        cfg.method_kwargs.sk_iters = omegaconf_select(cfg, "method_kwargs.sk_iters", 3)
        cfg.method_kwargs.freeze_prototypes_epochs = omegaconf_select(
            cfg,
            "method_kwargs.freeze_prototypes_epochs",
            1,
        )
        cfg.method_kwargs.queue_size = omegaconf_select(cfg, "method_kwargs.queue_size", 0)
        cfg.method_kwargs.epoch_queue_starts = omegaconf_select(
            cfg,
            "method_kwargs.epoch_queue_starts",
            0,
        )
        cfg.method_kwargs.radial_loss_weight = omegaconf_select(
            cfg, "method_kwargs.radial_loss_weight", 0.0
        )

        return cfg

    @property
    def learnable_params(self) -> List[dict]:
        """Adds projector and prototypes parameters to the parent's learnable parameters.

        Returns:
            List[dict]: list of learnable parameters.
        """

        extra_learnable_params = [
            {"name": "projector", "params": self.projector.parameters()},
            {"name": "prototypes", "params": self.prototypes.parameters()},
        ]
        return super().learnable_params + extra_learnable_params

    def on_train_start(self):
        """Gets the world size and sets it in the sinkhorn and the queue."""
        # sinkhorn-knopp needs the world size
        world_size = self.trainer.world_size if self.trainer else 1
        self.sk = SinkhornKnopp(self.sk_iters, self.sk_epsilon, world_size)
        # queue also needs the world size
        if self.queue_size > 0:
            self.register_buffer(
                "queue",
                torch.zeros(
                    self.num_large_crops,
                    self.queue_size // world_size,
                    self.proj_output_dim,
                    device=self.device,
                ),
            )
        else:
            self.queue = None

    def forward(self, X: torch.Tensor) -> Dict[str, Any]:
        """Performs the forward pass of the backbone, the projector and the prototypes.

        Args:
            X (torch.Tensor): a batch of images in the tensor format.

        Returns:
            Dict[str, Any]:
                a dict containing the outputs of the parent,
                the projected features and the logits.
        """

        out = super().forward(X)
        z_unnormalized = self.projector(out["feats"])
        z = F.normalize(z_unnormalized)
        p = self.prototypes(z)
        out.update({"z": z, "p": p, "z_unnormalized": z_unnormalized})
        return out

    def multicrop_forward(self, X: torch.Tensor) -> Dict[str, Any]:
        """Performs the forward pass of the backbone, the projector and the prototypes.

        Args:
            X (torch.Tensor): a batch of images in the tensor format.

        Returns:
            Dict[str, Any]:
                a dict containing the outputs of the parent,
                the projected features and the logits.
        """

        out = super().multicrop_forward(X)
        z_unnormalized = self.projector(out["feats"])
        z = F.normalize(z_unnormalized)
        p = self.prototypes(z)
        out.update({"z": z, "p": p, "z_unnormalized": z_unnormalized})
        return out

    @torch.no_grad()
    def get_assignments(self, preds: List[torch.Tensor]) -> List[torch.Tensor]:
        """Computes cluster assignments from logits, optionally using a queue.

        Args:
            preds (List[torch.Tensor]): a batch of logits.

        Returns:
            List[torch.Tensor]: assignments for each sample in the batch.
        """
        bs = preds[0].size(0)
        assignments = []
        for i, p in enumerate(preds):
            # optionally use the queue
            if (
                self.queue is not None
                and self.queue_size > 0
                and self.current_epoch >= self.epoch_queue_starts
            ):
                p_queue = self.prototypes(self.queue[i])  # type: ignore
                p = torch.cat((p, p_queue))
            # compute assignments with sinkhorn-knopp
            assignments.append(self.sk(p)[:bs])
        return assignments

    def validation_step(self, batch: Sequence[Any], batch_idx: int):
        """Validation step for RadialSwAV."""

        X, targets = batch
        out = self.base_validation_step(X, targets)
        batch_size = targets.size(0)

        if not self.trainer.sanity_checking:
            # get online features
            feats = out["feats"]
            z = self.projector(feats)
            z = F.normalize(z)

            # --- log metrics for encoder ---
            gathered_feats = gather(feats)
            self.log(
                "encoder/val_radial_loss",
                chi2_radial_nll_loss_for_lightning_logging(gathered_feats),
                on_epoch=True,
                sync_dist=True,
            )
            # --- log metrics for projector ---
            gathered_z = gather(z)
            self.log(
                "projector/val_radial_loss",
                chi2_radial_nll_loss_for_lightning_logging(gathered_z),
                on_epoch=True,
                sync_dist=True,
            )

        if self.dataset_name == "CelebA":
            out_for_logging = {f"val_{k}": v for k, v in out.items() if "metrics" in k}

            for outcome_type in self.balanced_acc_quantities_order:
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"][
                    outcome_type
                ] += out[f"encoder_{outcome_type}"]
                if "proj_loss" in out:
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        outcome_type
                    ] += out[f"proj_{outcome_type}"]

            encoder_balanced_acc = compute_balanced_accuracy(
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"]["tps"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"]["fns"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"]["tns"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"]["fps"],
            )
            encoder_balanced_acc_dict = dict(
                zip(
                    [f"val_separate_metrics/encoder_{k}" for k in self.dataset_attr_names],
                    encoder_balanced_acc,
                )
            )

            metrics = {
                "batch_size": batch_size,
                "val_loss": out["loss"],
                **out_for_logging,
                **encoder_balanced_acc_dict,
            }

            if "proj_loss" in out:
                proj_balanced_acc = compute_balanced_accuracy(
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        "tps"
                    ],
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        "fns"
                    ],
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        "tns"
                    ],
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        "fps"
                    ],
                )
                proj_balanced_acc_dict = dict(
                    zip(
                        [f"val_separate_metrics/proj_{k}" for k in self.dataset_attr_names],
                        proj_balanced_acc,
                    )
                )
                metrics.update(
                    {"val_proj_loss": out.get("proj_loss"), **proj_balanced_acc_dict}
                )
        else:
            metrics = {
                "batch_size": batch_size,
                "val_loss": out.get("loss"),
                "val_acc1": out.get("acc1"),
                "val_acc5": out.get("acc5"),
            }
            if "proj_loss" in out:
                metrics.update(
                    {
                        "val_proj_loss": out.get("proj_loss"),
                        "val_proj_acc1": out.get("proj_acc1"),
                        "val_proj_acc5": out.get("proj_acc5"),
                    }
                )

        self.validation_step_outputs.append(metrics)
        return metrics

    def training_step(self, batch: Sequence[Any], batch_idx: int) -> torch.Tensor:
        """Training step for SwAV reusing BaseMethod training step.

        Args:
            batch (Sequence[Any]): a batch of data in the format of [img_indexes, [X], Y], where
                [X] is a list of size num_crops containing batches of images.
            batch_idx (int): index of the batch.

        Returns:
            torch.Tensor: total loss composed of SwAV loss and classification loss.
        """

        out = super().training_step(batch, batch_idx)
        class_loss = out["loss"]
        p = out["p"]
        z = out["z"]
        z_unnormalized = out["z_unnormalized"]

        # ------ swav loss ------
        assignments = self.get_assignments(p[: self.num_large_crops])
        swav_loss = swav_loss_func(p, assignments, self.temperature)

        # ------- update queue -------
        if self.queue is not None and self.queue_size > 0:
            stacked_z = torch.stack(z[: self.num_large_crops])
            bs = stacked_z.size(1)
            self.queue[:, bs:] = self.queue[:, :-bs].clone()
            self.queue[:, :bs] = stacked_z.detach()

        # ------ radial loss ------ # Note: applied to unnormalized features, consider testing on normalized features as well
        # radial loss on projector outputs for the two large crops
        radial_loss = chi2_radial_nll_loss(z_unnormalized[0], z_unnormalized[1])

        with torch.no_grad():
            # -- log metrics for encoder --
            feats1, feats2 = out["feats"]
            gathered_feats1 = gather(feats1)
            gathered_feats2 = gather(feats2)

            # from radialvicreg
            sim_loss_encoder = invariance_loss(feats1, feats2)
            self.log(
                "encoder/train_backbone_sim_loss",
                sim_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )
            var_loss_encoder = variance_loss(gathered_feats1, gathered_feats2)
            self.log(
                "encoder/train_backbone_var_loss",
                var_loss_encoder / self.features_dim,
                on_epoch=True,
                sync_dist=True,
            )
            cov_loss_encoder = covariance_loss(gathered_feats1, gathered_feats2)
            self.log(
                "encoder/train_backbone_cov_loss", cov_loss_encoder, on_epoch=True, sync_dist=True
            )
            radial_loss_encoder = chi2_radial_nll_loss(gathered_feats1, gathered_feats2)
            self.log(
                "encoder/train_backbone_radial_loss",
                radial_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )

            # norms
            self.log(
                "encoder_norm/train_encoder_feats1_norm_mean",
                torch.norm(gathered_feats1, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "encoder_norm/train_encoder_feats2_norm_mean",
                torch.norm(gathered_feats2, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            # uniformity
            self.log(
                "unif_loss/train_encoder_feats1_unif_loss",
                uniform_loss(gathered_feats1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "unif_loss/train_encoder_feats2_unif_loss",
                uniform_loss(gathered_feats2),
                on_epoch=True,
                sync_dist=True,
            )
            # anisotropy
            self.log(
                "anisotropy_loss/train_encoder_feats1_anisotropy",
                anisotropy_loss(gathered_feats1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "anisotropy_loss/train_encoder_feats2_anisotropy",
                anisotropy_loss(gathered_feats2),
                on_epoch=True,
                sync_dist=True,
            )
            # batch sparsity
            feat1_batch_sparse_max, feat1_batch_sparse_mean, feat1_batch_sparse_min = (
                batch_sparsity_metric(gathered_feats1)
            )
            feat2_batch_sparse_max, feat2_batch_sparse_mean, feat2_batch_sparse_min = (
                batch_sparsity_metric(gathered_feats2)
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_max",
                feat1_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_mean",
                feat1_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_min",
                feat1_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_max",
                feat2_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_mean",
                feat2_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_min",
                feat2_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            # embedding sparsity
            (
                feat1_embed_sparse_max,
                feat1_embed_sparse_mean,
                feat1_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_feats1)
            (
                feat2_embed_sparse_max,
                feat2_embed_sparse_mean,
                feat2_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_feats2)
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_max",
                feat1_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_mean",
                feat1_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_min",
                feat1_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_max",
                feat2_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_mean",
                feat2_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_min",
                feat2_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )

            # -- log metrics for projector --
            gathered_z1 = gather(z[0])
            gathered_z2 = gather(z[1])

            # norms
            self.log(
                "projector_norm/train_projector_z1_norm_mean",
                torch.norm(gathered_z1, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "projector_norm/train_projector_z2_norm_mean",
                torch.norm(gathered_z2, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            # uniformity
            self.log(
                "unif_loss/train_projector_z1_unif_loss",
                uniform_loss(gathered_z1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "unif_loss/train_projector_z2_unif_loss",
                uniform_loss(gathered_z2),
                on_epoch=True,
                sync_dist=True,
            )
            # anisotropy
            self.log(
                "anisotropy_loss/train_projector_z1_anisotropy",
                anisotropy_loss(gathered_z1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "anisotropy_loss/train_projector_z2_anisotropy",
                anisotropy_loss(gathered_z2),
                on_epoch=True,
                sync_dist=True,
            )
            # batch sparsity
            z1_batch_sparse_max, z1_batch_sparse_mean, z1_batch_sparse_min = (
                batch_sparsity_metric(gathered_z1)
            )
            z2_batch_sparse_max, z2_batch_sparse_mean, z2_batch_sparse_min = (
                batch_sparsity_metric(gathered_z2)
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_max",
                z1_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_mean",
                z1_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_min",
                z1_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_max",
                z2_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_mean",
                z2_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_min",
                z2_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            # embedding sparsity
            (
                z1_embed_sparse_max,
                z1_embed_sparse_mean,
                z1_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_z1)
            (
                z2_embed_sparse_max,
                z2_embed_sparse_mean,
                z2_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_z2)
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_max",
                z1_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_mean",
                z1_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_min",
                z1_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_max",
                z2_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_mean",
                z2_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_min",
                z2_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )

        metrics = {
            "train_swav_loss": swav_loss,
            "train_radial_loss": radial_loss,
        }
        self.log_dict(metrics, on_epoch=True, sync_dist=True)

        return swav_loss + self.radial_loss_weight * radial_loss + class_loss

    def on_after_backward(self):
        """Zeroes the gradients of the prototypes."""
        if self.current_epoch < self.freeze_prototypes_epochs:
            for p in self.prototypes.parameters():
                p.grad = None

    def load_state_dict(self, state_dict: "OrderedDict[str, Tensor]", strict: bool = True): # to help with continued pretraining
        """
        Overrides the default load_state_dict to handle the queue buffer, which is
        dynamically registered and sized in `on_train_start`. This prevents errors
        when resuming from a checkpoint, especially if the number of GPUs changes.
        """
        if "queue" in state_dict:
            # The queue state is not critical and will be re-filled,
            # so we can safely remove it.
            state_dict.pop("queue")

        # We use strict=False because the shape of the queue in the checkpoint
        # will not match the (possibly uninitialized) queue in the model.
        super().load_state_dict(state_dict, strict=False)
