# Copyright 2023 solo-learn development team.

# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all copies
# or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
# INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
# FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

from typing import Any, Dict, List, Sequence, Tuple

import omegaconf
import torch
import torch.nn as nn
from solo.losses.mocov3 import mocov3_loss_func
from solo.losses.radialvicreg import (
    anisotropy_loss,
    batch_sparsity_metric,
    chi2_radial_nll_loss,
    chi2_radial_nll_loss_for_lightning_logging,
    covariance_loss,
    embedding_sparsity_metric,
    invariance_loss,
    uniform_loss,
    variance_loss,
)
from solo.losses.vicreg_e2mc import entropy_and_hypercovariance_loss
from solo.methods.base import BaseMomentumMethod
from solo.utils.momentum import initialize_momentum_params
from solo.utils.misc import gather, omegaconf_select
from solo.utils.metrics import compute_balanced_accuracy


class RadialMoCoV3(BaseMomentumMethod):
    def __init__(self, cfg: omegaconf.DictConfig):
        """Implements MoCo V3 (https://arxiv.org/abs/2104.02057) with radial loss.

        Extra cfg settings:
            method_kwargs:
                proj_output_dim (int): number of dimensions of projected features.
                proj_hidden_dim (int): number of neurons of the hidden layers of the projector.
                pred_hidden_dim (int): number of neurons of the hidden layers of the predictor.
                temperature (float): temperature for the softmax in the contrastive loss.
                radial_loss_weight (float): weight of the radial loss.
        """

        super().__init__(cfg)

        self.temperature: float = cfg.method_kwargs.temperature
        self.radial_loss_weight: float = cfg.method_kwargs.radial_loss_weight

        proj_hidden_dim: int = cfg.method_kwargs.proj_hidden_dim
        proj_output_dim: int = cfg.method_kwargs.proj_output_dim
        pred_hidden_dim: int = cfg.method_kwargs.pred_hidden_dim

        if "resnet" in self.backbone_name:
            # projector
            self.projector = self._build_mlp(
                2,
                self.features_dim,
                proj_hidden_dim,
                proj_output_dim,
            )
            # momentum projector
            self.momentum_projector = self._build_mlp(
                2,
                self.features_dim,
                proj_hidden_dim,
                proj_output_dim,
            )

            # predictor
            self.predictor = self._build_mlp(
                2,
                proj_output_dim,
                pred_hidden_dim,
                proj_output_dim,
                last_bn=False,
            )
        else:
            # specifically for ViT but allow all the other backbones
            # projector
            self.projector = self._build_mlp(
                3,
                self.features_dim,
                proj_hidden_dim,
                proj_output_dim,
            )
            # momentum projector
            self.momentum_projector = self._build_mlp(
                3,
                self.features_dim,
                proj_hidden_dim,
                proj_output_dim,
            )

            # predictor
            self.predictor = self._build_mlp(
                2,
                proj_output_dim,
                pred_hidden_dim,
                proj_output_dim,
            )

        initialize_momentum_params(self.projector, self.momentum_projector)

    def _build_mlp(self, num_layers, input_dim, mlp_dim, output_dim, last_bn=True):
        mlp = []
        for l in range(num_layers):
            dim1 = input_dim if l == 0 else mlp_dim
            dim2 = output_dim if l == num_layers - 1 else mlp_dim

            mlp.append(nn.Linear(dim1, dim2, bias=False))

            if l < num_layers - 1:
                mlp.append(nn.BatchNorm1d(dim2))
                mlp.append(nn.ReLU(inplace=True))
            elif last_bn:
                # follow SimCLR's design
                mlp.append(nn.BatchNorm1d(dim2, affine=False))

        return nn.Sequential(*mlp)

    @staticmethod
    def add_and_assert_specific_cfg(cfg: omegaconf.DictConfig) -> omegaconf.DictConfig:
        """Adds method specific default values/checks for config.

        Args:
            cfg (omegaconf.DictConfig): DictConfig object.

        Returns:
            omegaconf.DictConfig: same as the argument, used to avoid errors.
        """

        cfg = super(RadialMoCoV3, RadialMoCoV3).add_and_assert_specific_cfg(cfg)

        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_output_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_hidden_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.pred_hidden_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.temperature")

        cfg.method_kwargs.radial_loss_weight = omegaconf_select(
            cfg, "method_kwargs.radial_loss_weight", 0.0
        )

        return cfg

    @property
    def learnable_params(self) -> List[dict]:
        """Adds projector and predictor parameters to the parent's learnable parameters.

        Returns:
            List[dict]: list of learnable parameters.
        """

        extra_learnable_params = [
            {"name": "projector", "params": self.projector.parameters()},
            {"name": "predictor", "params": self.predictor.parameters()},
        ]
        return super().learnable_params + extra_learnable_params

    @property
    def momentum_pairs(self) -> List[Tuple[Any, Any]]:
        """Adds (projector, momentum_projector) to the parent's momentum pairs.

        Returns:
            List[Tuple[Any, Any]]: list of momentum pairs.
        """

        extra_momentum_pairs = [(self.projector, self.momentum_projector)]
        return super().momentum_pairs + extra_momentum_pairs

    def forward(self, X: torch.Tensor) -> Dict[str, Any]:
        """Performs forward pass of the online backbone, projector and predictor.

        Args:
            X (torch.Tensor): batch of images in tensor format.

        Returns:
            Dict[str, Any]: a dict containing the outputs of the parent and the projected features.
        """

        out = super().forward(X)
        z = self.projector(out["feats"])
        q = self.predictor(z)
        out.update({"q": q, "z": z})
        return out

    @torch.no_grad()
    def momentum_forward(self, X: torch.Tensor) -> Dict:
        """Performs the forward pass of the momentum backbone and projector.

        Args:
            X (torch.Tensor): batch of images in tensor format.

        Returns:
            Dict[str, Any]: a dict containing the outputs of
                the parent and the momentum projected features.
        """

        out = super().momentum_forward(X)
        k = self.momentum_projector(out["feats"])
        out.update({"k": k})
        return out

    def validation_step(self, batch: Sequence[Any], batch_idx: int):
        """Validation step for RadialMoCoV3."""

        X, targets = batch

        out = self.base_validation_step(X, targets)
        momentum_out = self._shared_step_momentum(X, targets)
        if self.momentum_classifier:
            out.update(
                {
                    "momentum_val_loss": momentum_out["loss"],
                    "momentum_val_acc1": momentum_out["acc1"],
                    "momentum_val_acc5": momentum_out["acc5"],
                }
            )

        batch_size = targets.size(0)

        if not self.trainer.sanity_checking:
            # get online features
            feats = out["feats"]
            z = self.projector(feats)

            # --- log metrics for encoder ---
            gathered_feats = gather(feats)
            self.log(
                "encoder/val_radial_loss",
                chi2_radial_nll_loss_for_lightning_logging(gathered_feats),
                on_epoch=True,
                sync_dist=True,
            )
            # --- log metrics for projector ---
            gathered_z = gather(z)
            self.log(
                "projector/val_radial_loss",
                chi2_radial_nll_loss_for_lightning_logging(gathered_z),
                on_epoch=True,
                sync_dist=True,
            )

        if self.dataset_name == "CelebA": #for CelebA - future TODO: make this more general
            out_for_logging = {f"val_{k}": v for k, v in out.items() if "metrics" in k}

            for outcome_type in self.balanced_acc_quantities_order:
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"][
                    outcome_type
                ] += out[f"encoder_{outcome_type}"]
                if "proj_loss" in out:
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        outcome_type
                    ] += out[f"proj_{outcome_type}"]

            encoder_balanced_acc = compute_balanced_accuracy(
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"]["tps"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"]["fns"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"]["tns"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"]["fps"],
            )
            encoder_balanced_acc_dict = dict(
                zip(
                    [f"val_separate_metrics/encoder_{k}" for k in self.dataset_attr_names],
                    encoder_balanced_acc,
                )
            )

            metrics = {
                "batch_size": batch_size,
                "val_loss": out["loss"],
                **out_for_logging,
                **encoder_balanced_acc_dict,
            }

            if "proj_loss" in out:
                proj_balanced_acc = compute_balanced_accuracy(
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        "tps"
                    ],
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        "fns"
                    ],
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        "tns"
                    ],
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        "fps"
                    ],
                )
                proj_balanced_acc_dict = dict(
                    zip(
                        [f"val_separate_metrics/proj_{k}" for k in self.dataset_attr_names],
                        proj_balanced_acc,
                    )
                )
                metrics.update(
                    {"val_proj_loss": out.get("proj_loss"), **proj_balanced_acc_dict}
                )
        else:
            metrics = {
                "batch_size": batch_size,
                "val_loss": out.get("loss"),
                "val_acc1": out.get("acc1"),
                "val_acc5": out.get("acc5"),
            }
            if "proj_loss" in out:
                metrics.update(
                    {
                        "val_proj_loss": out.get("proj_loss"),
                        "val_proj_acc1": out.get("proj_acc1"),
                        "val_proj_acc5": out.get("proj_acc5"),
                    }
                )

        if self.momentum_classifier:
            metrics.update(
                {
                    "momentum_val_loss": out["momentum_val_loss"],
                    "momentum_val_acc1": out["momentum_val_acc1"],
                    "momentum_val_acc5": out["momentum_val_acc5"],
                }
            )

        self.validation_step_outputs.append(metrics)
        return metrics

    def training_step(self, batch: Sequence[Any], batch_idx: int) -> torch.Tensor:
        """Training step for MoCo V3 reusing BaseMethod training step.

        Args:
            batch (Sequence[Any]): a batch of data in the format of [img_indexes, [X], Y], where
                [X] is a list of size num_crops containing batches of images.
            batch_idx (int): index of the batch.

        Returns:
            torch.Tensor: total loss composed of MoCo V3 and classification loss.
        """

        out = super().training_step(batch, batch_idx)
        class_loss = out["loss"]

        feats1, feats2 = out["feats"]
        z1, z2 = out["z"]
        q1, q2 = out["q"]
        k1, k2 = out["momentum_k"]

        with torch.no_grad(): # to not complicate the gradient graph 
            # -- log metrics for encoder --
            gathered_feats1 = gather(feats1)
            gathered_feats2 = gather(feats2)

            # from radialvicreg
            sim_loss_encoder = invariance_loss(feats1, feats2)  # not gathered
            self.log(
                "encoder/train_backbone_sim_loss",
                sim_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )
            var_loss_encoder = variance_loss(gathered_feats1, gathered_feats2)
            self.log(
                "encoder/train_backbone_var_loss",
                var_loss_encoder / self.features_dim,
                on_epoch=True,
                sync_dist=True,
            )
            cov_loss_encoder = covariance_loss(gathered_feats1, gathered_feats2)
            self.log(
                "encoder/train_backbone_cov_loss", cov_loss_encoder, on_epoch=True, sync_dist=True
            )
            radial_loss_encoder = chi2_radial_nll_loss(gathered_feats1, gathered_feats2)
            self.log(
                "encoder/train_backbone_radial_loss",
                radial_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )

            ent_loss_encoder, hypercov_loss_encoder = entropy_and_hypercovariance_loss(
                gathered_feats1, gathered_feats2, epsilon=1e-7
            )
            self.log(
                "encoder/train_backbone_ent_loss",
                -ent_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "encoder/train_backbone_hypercov_loss",
                hypercov_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )

            # norms
            self.log(
                "encoder_norm/train_encoder_feats1_norm_mean",
                torch.norm(gathered_feats1, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "encoder_norm/train_encoder_feats2_norm_mean",
                torch.norm(gathered_feats2, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            # uniformity
            self.log(
                "unif_loss/train_encoder_feats1_unif_loss",
                uniform_loss(gathered_feats1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "unif_loss/train_encoder_feats2_unif_loss",
                uniform_loss(gathered_feats2),
                on_epoch=True,
                sync_dist=True,
            )
            # anisotropy
            self.log(
                "anisotropy_loss/train_encoder_feats1_anisotropy",
                anisotropy_loss(gathered_feats1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "anisotropy_loss/train_encoder_feats2_anisotropy",
                anisotropy_loss(gathered_feats2),
                on_epoch=True,
                sync_dist=True,
            )
            # batch sparsity
            feat1_batch_sparse_max, feat1_batch_sparse_mean, feat1_batch_sparse_min = (
                batch_sparsity_metric(gathered_feats1)
            )
            feat2_batch_sparse_max, feat2_batch_sparse_mean, feat2_batch_sparse_min = (
                batch_sparsity_metric(gathered_feats2)
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_max",
                feat1_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_mean",
                feat1_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_min",
                feat1_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_max",
                feat2_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_mean",
                feat2_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_min",
                feat2_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            # embedding sparsity
            (
                feat1_embed_sparse_max,
                feat1_embed_sparse_mean,
                feat1_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_feats1)
            (
                feat2_embed_sparse_max,
                feat2_embed_sparse_mean,
                feat2_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_feats2)
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_max",
                feat1_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_mean",
                feat1_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_min",
                feat1_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_max",
                feat2_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_mean",
                feat2_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_min",
                feat2_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )

            # -- log metrics for projector --
            gathered_z1 = gather(z1)
            gathered_z2 = gather(z2)

            # from radialvicreg
            sim_loss_projector = invariance_loss(z1, z2)  # not gathered
            self.log(
                "projector/train_projector_sim_loss",
                sim_loss_projector,
                on_epoch=True,
                sync_dist=True,
            )
            var_loss_projector = variance_loss(gathered_z1, gathered_z2)
            self.log(
                "projector/train_projector_var_loss",
                var_loss_projector / gathered_z1.size(1),
                on_epoch=True,
                sync_dist=True,
            )
            cov_loss_projector = covariance_loss(gathered_z1, gathered_z2)
            self.log(
                "projector/train_projector_cov_loss", cov_loss_projector, on_epoch=True, sync_dist=True
            )
            radial_loss_projector = chi2_radial_nll_loss(gathered_z1, gathered_z2)
            self.log(
                "projector/train_projector_radial_loss",
                radial_loss_projector,
                on_epoch=True,
                sync_dist=True,
            )

            ent_loss_projector, hypercov_loss_projector = entropy_and_hypercovariance_loss(
                gathered_z1, gathered_z2, epsilon=1e-7
            )
            self.log(
                "projector/train_projector_ent_loss",
                -ent_loss_projector,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "projector/train_projector_hypercov_loss",
                hypercov_loss_projector,
                on_epoch=True,
                sync_dist=True,
            )

            # norms
            self.log(
                "projector_norm/train_projector_z1_norm_mean",
                torch.norm(gathered_z1, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "projector_norm/train_projector_z2_norm_mean",
                torch.norm(gathered_z2, dim=1).mean(),
                on_epoch=True,
                sync_dist=True,
            )
            # uniformity
            self.log(
                "unif_loss/train_projector_z1_unif_loss",
                uniform_loss(gathered_z1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "unif_loss/train_projector_z2_unif_loss",
                uniform_loss(gathered_z2),
                on_epoch=True,
                sync_dist=True,
            )
            # anisotropy
            self.log(
                "anisotropy_loss/train_projector_z1_anisotropy",
                anisotropy_loss(gathered_z1),
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "anisotropy_loss/train_projector_z2_anisotropy",
                anisotropy_loss(gathered_z2),
                on_epoch=True,
                sync_dist=True,
            )
            # batch sparsity
            z1_batch_sparse_max, z1_batch_sparse_mean, z1_batch_sparse_min = (
                batch_sparsity_metric(gathered_z1)
            )
            z2_batch_sparse_max, z2_batch_sparse_mean, z2_batch_sparse_min = (
                batch_sparsity_metric(gathered_z2)
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_max",
                z1_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_mean",
                z1_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_min",
                z1_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_max",
                z2_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_mean",
                z2_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_min",
                z2_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            # embedding sparsity
            (
                z1_embed_sparse_max,
                z1_embed_sparse_mean,
                z1_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_z1)
            (
                z2_embed_sparse_max,
                z2_embed_sparse_mean,
                z2_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_z2)
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_max",
                z1_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_mean",
                z1_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_min",
                z1_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_max",
                z2_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_mean",
                z2_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_min",
                z2_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )

        contrastive_loss = mocov3_loss_func(
            q1, k2, temperature=self.temperature
        ) + mocov3_loss_func(q2, k1, temperature=self.temperature)

        # radial loss on projector outputs
        radial_loss = chi2_radial_nll_loss(z1, z2)

        metrics = {
            "train_contrastive_loss": contrastive_loss,
            "train_radial_loss": radial_loss,
        }
        self.log_dict(metrics, on_epoch=True, sync_dist=True)

        return contrastive_loss + self.radial_loss_weight * radial_loss + class_loss
