# Copyright 2023 solo-learn development team.

# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all copies
# or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
# INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
# FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

from typing import Any, Dict, List, Sequence

import omegaconf
import torch
import torch.nn as nn
from solo.losses.vicreg import vicreg_loss_func, uniform_loss
from solo.methods.base import BaseMethod
from solo.utils.misc import omegaconf_select


class VICReg(BaseMethod):
    def __init__(self, cfg: omegaconf.DictConfig):
        """Implements VICReg (https://arxiv.org/abs/2105.04906)

        Extra cfg settings:
            method_kwargs:
                proj_output_dim (int): number of dimensions of the projected features.
                proj_hidden_dim (int): number of neurons in the hidden layers of the projector.
                sim_loss_weight (float): weight of the invariance term.
                var_loss_weight (float): weight of the variance term.
                cov_loss_weight (float): weight of the covariance term.
                radial_loss_weight (float): weight of the radial term.
        """

        super().__init__(cfg)

        self.sim_loss_weight: float = cfg.method_kwargs.sim_loss_weight
        self.var_loss_weight: float = cfg.method_kwargs.var_loss_weight
        self.cov_loss_weight: float = cfg.method_kwargs.cov_loss_weight
        self.radial_loss_weight: float = cfg.method_kwargs.radial_loss_weight
        self.projector_type: str = cfg.method_kwargs.projector_type

        proj_hidden_dim: int = cfg.method_kwargs.proj_hidden_dim
        proj_output_dim: int = cfg.method_kwargs.proj_output_dim

        # projector
        if self.projector_type == "identity":
            self.projector = nn.Identity()
        elif self.projector_type == "mlp":
            self.projector = nn.Sequential(
                nn.Linear(self.features_dim, proj_hidden_dim),
                nn.BatchNorm1d(proj_hidden_dim),
                nn.ReLU(),
                nn.Linear(proj_hidden_dim, proj_hidden_dim),
                nn.BatchNorm1d(proj_hidden_dim),
                nn.ReLU(),
                nn.Linear(proj_hidden_dim, proj_output_dim),
            )
        else:
            raise ValueError

    @staticmethod
    def add_and_assert_specific_cfg(cfg: omegaconf.DictConfig) -> omegaconf.DictConfig:
        """Adds method specific default values/checks for config.

        Args:
            cfg (omegaconf.DictConfig): DictConfig object.

        Returns:
            omegaconf.DictConfig: same as the argument, used to avoid errors.
        """

        cfg = super(VICReg, VICReg).add_and_assert_specific_cfg(cfg)

        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_output_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_hidden_dim")

        cfg.method_kwargs.sim_loss_weight = omegaconf_select(
            cfg,
            "method_kwargs.sim_loss_weight",
            25.0,
        )
        cfg.method_kwargs.var_loss_weight = omegaconf_select(
            cfg,
            "method_kwargs.var_loss_weight",
            25.0,
        )
        cfg.method_kwargs.cov_loss_weight = omegaconf_select(
            cfg,
            "method_kwargs.cov_loss_weight",
            1.0,
        )
        cfg.method_kwargs.radial_loss_weight = omegaconf_select(
            cfg,
            "method_kwargs.radial_loss_weight",
            1.0,
        )

        return cfg

    @property
    def learnable_params(self) -> List[dict]:
        """Adds projector parameters to the parent's learnable parameters.

        Returns:
            List[dict]: list of learnable parameters.
        """

        extra_learnable_params = [{"name": "projector", "params": self.projector.parameters()}]
        return super().learnable_params + extra_learnable_params

    def forward(self, X: torch.Tensor) -> Dict[str, Any]:
        """Performs the forward pass of the backbone and the projector.

        Args:
            X (torch.Tensor): a batch of images in the tensor format.

        Returns:
            Dict[str, Any]: a dict containing the outputs of the parent and the projected features.
        """

        out = super().forward(X)

        # use torch.no_grad() for a cleaner computational graph
        with torch.no_grad():
            # log embedding layer feature norm
            backbone_feature_norm_mean = torch.norm(out["feats"], dim=1).mean() 
            backbone_feature_norm_var = torch.norm(out["feats"], dim=1).var() 

            self.log("encoder_norm/train_backbone_feature_norm_mean", backbone_feature_norm_mean, on_epoch=True, sync_dist=True)
            self.log("encoder_norm/train_backbone_feature_norm_var", backbone_feature_norm_var, on_epoch=True, sync_dist=True)

            # Logging uniform loss
            backbone_feature_unif_loss = uniform_loss(out["feats"])
            self.log("unif_loss/train_backbone_feature_unif_loss_embed", backbone_feature_unif_loss, on_epoch=True, sync_dist=True)

        z = self.projector(out["feats"])
        out.update({"z": z})
        return out

    def training_step(self, batch: Sequence[Any], batch_idx: int) -> torch.Tensor:
        """Training step for VICReg reusing BaseMethod training step.

        Args:
            batch (Sequence[Any]): a batch of data in the format of [img_indexes, [X], Y], where
                [X] is a list of size num_crops containing batches of images.
            batch_idx (int): index of the batch.

        Returns:
            torch.Tensor: total loss composed of VICReg loss and classification loss.
        """

        out = super().training_step(batch, batch_idx)
        class_loss = out["loss"]
        z1, z2 = out["z"]

        # ------- vicreg loss -------
        vicreg_loss, sim_loss, var_loss, cov_loss, radial_loss = vicreg_loss_func(
            z1,
            z2,
            sim_loss_weight=self.sim_loss_weight,
            var_loss_weight=self.var_loss_weight,
            cov_loss_weight=self.cov_loss_weight,
            radial_loss_weight=self.radial_loss_weight,
        )
        
        # use torch.no_grad() for a cleaner computational graph
        with torch.no_grad():
            # Logging norms
            z1_norm_mean = torch.norm(z1, dim=1).mean() 
            z2_norm_mean = torch.norm(z2, dim=1).mean() 
            self.log("projector_norm/train_z1_norm_mean", z1_norm_mean, on_epoch=True, sync_dist=True)
            self.log("projector_norm/train_z2_norm_mean", z2_norm_mean, on_epoch=True, sync_dist=True)

            z1_norm_var = torch.norm(z1, dim=1).var() 
            z2_norm_var = torch.norm(z2, dim=1).var() 
            self.log("projector_norm/train_z1_norm_var", z1_norm_var, on_epoch=True, sync_dist=True)
            self.log("projector_norm/train_z2_norm_var", z2_norm_var, on_epoch=True, sync_dist=True)

            # Logging uniform loss
            z1_unif_loss = uniform_loss(z1)
            z2_unif_loss = uniform_loss(z2)
            self.log("unif_loss/train_z1_unif_loss_projector", z1_unif_loss, on_epoch=True, sync_dist=True)
            self.log("unif_loss/train_z2_unif_loss_projector", z2_unif_loss, on_epoch=True, sync_dist=True)

        self.log("train_vicreg_loss", vicreg_loss, on_epoch=True, sync_dist=True)
        self.log("train_sim_loss", sim_loss, on_epoch=True, sync_dist=True)
        self.log("train_var_loss", var_loss, on_epoch=True, sync_dist=True)
        self.log("train_cov_loss", cov_loss, on_epoch=True, sync_dist=True)
        self.log("train_radial_loss", radial_loss, on_epoch=True, sync_dist=True)

        return vicreg_loss + class_loss
